<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Reservation;
use App\Models\Customer;
use App\Models\Salon;
use App\Models\Menu;
use App\Models\Extra;
use App\Models\ActivityLog;
use App\Models\Transaction;
use App\Models\Account;
use App\Models\Invoice;
use App\Models\Contract;
use App\Services\SmsService;
use App\Models\Setting;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class ReservationController extends Controller
{
    public function index(Request $request)
    {
        try {
            $user = Auth::user();
            $query = Reservation::with(['customer', 'salon', 'menu']);

            // Tarih kolonunu kontrol et
            $dateColumn = $this->getReservationDateColumn();

            // Sıralama - Önce sıralamayı kontrol et, join gerekip gerekmediğini belirle
            $sortBy = $request->get('sort', 'date_asc'); // Default: En yakın tarih (asc)
            $needsJoin = ($sortBy === 'name_asc');
            
            if ($needsJoin) {
                // A-Z (müşteri adına göre) - Join yapılacak
                $query->leftJoin('customers', 'reservations.customer_id', '=', 'customers.id')
                      ->select('reservations.*');
            }

            // Rezervasyoncu ise sadece kendi rezervasyonlarını göster
            if ($user->role === 'rezervasyoncu' || $user->role === 'admin') {
                // Join yapıldıysa tablo adını belirt
                if ($needsJoin) {
                    $query->where('reservations.created_by', $user->id);
                } else {
                    $query->where('created_by', $user->id);
                }
            }

            // Filtreler
            if ($request->filled('date')) {
                if ($needsJoin) {
                    $query->whereDate('reservations.' . $dateColumn, $request->date);
                } else {
                    $query->whereDate($dateColumn, $request->date);
                }
            }
            if ($request->filled('salon_id')) {
                if ($needsJoin) {
                    $query->where('reservations.salon_id', $request->salon_id);
                } else {
                    $query->where('salon_id', $request->salon_id);
                }
            }
            if ($request->filled('status')) {
                if ($needsJoin) {
                    $query->where('reservations.status', $request->status);
                } else {
                    $query->where('status', $request->status);
                }
            }
            if ($request->filled('customer_name')) {
                try {
                    $query->whereHas('customer', function($q) use ($request, $user) {
                        $q->where('name', 'like', '%' . $request->customer_name . '%');
                        // Rezervasyoncu ise sadece kendi müşterilerini göster
                        if ($user->role === 'rezervasyoncu' || $user->role === 'admin') {
                            $q->where('created_by', $user->id);
                        }
                    });
                } catch (\Exception $e) {
                    // İlişki hatası durumunda devam et
                }
            }

            // Sıralama uygula - Önce status'a göre (completed olanlar alta), sonra seçilen sıralamaya göre
            // Önce status'a göre sırala: completed olmayanlar önce (0), completed olanlar sonra (1)
            if ($needsJoin) {
                $query->orderByRaw("CASE WHEN reservations.status = 'completed' THEN 1 ELSE 0 END");
            } else {
                $query->orderByRaw("CASE WHEN status = 'completed' THEN 1 ELSE 0 END");
            }
            
            switch ($sortBy) {
                case 'date_asc':
                    // En yakın tarih (ascending - bugünden ileriye)
                    if ($needsJoin) {
                        $query->orderBy('reservations.' . $dateColumn, 'asc');
                    } else {
                        $query->orderBy($dateColumn, 'asc');
                    }
                    break;
                case 'name_asc':
                    // A-Z (müşteri adına göre)
                    $query->orderBy('customers.name', 'asc');
                    break;
                case 'date_desc':
                    // Yeniden Eskiye (tarihe göre desc)
                    if ($needsJoin) {
                        $query->orderBy('reservations.' . $dateColumn, 'desc');
                    } else {
                        $query->orderBy($dateColumn, 'desc');
                    }
                    break;
                default:
                    if ($needsJoin) {
                        $query->orderBy('reservations.' . $dateColumn, 'asc');
                    } else {
                        $query->orderBy($dateColumn, 'asc');
                    }
            }

            $reservations = $query->paginate(20);
            
            // Salonları filtrele
            $salonQuery = Salon::where('is_active', true);
            if ($user->role === 'rezervasyoncu' || $user->role === 'admin') {
                $salonQuery->where('created_by', $user->id);
            }
            $salons = $salonQuery->get();

            return view('admin.reservations.index', compact('reservations', 'salons'));
        } catch (\Exception $e) {
            \Log::error('Reservation Index Error: ' . $e->getMessage());
            $reservations = \Illuminate\Pagination\Paginator::make([], 0, 20);
            $salons = collect([]);
            return view('admin.reservations.index', compact('reservations', 'salons'))
                ->with('error', 'Rezervasyonlar yüklenirken bir hata oluştu.');
        }
    }
    

    public function create()
    {
        try {
            $user = Auth::user();
            $query = Customer::query();
            
            // Rezervasyoncu ise sadece kendi müşterilerini göster
            if ($user->role === 'rezervasyoncu' || $user->role === 'admin') {
                $query->where('created_by', $user->id);
            }
            
            $customers = $query->orderBy('name')->get();
            // Salonları filtrele
            $salonQuery = Salon::where('is_active', true);
            if ($user->role === 'rezervasyoncu' || $user->role === 'admin') {
                $salonQuery->where('created_by', $user->id);
            }
            $salons = $salonQuery->get();
            
            // Menüler ve ekstralar
            try {
                $menus = \App\Models\Menu::where('is_active', true)->orderBy('name')->get();
            } catch (\Exception $e) {
                $menus = collect([]);
            }
            
            try {
                $extras = \App\Models\Extra::where('is_active', true)->orderBy('name')->get();
            } catch (\Exception $e) {
                $extras = collect([]);
            }

            return view('admin.reservations.create', compact('customers', 'salons', 'menus', 'extras'));
        } catch (\Exception $e) {
            \Log::error('Reservation Create Error: ' . $e->getMessage());
            return view('admin.reservations.create', [
                'customers' => collect([]),
                'salons' => collect([]),
                'menus' => collect([]),
                'extras' => collect([])
            ])->with('error', 'Sayfa yüklenirken bir hata oluştu.');
        }
    }
    
    /**
     * Seçilen tarih ve salon için dolu saatleri getir
     */
    public function checkAvailability(Request $request)
    {
        try {
            $date = $request->get('date');
            $salonId = $request->get('salon_id');
            
            if (!$date || !$salonId) {
                return response()->json(['busy_hours' => []]);
            }
            
            $user = Auth::user();
            
            // O gün o salondaki rezervasyonları getir
            $query = Reservation::where('salon_id', $salonId)
                ->whereDate('date', $date)
                ->whereIn('status', ['open', 'confirmed']);
            
            // Rezervasyoncu/admin ise sadece kendi salonları
            if ($user->role === 'rezervasyoncu' || $user->role === 'admin') {
                $salon = Salon::where('id', $salonId)->where('created_by', $user->id)->first();
                if (!$salon) {
                    return response()->json(['busy_hours' => [], 'error' => 'Salon bulunamadı']);
                }
            }
            
            $reservations = $query->select('start_time', 'end_time', 'customer_id')
                ->with('customer:id,name')
                ->get();
            
            $busyHours = $reservations->map(function($res) {
                return [
                    'start_time' => $res->start_time,
                    'end_time' => $res->end_time,
                    'customer' => $res->customer->name ?? 'Müşteri'
                ];
            });
            
            return response()->json([
                'busy_hours' => $busyHours,
                'date' => $date,
                'salon_id' => $salonId
            ]);
        } catch (\Exception $e) {
            \Log::error('Check Availability Error: ' . $e->getMessage());
            return response()->json(['busy_hours' => [], 'error' => $e->getMessage()]);
        }
    }

    public function store(Request $request)
    {
        try {
            $validated = $request->validate([
                'customer_id' => 'required|exists:customers,id',
                'salon_id' => 'required|exists:salons,id',
                'date' => 'required|date',
                'start_time' => 'required',
                'end_time' => 'required',
                'guest_count' => 'nullable|integer|min:1',
                'total_price' => 'required|numeric|min:0|max:99999999.99',
                'deposit_paid' => 'nullable|numeric|min:0|max:99999999.99',
                'status' => 'nullable|in:open,confirmed,cancelled,completed',
                'notes' => 'nullable|string',
                'meals' => 'nullable|array',
                'extras' => 'nullable|array',
                'organization_price' => 'nullable|numeric|min:0|max:99999999.99',
                'bride_name' => 'required|string|max:255',
                'groom_name' => 'required|string|max:255',
                'bride_groom_phone' => 'required|string|max:20',
                'bride_groom_name' => 'nullable|string|max:255', // Hidden input için (geriye dönük uyumluluk)
            ]);

            $user = Auth::user();
            
            // Rezervasyoncu ise müşteri kontrolü
            if ($user->role === 'rezervasyoncu' || $user->role === 'admin') {
                $customer = Customer::findOrFail($validated['customer_id']);
                if ($customer->created_by !== $user->id) {
                    return redirect()->back()
                        ->withInput()
                        ->with('error', 'Bu müşteriye erişim yetkiniz yok.');
                }
            }
            
            // Toplam fiyatı hesapla: Rezervasyon ücreti + Yemekler + Organizasyon + Ekstralar
            $reservationPrice = (float)($validated['total_price'] ?? 0);
            $mealTotal = 0;
            
            if ($request->filled('meals') && is_array($request->meals)) {
                foreach ($request->meals as $meal) {
                    if (isset($meal['name']) && isset($meal['quantity']) && isset($meal['price'])) {
                        $mealTotal += (float)$meal['quantity'] * (float)$meal['price'];
                    }
                }
            }
            
            // Organizasyon toplamı
            $organizationTotal = 0;
            if ($request->filled('organizations') && is_array($request->organizations)) {
                foreach ($request->organizations as $org) {
                    if (isset($org['quantity']) && isset($org['price'])) {
                        $organizationTotal += (float)$org['quantity'] * (float)$org['price'];
                    }
                }
            }
            
            $extraTotal = 0;
            
            if ($request->filled('extras') && is_array($request->extras)) {
                foreach ($request->extras as $extra) {
                    if (isset($extra['name']) && isset($extra['price'])) {
                        $extraTotal += (float)$extra['price'];
                    }
                }
            }
            
            // Toplam fiyat = Rezervasyon ücreti + Yemekler + Organizasyonlar + Ekstralar
            $totalPrice = $reservationPrice + $mealTotal + $organizationTotal + $extraTotal;
            
            // Fiyat kontrolü - Toplam fiyat 0'dan büyük olmalı
            if ($totalPrice <= 0) {
                return redirect()->back()
                    ->withInput()
                    ->with('error', 'Rezervasyon oluşturmak için en az bir fiyat girişi yapmanız gerekmektedir. Lütfen salon ücreti, yemek, organizasyon veya ekstra fiyatlarından en az birini giriniz.');
            }
            
            // Maksimum değer kontrolü (decimal(10,2) = 99999999.99)
            $maxValue = 99999999.99;
            if ($totalPrice > $maxValue) {
                \Log::error('Total price exceeds maximum value', [
                    'total_price' => $totalPrice,
                    'max_value' => $maxValue,
                    'reservation_price' => $reservationPrice,
                    'meal_total' => $mealTotal,
                    'organization_price' => $organizationPrice,
                    'extra_total' => $extraTotal,
                ]);
                return redirect()->back()
                    ->withInput()
                    ->with('error', 'Toplam tutar çok büyük! Maksimum tutar: ' . number_format($maxValue, 2, ',', '.') . ' ₺');
            }
            
            $depositPaid = (float)($validated['deposit_paid'] ?? 0);
            $remainingAmount = $totalPrice - $depositPaid;
            
            // Rezervasyon kodu oluştur
            $code = 'REZ-' . strtoupper(uniqid());
            
            // Tarihi düzelt (sadece tarih, saat olmadan)
            $reservationDate = date('Y-m-d', strtotime($validated['date']));
            
            // Rezervasyon oluştur
            $reservation = Reservation::create([
                'code' => $code,
                'customer_id' => $validated['customer_id'],
                'salon_id' => $validated['salon_id'],
                'date' => $reservationDate,
                'reservation_date' => $reservationDate,
                'start_time' => $validated['start_time'],
                'end_time' => $validated['end_time'],
                'guest_count' => $validated['guest_count'] ?? 1,
                'total_price' => $totalPrice,
                'deposit_paid' => $depositPaid,
                'remaining_amount' => $remainingAmount,
                'status' => 'confirmed', // Her zaman onaylı olarak oluştur
                'notes' => $validated['notes'] ?? null,
                'created_by' => $user->id,
            ]);
            
            // Yemek, ekstra ve organizasyon verilerini items_data alanına JSON olarak kaydet (fatura için)
            $itemsData = [];
            
            // Yemekleri ekle
            if ($request->filled('meals') && is_array($request->meals)) {
                $itemsData['meals'] = [];
                foreach ($request->meals as $meal) {
                    if (isset($meal['name']) && !empty(trim($meal['name'])) && isset($meal['price']) && (float)$meal['price'] > 0) {
                        $itemsData['meals'][] = [
                            'name' => trim($meal['name']),
                            'quantity' => isset($meal['quantity']) && (int)$meal['quantity'] > 0 ? (int)$meal['quantity'] : 1,
                            'price' => (float)$meal['price']
                        ];
                    }
                }
            }
            
            // Organizasyonları ekle
            if ($request->filled('organizations') && is_array($request->organizations)) {
                $itemsData['organizations'] = [];
                foreach ($request->organizations as $org) {
                    if (isset($org['name']) && !empty(trim($org['name'])) && isset($org['price']) && (float)$org['price'] > 0) {
                        $itemsData['organizations'][] = [
                            'name' => trim($org['name']),
                            'quantity' => isset($org['quantity']) && (int)$org['quantity'] > 0 ? (int)$org['quantity'] : 1,
                            'price' => (float)$org['price']
                        ];
                    }
                }
            }
            
            // Ekstraları ekle
            if ($request->filled('extras') && is_array($request->extras)) {
                $itemsData['extras'] = [];
                foreach ($request->extras as $extra) {
                    if (isset($extra['name']) && !empty(trim($extra['name'])) && isset($extra['price']) && (float)$extra['price'] > 0) {
                        $itemsData['extras'][] = [
                            'name' => trim($extra['name']),
                            'price' => (float)$extra['price']
                        ];
                    }
                }
            }
            
            // Gelin ve Damat bilgilerini ekle (birleştirilmiş)
            // Önce form'dan gelen bride_groom_name'i parse et
            $brideGroomName = $request->input('bride_groom_name', '');
            $brideName = $validated['bride_name'] ?? '';
            $groomName = $validated['groom_name'] ?? '';
            
            // Eğer bride_groom_name varsa ve & içeriyorsa, ayır
            if (empty($brideName) && empty($groomName) && !empty($brideGroomName)) {
                if (strpos($brideGroomName, '&') !== false) {
                    $parts = explode('&', $brideGroomName);
                    $groomName = trim($parts[0] ?? '');
                    $brideName = trim($parts[1] ?? '');
                } else {
                    // Tek isim varsa, gelin olarak kabul et
                    $brideName = trim($brideGroomName);
                }
            }
            
            if ($brideName && $groomName) {
                $itemsData['bride_groom_name'] = $groomName . ' & ' . $brideName;
                $itemsData['bride_name'] = $brideName;
                $itemsData['groom_name'] = $groomName;
            } elseif ($brideName) {
                $itemsData['bride_groom_name'] = $brideName;
                $itemsData['bride_name'] = $brideName;
            } elseif ($groomName) {
                $itemsData['bride_groom_name'] = $groomName;
                $itemsData['groom_name'] = $groomName;
            } elseif (!empty($brideGroomName)) {
                $itemsData['bride_groom_name'] = $brideGroomName;
            }
            
            if ($request->filled('bride_groom_phone')) {
                $itemsData['bride_groom_phone'] = $validated['bride_groom_phone'];
            }
            
            // items_data alanına kaydet (notes alanına değil)
            if (!empty($itemsData)) {
                $reservation->items_data = json_encode($itemsData);
                $reservation->save();
            }

            // Müşteriye toplam tutarı borç olarak ekle
            try {
                $customer = Customer::find($validated['customer_id']);
                if ($customer && $totalPrice > 0) {
                    $customer->increment('balance', $totalPrice);
                    \Log::info('Müşteri bakiyesine borç eklendi', [
                        'customer_id' => $customer->id,
                        'amount' => $totalPrice,
                        'reservation_id' => $reservation->id,
                    ]);
                }
            } catch (\Exception $e) {
                \Log::error('Müşteri bakiyesi güncellenemedi: ' . $e->getMessage());
            }

            // Alınan kapora varsa kasaya GELİR olarak ekle
            if ($depositPaid > 0) {
                try {
                    // Ana kasa hesabını getir veya oluştur
                    $cashAccount = Account::getCashAccount();
                    
                    // Müşteri ve salon bilgisi
                    $customer = Customer::find($validated['customer_id']);
                    $salon = Salon::find($validated['salon_id']);
                    
                    // Açıklama oluştur
                    $description = "💰 KAPORA ALINDI - Rezervasyon #{$reservation->code}";
                    if ($customer) $description .= " | Müşteri: {$customer->name}";
                    if ($salon) $description .= " | Salon: {$salon->name}";
                    $description .= " | Tarih: " . date('d.m.Y', strtotime($reservationDate));
                    
                    // Kasaya gelir olarak ekle
                    Transaction::create([
                        'account_id' => $cashAccount->id,
                        'reservation_id' => $reservation->id,
                        'customer_id' => $validated['customer_id'],
                        'type' => 'income',
                        'amount' => $depositPaid,
                        'date' => now()->format('Y-m-d'),
                        'description' => $description,
                        'created_by' => $user->id,
                    ]);
                    
                    // Müşteri bakiyesinden kapora miktarını düş
                    if ($customer && $customer->balance !== null) {
                        $customer->decrement('balance', $depositPaid);
                    }
                    
                    \Log::info('✅ Kapora kasaya GELİR olarak eklendi', [
                        'reservation_id' => $reservation->id,
                        'amount' => $depositPaid,
                        'kasa_bakiye' => $cashAccount->fresh()->balance,
                    ]);
                    
                } catch (\Exception $e) {
                    \Log::error('❌ Kapora kasaya eklenemedi: ' . $e->getMessage());
                }
            }

            // Fatura ve Sözleşme PDF'lerini oluştur - HER ZAMAN oluştur
            $pdfLinks = null;
            try {
                $pdfLinks = $this->generateInvoiceAndContract($reservation);
                \Log::info('PDF linkleri oluşturuldu', [
                    'reservation_id' => $reservation->id,
                    'invoice_url' => $pdfLinks['invoice_url'] ?? null,
                    'contract_url' => $pdfLinks['contract_url'] ?? null,
                ]);
            } catch (\Exception $e) {
                \Log::error('PDF oluşturma hatası: ' . $e->getMessage(), [
                    'reservation_id' => $reservation->id,
                    'trace' => $e->getTraceAsString(),
                ]);
            }
            
            // Eğer create_invoice flag'i varsa, önce faturayı oluştur (SMS'te görüntüleme sayfası linki göndermek için)
            $invoiceUrl = null;
            if ($request->input('create_invoice') == '1') {
                try {
                    $invoice = $this->createInvoiceForReservation($reservation);
                    $invoiceUrl = $invoice->public_url; // Görüntüleme sayfası linki
                    \Log::info('Fatura oluşturuldu (SMS için)', [
                        'reservation_id' => $reservation->id,
                        'invoice_id' => $invoice->id,
                        'public_url' => $invoiceUrl,
                    ]);
                } catch (\Exception $e) {
                    \Log::error('Fatura oluşturma hatası (SMS için): ' . $e->getMessage());
                    // Hata olsa bile devam et, PDF linkini kullan
                }
            }
            
            // Rezervasyon tarih/saat bilgisini hazırla (her iki mesaj için de kullanılacak)
            $reservationDate = $reservation->date ? $reservation->date->format('Y-m-d') : null;
            $reservationTime = '';
            if ($reservation->start_time && $reservation->end_time) {
                $reservationTime = $reservation->start_time . ' - ' . $reservation->end_time;
            } elseif ($reservation->start_time) {
                $reservationTime = $reservation->start_time;
            }
            
            // Varsayılan sözleşme linkini al
            $defaultContract = Contract::getDefaultForCompany(auth()->user()->company_id ?? null);
            $contractUrl = $defaultContract ? $defaultContract->public_url : ($pdfLinks['contract_url'] ?? null);
            
            // SMS'te gönderilecek fatura linki: Eğer fatura oluşturulmuşsa görüntüleme sayfası, yoksa PDF linki
            $smsInvoiceUrl = $invoiceUrl ?? ($pdfLinks['invoice_url'] ?? null);
            
            \Log::info('WhatsApp mesaj gönderme hazırlığı', [
                'reservation_id' => $reservation->id,
                'has_customer' => $reservation->customer ? true : false,
                'customer_phone' => $reservation->customer->phone ?? null,
                'has_pdf_links' => $pdfLinks !== null,
                'invoice_url' => $smsInvoiceUrl,
                'contract_url' => $contractUrl,
                'is_view_page' => $invoiceUrl !== null,
            ]);
            
            // SMS bakiyesi kontrolü - Hem müşteriye hem gelin/damat numarasına gönderilse 2 SMS kesilir
            $smsBalanceLow = false;
            $user = Auth::user();
            if ($user && !$user->isSuperAdmin()) {
                // Müşteri ve gelin/damat numarası varsa 2 SMS gerekli
                $hasCustomerPhone = $reservation->customer && $reservation->customer->phone;
                $brideGroomPhone = $validated['bride_groom_phone'] ?? $request->input('bride_groom_phone');
                $hasBrideGroomPhone = $brideGroomPhone && strlen(preg_replace('/[^0-9]/', '', $brideGroomPhone)) >= 10;
                
                // Toplam SMS sayısını hesapla (müşteriye 1 + gelin/damada 1 = 2 SMS)
                $requiredSms = 0;
                if ($hasCustomerPhone) $requiredSms += 1;
                if ($hasBrideGroomPhone) $requiredSms += 1;
                
                if ($requiredSms > 0 && !$user->hasSmsBalance($requiredSms)) {
                    $smsBalanceLow = true;
                    \Log::warning('⚠️ SMS bakiyesi yetersiz', [
                        'user_id' => $user->id,
                        'sms_balance' => $user->sms_balance,
                        'has_customer_phone' => $hasCustomerPhone,
                        'has_bride_groom_phone' => $hasBrideGroomPhone,
                    ]);
                }
            }
            
            // Müşteriye ve Gelin/Damat numarasına mesaj gönder - WhatsApp ve SMS
            $customerWhatsAppSent = false;
            $customerSmsSent = false;
            $brideGroomWhatsAppSent = false;
            $brideGroomSmsSent = false;
            $smsBalanceDeductedCount = 0; // SMS bakiyesi kaç kez kesildi (müşteriye 1, gelin/damada 1 = toplam 2)
            
            $messageService = app(\App\Services\MessageService::class);
            
            // Müşteriye WhatsApp mesajı gönder
            if ($reservation->customer && $reservation->customer->phone) {
                try {
                    $whatsappResult = $messageService->sendInvoiceAndContractViaWhatsApp(
                        phone: $reservation->customer->phone,
                        invoiceUrl: $smsInvoiceUrl,
                        contractUrl: $contractUrl,
                        salonId: $reservation->salon_id,
                        reservationDate: $reservationDate,
                        reservationTime: $reservationTime,
                        customerName: $reservation->customer->name
                    );
                    
                    if ($whatsappResult) {
                        $customerWhatsAppSent = true;
                        \Log::info('✅ Rezervasyon oluşturulduğunda müşteriye WhatsApp mesajı gönderildi', [
                            'reservation_id' => $reservation->id,
                            'phone' => $reservation->customer->phone,
                            'invoice_url' => $smsInvoiceUrl,
                            'contract_url' => $contractUrl,
                        ]);
                    } else {
                        \Log::warning('⚠️ Rezervasyon oluşturulduğunda müşteriye WhatsApp mesajı gönderilemedi', [
                            'reservation_id' => $reservation->id,
                            'phone' => $reservation->customer->phone,
                        ]);
                    }
                } catch (\Exception $e) {
                    \Log::error('❌ Müşteriye WhatsApp mesaj gönderme hatası: ' . $e->getMessage(), [
                        'reservation_id' => $reservation->id,
                        'phone' => $reservation->customer->phone,
                        'trace' => $e->getTraceAsString()
                    ]);
                }
            }
            
            // Müşteriye SMS mesajı gönder
            if ($reservation->customer && $reservation->customer->phone) {
                // Telefon numarasını temizle
                $customerPhone = preg_replace('/[^0-9]/', '', $reservation->customer->phone);
                
                if (strlen($customerPhone) >= 10) {
                    if (!$smsBalanceLow) {
                        try {
                            $result = $messageService->sendInvoiceAndContractViaSms(
                                phone: $customerPhone,
                                invoiceUrl: $smsInvoiceUrl,
                                contractUrl: $contractUrl,
                                salonId: $reservation->salon_id,
                                reservationDate: $reservationDate,
                                reservationTime: $reservationTime,
                                customerName: $reservation->customer->name,
                                skipBalanceDeduction: false // İlk SMS'te bakiyeyi kes
                            );
                            
                            if ($result) {
                                $customerSmsSent = true;
                                $smsBalanceDeductedCount += 1; // Müşteriye SMS gönderildi, 1 SMS düşürüldü
                                \Log::info('✅ Rezervasyon oluşturulduğunda müşteriye SMS mesajı gönderildi', [
                                    'reservation_id' => $reservation->id,
                                    'phone' => $customerPhone,
                                    'invoice_url' => $smsInvoiceUrl,
                                    'contract_url' => $contractUrl,
                                    'sms_balance_deducted' => 1,
                                ]);
                            } else {
                                \Log::warning('⚠️ Rezervasyon oluşturulduğunda müşteriye SMS mesajı gönderilemedi', [
                                    'reservation_id' => $reservation->id,
                                    'phone' => $customerPhone,
                                ]);
                            }
                        } catch (\Exception $e) {
                            \Log::error('❌ Müşteriye SMS mesaj gönderme hatası: ' . $e->getMessage(), [
                                'reservation_id' => $reservation->id,
                                'phone' => $customerPhone,
                                'trace' => $e->getTraceAsString()
                            ]);
                        }
                    } else {
                        \Log::warning('⚠️ SMS bakiyesi yetersiz, müşteriye SMS mesajı gönderilemedi', [
                            'reservation_id' => $reservation->id,
                            'customer_id' => $reservation->customer_id,
                            'phone' => $customerPhone,
                        ]);
                    }
                } else {
                    \Log::warning('⚠️ Müşteri telefon numarası geçersiz', [
                        'reservation_id' => $reservation->id,
                        'customer_id' => $reservation->customer_id,
                        'phone' => $customerPhone,
                    ]);
                }
            } else {
                \Log::warning('⚠️ Müşteri telefon numarası yok, mesaj gönderilemedi', [
                    'reservation_id' => $reservation->id,
                    'customer_id' => $reservation->customer_id,
                ]);
            }
            
            // Gelin ve Damat numarasına mesaj gönder
            $brideGroomPhone = $validated['bride_groom_phone'] ?? $request->input('bride_groom_phone');
            if ($brideGroomPhone) {
                try {
                    // Telefon numarasını temizle (sadece rakamlar)
                    $brideGroomPhone = preg_replace('/[^0-9]/', '', $brideGroomPhone);
                    
                    // Eğer numara geçerliyse mesaj gönder
                    if (strlen($brideGroomPhone) >= 10) {
                        // Gelin ve Damat isimlerini birleştir (& ile)
                        $brideName = $validated['bride_name'] ?? '';
                        $groomName = $validated['groom_name'] ?? '';
                        
                        // Eğer items_data'da varsa oradan al
                        $itemsData = json_decode($reservation->items_data ?? '{}', true);
                        if (empty($brideName) && isset($itemsData['bride_name'])) {
                            $brideName = $itemsData['bride_name'];
                        }
                        if (empty($groomName) && isset($itemsData['groom_name'])) {
                            $groomName = $itemsData['groom_name'];
                        }
                        
                        $brideGroomName = '';
                        if ($groomName && $brideName) {
                            $brideGroomName = $groomName . ' & ' . $brideName;
                        } elseif ($groomName) {
                            $brideGroomName = $groomName;
                        } elseif ($brideName) {
                            $brideGroomName = $brideName;
                        } elseif (isset($itemsData['bride_groom_name'])) {
                            $brideGroomName = $itemsData['bride_groom_name'];
                        } else {
                            $brideGroomName = $reservation->customer->name ?? 'Müşteri';
                        }
                        
                        // Gelin/Damat numarasına WhatsApp mesajı gönder
                        try {
                            $whatsappResult = $messageService->sendInvoiceAndContractViaWhatsApp(
                                phone: $brideGroomPhone,
                                invoiceUrl: $smsInvoiceUrl,
                                contractUrl: $contractUrl,
                                salonId: $reservation->salon_id,
                                reservationDate: $reservationDate,
                                reservationTime: $reservationTime,
                                customerName: $brideGroomName
                            );
                            
                            if ($whatsappResult) {
                                $brideGroomWhatsAppSent = true;
                                \Log::info('✅ Rezervasyon oluşturulduğunda gelin/damat numarasına WhatsApp mesajı gönderildi', [
                                    'reservation_id' => $reservation->id,
                                    'phone' => $brideGroomPhone,
                                    'name' => $brideGroomName,
                                ]);
                            } else {
                                \Log::warning('⚠️ Rezervasyon oluşturulduğunda gelin/damat numarasına WhatsApp mesajı gönderilemedi', [
                                    'reservation_id' => $reservation->id,
                                    'phone' => $brideGroomPhone,
                                ]);
                            }
                        } catch (\Exception $e) {
                            \Log::error('❌ Gelin/Damat numarasına WhatsApp mesaj gönderme hatası: ' . $e->getMessage(), [
                                'reservation_id' => $reservation->id,
                                'phone' => $brideGroomPhone,
                                'trace' => $e->getTraceAsString()
                            ]);
                        }
                        
                        // Gelin/Damat numarasına SMS mesajı gönder
                        if (!$smsBalanceLow) {
                            try {
                                // Gelin/damada SMS gönderildiğinde de bakiyeyi kes (müşteriye gönderildiyse de kes)
                                // Her rezervasyonda 2 SMS düşürülmeli: müşteriye 1 + gelin/damada 1
                                $result = $messageService->sendInvoiceAndContractViaSms(
                                    phone: $brideGroomPhone,
                                    invoiceUrl: $smsInvoiceUrl,
                                    contractUrl: $contractUrl,
                                    salonId: $reservation->salon_id,
                                    reservationDate: $reservationDate,
                                    reservationTime: $reservationTime,
                                    customerName: $brideGroomName,
                                    skipBalanceDeduction: false // Gelin/damada SMS'te de bakiyeyi kes
                                );
                                
                                if ($result) {
                                    $brideGroomSmsSent = true;
                                    $smsBalanceDeductedCount += 1; // Gelin/damada SMS gönderildi, 1 SMS daha düşürüldü
                                    \Log::info('✅ Rezervasyon oluşturulduğunda gelin/damat numarasına SMS mesajı gönderildi', [
                                        'reservation_id' => $reservation->id,
                                        'phone' => $brideGroomPhone,
                                        'name' => $brideGroomName,
                                        'sms_balance_deducted' => 1,
                                        'total_sms_deducted' => $smsBalanceDeductedCount,
                                    ]);
                                } else {
                                    \Log::warning('⚠️ Rezervasyon oluşturulduğunda gelin/damat numarasına SMS mesajı gönderilemedi', [
                                        'reservation_id' => $reservation->id,
                                        'phone' => $brideGroomPhone,
                                    ]);
                                }
                            } catch (\Exception $e) {
                                \Log::error('❌ Gelin/Damat numarasına SMS mesaj gönderme hatası: ' . $e->getMessage(), [
                                    'reservation_id' => $reservation->id,
                                    'phone' => $brideGroomPhone,
                                    'trace' => $e->getTraceAsString()
                                ]);
                            }
                        } else {
                            \Log::warning('⚠️ SMS bakiyesi yetersiz, gelin/damat numarasına SMS mesajı gönderilemedi', [
                                'reservation_id' => $reservation->id,
                                'phone' => $brideGroomPhone,
                            ]);
                        }
                    } else {
                        \Log::warning('⚠️ Gelin/Damat telefon numarası geçersiz', [
                            'reservation_id' => $reservation->id,
                            'phone' => $brideGroomPhone,
                        ]);
                    }
                } catch (\Exception $e) {
                    \Log::error('❌ Gelin/Damat numarasına mesaj gönderme hatası: ' . $e->getMessage(), [
                        'reservation_id' => $reservation->id,
                        'phone' => $brideGroomPhone,
                        'trace' => $e->getTraceAsString()
                    ]);
                }
            } else {
                \Log::warning('⚠️ Gelin/Damat telefon numarası yok, mesaj gönderilemedi', [
                    'reservation_id' => $reservation->id,
                ]);
            }

            // Activity log
            try {
                ActivityLog::create([
                    'user_id' => $user->id,
                    'action' => 'create_reservation',
                    'description' => "Yeni rezervasyon oluşturuldu: {$reservation->code}",
                    'ip_address' => $request->ip(),
                ]);
            } catch (\Exception $e) {
                \Log::warning('Activity log kaydedilemedi: ' . $e->getMessage());
            }

            // Fatura oluştur (create_invoice flag'i varsa)
            // NOT: Eğer SMS gönderilecekse, fatura zaten yukarıda oluşturuldu
            $invoiceCreated = false;
            if ($request->input('create_invoice') == '1' && !$invoiceUrl) {
                // Eğer fatura henüz oluşturulmadıysa (SMS gönderilmediyse), şimdi oluştur
                try {
                    $invoice = $this->createInvoiceForReservation($reservation);
                    $invoiceCreated = true;
                    $invoiceUrl = $invoice->public_url;
                } catch (\Exception $e) {
                    \Log::error('Fatura oluşturma hatası: ' . $e->getMessage());
                }
            } elseif ($invoiceUrl) {
                // Eğer fatura zaten oluşturulduysa (SMS için), sadece flag'i set et
                $invoiceCreated = true;
            }

            $successMessage = 'Rezervasyon başarıyla oluşturuldu.';
            if ($invoiceCreated) {
                $successMessage .= ' Fatura da oluşturuldu ve müşteriye gönderildi.';
            }
            
            // SMS bakiyesi düşükse uyarı mesajı ekle
            if ($smsBalanceLow) {
                $smsWarningMessage = 'SMS bakiyeniz kalmamıştır. SMS gönderilemedi.';
                return redirect()->route('admin.rezervasyonlar.index')
                    ->with('success', $successMessage)
                    ->with('sms_warning', $smsWarningMessage);
            }

            return redirect()->route('admin.rezervasyonlar.index')
                ->with('success', $successMessage);
                
        } catch (\Illuminate\Validation\ValidationException $e) {
            return redirect()->back()
                ->withErrors($e->errors())
                ->withInput();
        } catch (\Exception $e) {
            \Log::error('Reservation Store Error: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString()
            ]);
            return redirect()->back()
                ->withInput()
                ->with('error', 'Rezervasyon oluşturulurken bir hata oluştu: ' . $e->getMessage());
        }
    }

    public function checkAvailableTimes(Request $request)
    {
        try {
            // JSON isteği kontrolü
            if ($request->isJson()) {
                $date = $request->json('date');
                $salonId = $request->json('salon_id');
            } else {
                $request->validate([
                    'date' => 'required|date',
                    'salon_id' => 'required|exists:salons,id',
                ]);
                $date = $request->date;
                $salonId = $request->salon_id;
            }

            if (!$date || !$salonId) {
                return response()->json([
                    'success' => false,
                    'booked_slots' => [],
                    'error' => 'Tarih ve salon ID gerekli'
                ], 400);
            }

            $dateColumn = $this->getReservationDateColumn();

            // Bu tarih ve salon için mevcut rezervasyonları getir (iptal edilmemiş)
            $user = Auth::user();
            $query = Reservation::where($dateColumn, $date)
                ->where('salon_id', $salonId)
                ->where('status', '!=', 'cancelled');
            
            // Rezervasyoncu ise sadece kendi rezervasyonlarını göster
            if ($user->role === 'rezervasyoncu' || $user->role === 'admin') {
                $query->where('created_by', $user->id);
            }
            
            $reservations = $query->get(['id', 'start_time', 'end_time', 'status']);

            // Dolu saat aralıklarını hesapla
            $bookedSlots = [];
            foreach ($reservations as $reservation) {
                try {
                    if (!$reservation->start_time || !$reservation->end_time) {
                        continue;
                    }
                    
                    $start = strtotime($reservation->start_time);
                    $end = strtotime($reservation->end_time);
                    
                    if ($start === false || $end === false || $start >= $end) {
                        continue;
                    }
                    
                    // 30 dakika aralıklarla dolu saatleri işaretle
                    for ($time = $start; $time < $end; $time += 1800) { // 1800 saniye = 30 dakika
                        $slot = date('H:i', $time);
                        if (!in_array($slot, $bookedSlots)) {
                            $bookedSlots[] = $slot;
                        }
                    }
                } catch (\Exception $e) {
                    continue;
                }
            }

            // Sırala ve tekrarları kaldır
            sort($bookedSlots);
            $bookedSlots = array_values(array_unique($bookedSlots));

            return response()->json([
                'success' => true,
                'booked_slots' => $bookedSlots
            ]);
        } catch (\Exception $e) {
            \Log::error('Check Available Times Error: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString(),
            ]);
            return response()->json([
                'success' => false,
                'booked_slots' => [],
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function show($id)
    {
        try {
            $user = Auth::user();
            $query = Reservation::with(['customer', 'salon']);
            
            // Rezervasyoncu veya admin ise sadece kendi rezervasyonunu görebilir
            if ($user->role === 'rezervasyoncu' || $user->role === 'admin') {
                $query->where('created_by', $user->id);
            }
            
            $reservation = $query->findOrFail($id);

            // AJAX isteği ise JSON döndür
            if (request()->ajax() || request()->wantsJson()) {
                // items_data veya notes'tan JSON verilerini çıkar
                $itemsData = json_decode($reservation->items_data ?? '{}', true);
                if (!is_array($itemsData) || empty($itemsData)) {
                    // Eski sistemle uyumluluk için notes'tan da kontrol et
                    $itemsData = json_decode($reservation->notes ?? '{}', true);
                    if (!is_array($itemsData)) {
                        $itemsData = [];
                    }
                }
                
                // Fatura kalemlerini hazırla
                $items = [];
                
                // Salon ücreti hesapla
                $mealTotal = 0;
                $organizationPrice = $itemsData['organization_price'] ?? 0;
                $extraTotal = 0;
                
                if (isset($itemsData['meals']) && is_array($itemsData['meals'])) {
                    foreach ($itemsData['meals'] as $meal) {
                        if (isset($meal['quantity']) && isset($meal['price'])) {
                            $mealTotal += $meal['quantity'] * $meal['price'];
                        }
                    }
                }
                
                if (isset($itemsData['extras']) && is_array($itemsData['extras'])) {
                    foreach ($itemsData['extras'] as $extra) {
                        if (isset($extra['price'])) {
                            $extraTotal += $extra['price'];
                        }
                    }
                }
                
                $salonPrice = $reservation->total_price - ($mealTotal + $organizationPrice + $extraTotal);
                
                if ($salonPrice > 0) {
                    $items[] = ['name' => 'Salon Ücreti', 'quantity' => '-', 'price' => $salonPrice, 'total' => $salonPrice];
                }
                
                if (isset($itemsData['meals']) && is_array($itemsData['meals'])) {
                    foreach ($itemsData['meals'] as $meal) {
                        if (isset($meal['name']) && isset($meal['quantity']) && isset($meal['price'])) {
                            $items[] = [
                                'name' => $meal['name'],
                                'quantity' => $meal['quantity'] . ' Kişi',
                                'price' => $meal['price'],
                                'total' => $meal['quantity'] * $meal['price']
                            ];
                        }
                    }
                }
                
                if ($organizationPrice > 0) {
                    $items[] = ['name' => 'Organizasyon', 'quantity' => '-', 'price' => $organizationPrice, 'total' => $organizationPrice];
                }
                
                if (isset($itemsData['extras']) && is_array($itemsData['extras'])) {
                    foreach ($itemsData['extras'] as $extra) {
                        if (isset($extra['name']) && isset($extra['price']) && $extra['price'] > 0) {
                            $items[] = [
                                'name' => $extra['name'],
                                'quantity' => '-',
                                'price' => $extra['price'],
                                'total' => $extra['price']
                            ];
                        }
                    }
                }
                
                // Gelin ve Damat bilgilerini items_data'dan al
                $brideGroomName = $itemsData['bride_groom_name'] ?? null;
                $brideName = $itemsData['bride_name'] ?? null;
                $groomName = $itemsData['groom_name'] ?? null;
                $brideGroomPhone = $itemsData['bride_groom_phone'] ?? null;
                
                return response()->json([
                    'reservation' => $reservation,
                    'customer' => $reservation->customer,
                    'salon' => $reservation->salon,
                    'items' => $items,
                    'total_price' => $reservation->total_price,
                    'deposit_paid' => $reservation->deposit_paid ?? 0,
                    'remaining_amount' => $reservation->remaining_amount ?? 0,
                    'bride_groom_name' => $brideGroomName,
                    'bride_name' => $brideName,
                    'groom_name' => $groomName,
                    'bride_groom_phone' => $brideGroomPhone,
                ]);
            }

            return view('admin.reservations.show', compact('reservation'));
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            \Log::error('Reservation Not Found: ' . $e->getMessage(), [
                'id' => $id,
                'user_id' => Auth::id(),
            ]);
            
            // AJAX isteği ise JSON döndür
            if (request()->ajax() || request()->wantsJson()) {
                return response()->json([
                    'error' => 'Rezervasyon bulunamadı.',
                    'message' => 'Bu rezervasyonu görüntüleme yetkiniz olmayabilir veya rezervasyon silinmiş olabilir.'
                ], 404);
            }
            
            abort(404, 'Rezervasyon bulunamadı.');
        } catch (\Exception $e) {
            \Log::error('Reservation Show Error: ' . $e->getMessage(), [
                'id' => $id,
                'user_id' => Auth::id(),
                'trace' => $e->getTraceAsString()
            ]);
            
            // AJAX isteği ise JSON döndür
            if (request()->ajax() || request()->wantsJson()) {
                return response()->json([
                    'error' => 'Rezervasyon bilgileri yüklenirken bir hata oluştu.',
                    'message' => $e->getMessage()
                ], 500);
            }
            
            return redirect()->route('admin.rezervasyonlar.index')->with('error', 'Rezervasyon bilgileri yüklenirken bir hata oluştu.');
        }
    }
    
    public function confirm($id)
    {
        $user = Auth::user();
        $query = Reservation::query();
        
        // Rezervasyoncu ise sadece kendi rezervasyonunu onaylayabilir
        if ($user->role === 'rezervasyoncu') {
            $query->where('created_by', $user->id);
        }
        
        $reservation = $query->findOrFail($id);
        
        $reservation->update([
            'status' => 'confirmed'
        ]);
        
        ActivityLog::create([
            'user_id' => Auth::id(),
            'action' => 'confirm_reservation',
            'description' => "Rezervasyon onaylandı: ID {$reservation->id}",
            'ip_address' => request()->ip(),
        ]);
        
        if (request()->ajax() || request()->wantsJson()) {
            return response()->json([
                'success' => true,
                'message' => 'Rezervasyon başarıyla onaylandı.',
            ]);
        }
        
        return redirect()->back()->with('success', 'Rezervasyon başarıyla onaylandı.');
    }
    
    public function complete($id)
    {
        try {
            $user = Auth::user();
            $query = Reservation::with(['customer', 'salon']);
            
            // Rezervasyoncu veya admin ise sadece kendi rezervasyonunu tamamlayabilir
            if ($user->role === 'rezervasyoncu' || $user->role === 'admin') {
                $query->where('created_by', $user->id);
            }
            
            $reservation = $query->findOrFail($id);
            
            // Kalan tutar varsa kasaya ekle
            $remainingAmount = $reservation->remaining_amount ?? 0;
            if ($remainingAmount > 0) {
            try {
                // Kasa hesabını bul veya oluştur
                $cashAccount = Account::where('type', 'cash')
                    ->where('is_active', true)
                    ->first();
                
                if (!$cashAccount) {
                    $cashAccount = Account::create([
                        'name' => 'Ana Kasa',
                        'type' => 'cash',
                        'balance' => 0,
                        'is_active' => true,
                    ]);
                }
                
                // Transaction oluştur - kalan ödeme
                $description = "Kalan Ödeme Alındı - Rezervasyon #{$reservation->code}";
                if ($reservation->customer) {
                    $description .= " | Müşteri: {$reservation->customer->name}";
                }
                if ($reservation->salon) {
                    $description .= " | Salon: {$reservation->salon->name}";
                }
                
                Transaction::create([
                    'account_id' => $cashAccount->id,
                    'reservation_id' => $reservation->id,
                    'customer_id' => $reservation->customer_id,
                    'type' => 'income',
                    'amount' => $remainingAmount,
                    'date' => now()->format('Y-m-d'),
                    'description' => $description,
                    'created_by' => $user->id,
                ]);
                
                // Müşteri bakiyesinden düş
                if ($reservation->customer) {
                    $reservation->customer->decrement('balance', $remainingAmount);
                }
                
                // Rezervasyonun kalan bakiyesini sıfırla
                $reservation->update([
                    'remaining_amount' => 0,
                    'deposit_paid' => $reservation->total_price // Toplam ödeme yapıldı
                ]);
                
                \Log::info('Kalan ödeme kasaya eklendi', [
                    'reservation_id' => $reservation->id,
                    'amount' => $remainingAmount,
                ]);
                
                } catch (\Exception $e) {
                    \Log::error('Kalan ödeme kaydedilemedi: ' . $e->getMessage());
                }
            }
            
            $reservation->update([
                'status' => 'completed'
            ]);
            
            try {
                ActivityLog::create([
                    'user_id' => Auth::id(),
                    'action' => 'complete_reservation',
                    'description' => "Rezervasyon tamamlandı: #{$reservation->code}" . ($remainingAmount > 0 ? " - Kalan ödeme: " . number_format($remainingAmount, 2) . " ₺" : ""),
                    'ip_address' => request()->ip(),
                ]);
            } catch (\Exception $e) {
                \Log::warning('ActivityLog kaydedilemedi: ' . $e->getMessage());
            }
            
            if (request()->ajax() || request()->wantsJson()) {
                return response()->json([
                    'success' => true,
                    'message' => 'Rezervasyon başarıyla tamamlandı.' . ($remainingAmount > 0 ? ' Kalan ' . number_format($remainingAmount, 2) . ' ₺ kasaya eklendi.' : ''),
                ]);
            }
            
            return redirect()->back()->with('success', 'Rezervasyon başarıyla tamamlandı.' . ($remainingAmount > 0 ? ' Kalan ' . number_format($remainingAmount, 2) . ' ₺ kasaya eklendi.' : ''));
        } catch (\Exception $e) {
            \Log::error('Rezervasyon tamamlama hatası: ' . $e->getMessage(), [
                'id' => $id,
                'trace' => $e->getTraceAsString()
            ]);
            
            if (request()->ajax() || request()->wantsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Rezervasyon tamamlanırken bir hata oluştu: ' . $e->getMessage()
                ], 500);
            }
            
            return redirect()->back()->with('error', 'Rezervasyon tamamlanırken bir hata oluştu.');
        }
    }

    public function edit($id)
    {
        $user = Auth::user();
        $query = Reservation::with('extras');
        
        // Rezervasyoncu ise sadece kendi rezervasyonunu düzenleyebilir
        if ($user->role === 'rezervasyoncu') {
            $query->where('created_by', $user->id);
        }
        
        $reservation = $query->findOrFail($id);
        
        // Müşterileri filtrele
        $customerQuery = Customer::query();
        if ($user->role === 'rezervasyoncu') {
            $customerQuery->where('created_by', $user->id);
        }
        $customers = $customerQuery->orderBy('name')->get();
        
        // Salonları filtrele
        $salonQuery = Salon::where('is_active', true);
        if ($user->role === 'rezervasyoncu' || $user->role === 'admin') {
            $salonQuery->where('created_by', $user->id);
        }
        $salons = $salonQuery->get();
        
        $menus = Menu::where('is_active', true)->get();
        $extras = Extra::where('is_active', true)->get();

        return view('admin.reservations.edit', compact('reservation', 'customers', 'salons', 'menus', 'extras'));
    }

    public function update(Request $request, $id)
    {
        $user = Auth::user();
        $query = Reservation::query();
        
        // Rezervasyoncu ise sadece kendi rezervasyonunu güncelleyebilir
        if ($user->role === 'rezervasyoncu') {
            $query->where('created_by', $user->id);
        }
        
        $reservation = $query->findOrFail($id);
        
        // Müşteri kontrolü - rezervasyoncu ise sadece kendi müşterilerini kullanabilir
        if ($user->role === 'rezervasyoncu') {
            $customer = Customer::where('id', $request->customer_id)
                ->where('created_by', $user->id)
                ->firstOrFail();
        }

        $validated = $request->validate([
            'customer_id' => 'required|exists:customers,id',
            'salon_id' => 'required|exists:salons,id',
            'date' => 'required|date',
            'start_time' => 'required',
            'end_time' => 'required',
            'guest_count' => 'required|integer|min:1',
            'total_price' => 'required|numeric|min:0',
            'deposit_paid' => 'nullable|numeric|min:0',
            'status' => 'required|in:open,confirmed,cancelled,completed',
            'notes' => 'nullable|string',
            'extras' => 'nullable|array',
            'bride_name' => 'nullable|string|max:255',
            'groom_name' => 'nullable|string|max:255',
            'bride_groom_phone' => 'nullable|string|max:20',
        ]);

        $validated['updated_by'] = Auth::id();
        $validated['remaining_amount'] = $validated['total_price'] - ($validated['deposit_paid'] ?? 0);
        
        // Gelin ve damat bilgilerini items_data'ya kaydet
        $itemsData = json_decode($reservation->items_data ?? '{}', true);
        if ($request->filled('bride_name')) {
            $itemsData['bride_name'] = $request->bride_name;
        }
        if ($request->filled('groom_name')) {
            $itemsData['groom_name'] = $request->groom_name;
        }
        if ($request->filled('bride_groom_phone')) {
            $itemsData['bride_groom_phone'] = $request->bride_groom_phone;
        }
        if (!empty($itemsData)) {
            $validated['items_data'] = json_encode($itemsData);
        }

        $reservation->update($validated);

        // Extras güncelle
        $reservation->extras()->detach();
        if ($request->filled('extras')) {
            foreach ($request->extras as $extraId => $data) {
                if (isset($data['quantity']) && $data['quantity'] > 0) {
                    $reservation->extras()->attach($extraId, [
                        'quantity' => $data['quantity'],
                        'price' => $data['price'] ?? 0,
                    ]);
                }
            }
        }

        ActivityLog::create([
            'user_id' => Auth::id(),
            'action' => 'update_reservation',
            'description' => "Rezervasyon güncellendi: ID {$reservation->id}",
            'ip_address' => $request->ip(),
        ]);

        return redirect()->route('admin.rezervasyonlar.index')
            ->with('success', 'Rezervasyon başarıyla güncellendi.');
    }

    public function destroy($id)
    {
        $user = Auth::user();
        $query = Reservation::query();
        
        // Rezervasyoncu ise sadece kendi rezervasyonunu silebilir
        if ($user->role === 'rezervasyoncu') {
            $query->where('created_by', $user->id);
        }
        
        $reservation = $query->findOrFail($id);
        $id = $reservation->id;
        $reservation->delete();

        ActivityLog::create([
            'user_id' => Auth::id(),
            'action' => 'delete_reservation',
            'description' => "Rezervasyon silindi: ID {$id}",
            'ip_address' => request()->ip(),
        ]);

        return redirect()->route('admin.rezervasyonlar.index')
            ->with('success', 'Rezervasyon başarıyla silindi.');
    }

    public function daily(Request $request)
    {
        try {
            $date = $request->get('date', now()->format('Y-m-d'));
            $dateColumn = $this->getReservationDateColumn();
            $reservations = Reservation::whereDate($dateColumn, $date)
                ->with(['customer', 'salon'])
                ->orderBy('start_time')
                ->get();

            return view('admin.reservations.daily', compact('reservations', 'date'));
        } catch (\Exception $e) {
            \Log::error('Reservation Daily Error: ' . $e->getMessage());
            return view('admin.reservations.daily', [
                'reservations' => collect([]),
                'date' => $request->get('date', now()->format('Y-m-d'))
            ])->with('error', 'Rezervasyonlar yüklenirken bir hata oluştu.');
        }
    }

    public function cancelled()
    {
        $reservations = Reservation::where('status', 'cancelled')
            ->with(['customer', 'salon'])
            ->orderBy('date', 'desc')
            ->paginate(20);

        return view('admin.reservations.cancelled', compact('reservations'));
    }

    public function open()
    {
        $reservations = Reservation::where('status', 'open')
            ->with(['customer', 'salon'])
            ->orderBy('date', 'asc')
            ->paginate(20);

        return view('admin.reservations.open', compact('reservations'));
    }

    /**
     * Fatura ve Sözleşme PDF'lerini oluştur ve kaydet
     */
    private function generateInvoiceAndContract($reservation)
    {
        // Ayarları al
        $settings = Setting::where('group', 'invoice')->pluck('value', 'key');
        $logo = $settings['invoice_logo'] ?? null;
        $address = $reservation->salon->address ?? ($settings['invoice_address'] ?? '');
        $taxOffice = $settings['invoice_tax_office'] ?? '';
        $taxNumber = $settings['invoice_tax_number'] ?? '';

        // Fatura verilerini hazırla
        $invoiceData = $this->prepareInvoiceData($reservation, $logo, $address, $taxOffice, $taxNumber);
        
        // Fatura PDF oluştur
        $invoicePdf = Pdf::loadView('admin.invoices.pdf', $invoiceData);
        $invoicePdf->setOption('fontDir', storage_path('fonts'));
        $invoicePdf->setOption('defaultFont', 'dejavu sans');
        $invoicePath = 'invoices/reservation-' . $reservation->id . '-' . time() . '.pdf';
        Storage::disk('public')->put($invoicePath, $invoicePdf->output());
        
        // Varsayılan sözleşme linkini al
        $defaultContract = Contract::getDefaultForCompany(auth()->user()->company_id);
        $contractUrl = null;
        
        if ($defaultContract) {
            // Varsayılan sözleşme varsa onun public linkini kullan
            $contractUrl = $defaultContract->public_url;
        } else {
            // Yoksa PDF oluştur
            $contractPdf = $this->generateContractPdf($reservation, $logo, $address);
            $contractPath = 'contracts/reservation-' . $reservation->id . '-' . time() . '.pdf';
            Storage::disk('public')->put($contractPath, $contractPdf->output());
            $contractUrl = url('/storage/' . $contractPath);
        }
        
        // PDF linklerini items_data'ya kaydet (notes'a değil!)
        $itemsData = json_decode($reservation->items_data ?? '{}', true);
        if (!is_array($itemsData)) {
            $itemsData = [];
        }
        $itemsData['invoice_pdf'] = $invoicePath;
        $itemsData['invoice_url'] = url('/storage/' . $invoicePath);
        $itemsData['contract_url'] = $contractUrl;
        
        $reservation->items_data = json_encode($itemsData);
        $reservation->save();
        
        return [
            'invoice_url' => $itemsData['invoice_url'],
            'contract_url' => $contractUrl
        ];
    }

    /**
     * Sözleşme PDF oluştur
     */
    private function generateContractPdf($reservation, $logo, $address)
    {
        // Salon'a özel sözleşme içeriği (şimdilik genel)
        $contractContent = "REZERVASYON SÖZLEŞMESİ\n\n";
        $contractContent .= "Bu sözleşme, " . $reservation->customer->name . " ile yapılmıştır.\n\n";
        $contractContent .= "Rezervasyon Detayları:\n";
        $contractContent .= "Tarih: " . $reservation->date->format('d.m.Y') . "\n";
        $contractContent .= "Saat: " . $reservation->start_time . " - " . $reservation->end_time . "\n";
        $contractContent .= "Salon: " . $reservation->salon->name . "\n";
        $contractContent .= "Toplam Tutar: " . number_format($reservation->total_price, 2, ',', '.') . " ₺\n";
        $contractContent .= "Alınan Kapora: " . number_format($reservation->deposit_paid ?? 0, 2, ',', '.') . " ₺\n\n";
        $contractContent .= "Taraflar bu sözleşmeyi kabul etmişlerdir.\n\n";
        $contractContent .= "İmza: ___________________\n";
        $contractContent .= "Tarih: " . now()->format('d.m.Y');
        
        $data = [
            'reservation' => $reservation,
            'content' => $contractContent,
            'logo' => $logo,
            'address' => $address,
        ];
        
        $pdf = Pdf::loadView('admin.contracts.pdf', $data);
        $pdf->setOption('fontDir', storage_path('fonts'));
        $pdf->setOption('defaultFont', 'dejavu sans');
        return $pdf;
    }

    /**
     * Rezervasyon için fatura oluştur
     */
    private function createInvoiceForReservation($reservation)
    {
        // Rezervasyonu yükle
        $reservation = Reservation::with(['customer', 'salon'])->find($reservation->id);
        
        // Fatura numarası oluştur
        $invoiceNumber = 'FAT-' . date('Y') . '-' . str_pad(Invoice::count() + 1, 6, '0', STR_PAD_LEFT);
        
        // Public token oluştur
        $publicToken = Str::random(32);
        
        // Ayarları al
        $settings = Setting::where('group', 'invoice')->pluck('value', 'key');
        $logo = $settings['invoice_logo'] ?? null;
        $address = $reservation->salon->address ?? ($settings['invoice_address'] ?? '');
        $taxOffice = $settings['invoice_tax_office'] ?? '';
        $taxNumber = $settings['invoice_tax_number'] ?? '';
        
        // Fatura verilerini hazırla
        $invoiceData = $this->prepareInvoiceData($reservation, $logo, $address, $taxOffice, $taxNumber);
        
        // PDF oluştur
        $pdf = Pdf::loadView('admin.invoices.pdf', $invoiceData);
        
        // PDF'i kaydet
        $pdfPath = 'invoices/' . $invoiceNumber . '.pdf';
        Storage::disk('public')->put($pdfPath, $pdf->output());
        
        // Faturayı veritabanına kaydet
        $invoice = Invoice::create([
            'reservation_id' => $reservation->id,
            'customer_id' => $reservation->customer_id,
            'invoice_number' => $invoiceNumber,
            'total_amount' => $reservation->total_price,
            'deposit_paid' => $reservation->deposit_paid ?? 0,
            'remaining_amount' => $reservation->remaining_amount ?? ($reservation->total_price - ($reservation->deposit_paid ?? 0)),
            'invoice_date' => now(),
            'pdf_path' => $pdfPath,
            'public_token' => $publicToken,
            'public_url' => $this->getInvoicePublicUrl($publicToken),
            'status' => 'sent',
            'items_json' => $invoiceData['items'],
            'created_by' => Auth::id(),
        ]);
        
        return $invoice;
    }
    
    /**
     * Fatura verilerini hazırla
     */
    private function prepareInvoiceData($reservation, $logo, $address, $taxOffice, $taxNumber, $invoice = null)
    {
        $items = [];
        
        // items_data veya notes'tan JSON verilerini çıkar
        $itemsData = json_decode($reservation->items_data ?? '{}', true);
        if (!is_array($itemsData) || empty($itemsData)) {
            $itemsData = json_decode($reservation->notes ?? '{}', true);
            if (!is_array($itemsData)) {
                $itemsData = [];
            }
        }
        
        $mealTotal = 0;
        $organizationTotal = 0;
        $extraTotal = 0;
        
        // Yemek toplamı
        if (isset($itemsData['meals']) && is_array($itemsData['meals'])) {
            foreach ($itemsData['meals'] as $meal) {
                if (isset($meal['name']) && isset($meal['quantity']) && isset($meal['price'])) {
                    $total = $meal['quantity'] * $meal['price'];
                    $mealTotal += $total;
                }
            }
        }
        
        // Organizasyon toplamı
        if (isset($itemsData['organizations']) && is_array($itemsData['organizations'])) {
            foreach ($itemsData['organizations'] as $org) {
                if (isset($org['quantity']) && isset($org['price'])) {
                    $total = $org['quantity'] * $org['price'];
                    $organizationTotal += $total;
                }
            }
        }
        
        // Ekstra toplamı
        if (isset($itemsData['extras']) && is_array($itemsData['extras'])) {
            foreach ($itemsData['extras'] as $extra) {
                if (isset($extra['name']) && isset($extra['price']) && $extra['price'] > 0) {
                    $extraTotal += $extra['price'];
                }
            }
        }
        
        // Salon ücreti
        $salonPrice = $reservation->total_price - ($mealTotal + $organizationTotal + $extraTotal);
        
        if ($salonPrice > 0) {
            $items[] = [
                'name' => 'Salon Ucreti',
                'quantity' => '-',
                'price' => $salonPrice,
                'total' => $salonPrice
            ];
        }
        
        // Yemekler
        if (isset($itemsData['meals']) && is_array($itemsData['meals'])) {
            foreach ($itemsData['meals'] as $meal) {
                if (isset($meal['name']) && isset($meal['quantity']) && isset($meal['price'])) {
                    $total = $meal['quantity'] * $meal['price'];
                    $items[] = [
                        'name' => $meal['name'],
                        'quantity' => $meal['quantity'] . ' Kisi',
                        'price' => $meal['price'],
                        'total' => $total
                    ];
                }
            }
        }
        
        // Ekstralar
        if (isset($itemsData['extras']) && is_array($itemsData['extras'])) {
            foreach ($itemsData['extras'] as $extra) {
                if (isset($extra['name']) && isset($extra['price']) && $extra['price'] > 0) {
                    $items[] = [
                        'name' => $extra['name'],
                        'quantity' => '-',
                        'price' => $extra['price'],
                        'total' => $extra['price']
                    ];
                }
            }
        }
        
        // Toplam fiyat ve kapora
        $grandTotal = $reservation->total_price;
        $depositPaid = $reservation->deposit_paid ?? 0;
        
        // Salon adı ve adresi
        $salonName = $reservation->salon->name ?? 'Dugun Salonu';
        $salonAddress = $reservation->salon->address ?? $address;
        
        return [
            'reservation' => $reservation,
            'invoice' => $invoice,
            'items' => $items,
            'subtotal' => $grandTotal,
            'grandTotal' => $grandTotal,
            'depositPaid' => $depositPaid,
            'logo' => $logo,
            'address' => $address,
            'salonName' => $salonName,
            'salonAddress' => $salonAddress,
            'taxOffice' => $taxOffice,
            'taxNumber' => $taxNumber,
        ];
    }
    
    /**
     * Rezervasyon durumunu güncelle
     */
    public function updateStatus(Request $request, $id)
    {
        try {
            $user = Auth::user();
            $query = Reservation::query();
            
            // Süper admin değilse sadece kendi rezervasyonlarını güncelleyebilir
            if ($user->role !== 'super_admin') {
                $query->where('created_by', $user->id);
            }
            
            $reservation = $query->findOrFail($id);
            
            $validated = $request->validate([
                'status' => 'required|in:open,confirmed,completed,cancelled',
                'mark_paid' => 'nullable|boolean'
            ]);
            
            $oldStatus = $reservation->status;
            $oldDepositPaid = $reservation->deposit_paid;
            
            // mark_paid değerini al (JSON body'den veya form'dan)
            $markPaid = $request->input('mark_paid', false);
            if ($request->isJson()) {
                $markPaid = $request->json('mark_paid', false);
            }
            $markPaid = filter_var($markPaid, FILTER_VALIDATE_BOOLEAN);
            
            // Ödeme alındı olarak işaretle
            if ($markPaid && $validated['status'] === 'completed') {
                // Rezervasyonu customer ile birlikte yükle
                $reservation->load('customer');
                
                $remainingAmount = $reservation->total_price - ($reservation->deposit_paid ?? 0);
                
                try {
                    // 1. Kasa hesabını al
                    $cashAccount = \App\Models\Account::getCashAccount($user->id);
                    
                    // 2. Ödeme işlemleri (kalan ödeme varsa)
                    if ($remainingAmount > 0) {
                        // Kasaya gelir kaydı oluştur (Transaction)
                        \App\Models\Transaction::create([
                            'account_id' => $cashAccount->id,
                            'reservation_id' => $reservation->id,
                            'customer_id' => $reservation->customer_id,
                            'type' => 'income',
                            'amount' => $remainingAmount,
                            'description' => "Rezervasyon #{$reservation->code} kalan ödeme alındı - " . ($reservation->customer->name ?? 'Müşteri'),
                            'date' => now()->format('Y-m-d'),
                            'created_by' => $user->id
                        ]);
                        
                        // Müşteri bakiyesinden düş (varsa)
                        // Balance pozitif = müşteri bize borçlu, negatif = biz müşteriye borçluyuz
                        // Ödeme alındığında borç azalır (decrement)
                        if ($reservation->customer_id) {
                            $customer = \App\Models\Customer::find($reservation->customer_id);
                            if ($customer && $customer->balance !== null) {
                                $customer->decrement('balance', $remainingAmount);
                            }
                        }
                    }
                    
                    // 3. Rezervasyonun kalan bakiyesini sıfırla ve deposit_paid'i güncelle
                    $reservation->remaining_amount = 0;
                    $reservation->deposit_paid = $reservation->total_price;
                    
                    \Log::info('Ödeme alındı - rezervasyon güncellendi', [
                        'reservation_id' => $reservation->id,
                        'remaining_amount' => $remainingAmount,
                        'new_remaining' => 0,
                        'new_deposit_paid' => $reservation->total_price,
                    ]);
                    
                } catch (\Exception $e) {
                    // Hata olursa logla ama devam et
                    \Log::error('Ödeme işlemi hatası: ' . $e->getMessage(), [
                        'reservation_id' => $reservation->id,
                        'trace' => $e->getTraceAsString()
                    ]);
                    // Hata olsa bile deposit_paid'i güncelle
                    $reservation->remaining_amount = 0;
                    $reservation->deposit_paid = $reservation->total_price;
                }
            }
            
            $reservation->status = $validated['status'];
            $reservation->save();
            
            // Aktivite logu
            $description = "Rezervasyon #{$reservation->id} durumu '{$validated['status']}' olarak güncellendi";
            if ($markPaid) {
                $description .= " (Ödeme alındı)";
            }
            
            try {
                ActivityLog::create([
                    'user_id' => $user->id,
                    'action' => 'reservation_status_updated',
                    'model_type' => 'Reservation',
                    'model_id' => $reservation->id,
                    'description' => $description,
                    'properties' => json_encode([
                        'old_status' => $oldStatus,
                        'new_status' => $validated['status'],
                        'mark_paid' => $markPaid,
                        'old_deposit' => $oldDepositPaid,
                        'new_deposit' => $reservation->deposit_paid
                    ])
                ]);
            } catch (\Exception $e) {
                \Log::warning('ActivityLog kaydedilemedi: ' . $e->getMessage());
            }
            
            return response()->json([
                'success' => true,
                'message' => $markPaid ? 'Ödeme alındı ve rezervasyon tamamlandı.' : 'Rezervasyon durumu güncellendi.',
                'status' => $reservation->status
            ]);
        } catch (\Exception $e) {
            \Log::error('Rezervasyon durum güncelleme hatası: ' . $e->getMessage(), [
                'id' => $id,
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Rezervasyon durumu güncellenirken bir hata oluştu: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Public URL oluştur (localhost'u adugun.com'a çevir)
     */
    private function getInvoicePublicUrl($token)
    {
        $url = url('/fatura/' . $token);
        // Eğer localhost içeriyorsa adugun.com'a çevir
        if (strpos($url, 'localhost') !== false || strpos($url, '127.0.0.1') !== false) {
            $url = str_replace(['http://localhost', 'https://localhost', 'http://127.0.0.1', 'https://127.0.0.1'], 'https://adugun.com', $url);
        }
        return $url;
    }
}

