<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Salon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class SalonController extends Controller
{
    public function index()
    {
        $user = Auth::user();
        $query = Salon::query();
        
        // Rezervasyoncu ise sadece kendi salonunu göster
        if ($user->role === 'rezervasyoncu' || $user->role === 'admin') {
            $query->where('created_by', $user->id);
        }
        
        // İlk salonu getir (sadece bir salon olacak)
        $salon = $query->orderBy('name')->first();
        
        if (!$salon) {
            return redirect()->route('admin.dashboard')
                ->with('error', 'Henüz salon oluşturulmamış. İlk kayıt sırasında salon oluşturulmalıdır.');
        }
        
        return view('admin.settings.salons.index', compact('salon'));
    }

    public function create()
    {
        // Yeni salon oluşturma kaldırıldı - sadece ilk kayıtta salon oluşturulabilir
        return redirect()->route('admin.settings.salons.index')
            ->with('error', 'Yeni salon oluşturma özelliği kaldırılmıştır. Sadece ilk kayıtta salon oluşturulabilir.');
    }

    public function store(Request $request)
    {
        $user = Auth::user();
        
        // Kullanıcının zaten salonu var mı kontrol et
        $existingSalon = Salon::where('created_by', $user->id)->first();
        
        // Eğer kullanıcının salonu varsa ve bu ilk kayıt değilse (dashboard setup wizard'dan gelmiyorsa), izin verme
        if ($existingSalon && !$request->has('is_setup_wizard')) {
            if ($request->ajax()) {
                return response()->json([
                    'success' => false, 
                    'message' => 'Yeni salon oluşturma özelliği kaldırılmıştır. Sadece ilk kayıtta salon oluşturulabilir.'
                ], 403);
            }
            
            return redirect()->route('admin.settings.salons.index')
                ->with('error', 'Yeni salon oluşturma özelliği kaldırılmıştır. Sadece ilk kayıtta salon oluşturulabilir.');
        }
        
        // İlk kayıt için salon oluşturma (setup wizard'dan geliyorsa)
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'phone' => 'required|string|max:20',
            'capacity' => 'required|integer|min:1',
            'address' => 'required|string',
            'image_path' => 'nullable|string',
            'is_active' => 'boolean',
        ], [
            'name.required' => 'Salon adı zorunludur.',
            'phone.required' => 'Telefon numarası zorunludur.',
            'capacity.required' => 'Salon kapasitesi zorunludur.',
            'capacity.min' => 'Kapasite en az 1 olmalıdır.',
            'address.required' => 'Adres zorunludur.',
        ]);
        
        // Description kaldırıldı - artık kullanılmıyor
        $validated['description'] = ''; // Boş string olarak kaydet

        $validated['is_active'] = $request->has('is_active') ? true : false;
        $validated['created_by'] = Auth::id();

        Salon::create($validated);

        if ($request->ajax()) {
            return response()->json(['success' => true, 'message' => 'Salon oluşturuldu.']);
        }

        return redirect()->route('admin.settings.salons.index')
            ->with('success', 'Salon oluşturuldu.');
    }

    public function edit($id)
    {
        $user = Auth::user();
        $query = Salon::query();
        
        // Rezervasyoncu ise sadece kendi salonunu görebilir
        if ($user->role === 'rezervasyoncu' || $user->role === 'admin') {
            $query->where('created_by', $user->id);
        }
        
        $salon = $query->findOrFail($id);
        return view('admin.settings.salons.edit', compact('salon'));
    }

    public function update(Request $request, $id)
    {
        $user = Auth::user();
        $query = Salon::query();
        
        // Rezervasyoncu ise sadece kendi salonunu güncelleyebilir
        if ($user->role === 'rezervasyoncu' || $user->role === 'admin') {
            $query->where('created_by', $user->id);
        }
        
        $salon = $query->findOrFail($id);

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'phone' => 'required|string|max:20',
            'capacity' => 'required|integer|min:1',
            'address' => 'required|string',
            'is_active' => 'boolean',
        ], [
            'name.required' => 'Salon adı zorunludur.',
            'phone.required' => 'Telefon numarası zorunludur.',
            'capacity.required' => 'Salon kapasitesi zorunludur.',
            'capacity.min' => 'Kapasite en az 1 olmalıdır.',
            'address.required' => 'Adres zorunludur.',
        ]);
        
        $validated['is_active'] = $request->has('is_active') ? true : false;
        // Description kaldırıldı - artık kullanılmıyor

        $salon->update($validated);

        if ($request->ajax()) {
            return response()->json(['success' => true, 'message' => 'Salon güncellendi.']);
        }

        return redirect()->route('admin.settings.salons.index')
            ->with('success', 'Salon güncellendi.');
    }

    public function destroy(Request $request, $id)
    {
        // Salon silme özelliği kaldırıldı
        if ($request->ajax()) {
            return response()->json([
                'success' => false, 
                'message' => 'Salon silme özelliği kaldırılmıştır.'
            ], 403);
        }
        
        return redirect()->route('admin.settings.salons.index')
            ->with('error', 'Salon silme özelliği kaldırılmıştır.');
    }
}

