<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Reservation;
use App\Models\Customer;
use App\Models\Company;
use App\Models\User;
use App\Models\Salon;
use App\Models\Extra;
use Illuminate\Http\Request;

class SearchController extends Controller
{
    public function search(Request $request)
    {
        // Authentication kontrolü
        if (!auth()->check()) {
            return response()->json([
                'reservations' => [],
                'customers' => [],
                'companies' => [],
                'users' => []
            ], 401);
        }
        
        $type = $request->get('type');
        
        // Eğer type=salons ise, direkt salonları döndür
        if ($type === 'salons') {
            $user = auth()->user();
            $query = Salon::where('is_active', true);
            
            // Rezervasyoncu ise sadece kendi salonlarını göster
            if ($user && ($user->role === 'rezervasyoncu' || $user->role === 'admin')) {
                $query->where('created_by', $user->id);
            }
            
            $salons = $query->orderBy('name')
                ->get()
                ->map(function($salon) {
                    return [
                        'id' => $salon->id,
                        'title' => $salon->name,
                        'subtitle' => $salon->capacity ? "Kapasite: {$salon->capacity} kişi" : null
                    ];
                });
            
            return response()->json(['salons' => $salons]);
        }
        
        // Eğer type=extras ise, direkt ekstraları döndür
        if ($type === 'extras') {
            $extras = Extra::where('is_active', true)
                ->orderBy('name')
                ->get()
                ->map(function($extra) {
                    return [
                        'id' => $extra->id,
                        'title' => $extra->name,
                        'price' => $extra->price ?? 0
                    ];
                });
            
            return response()->json(['extras' => $extras]);
        }
        
        $query = $request->get('q', '');
        
        if (strlen($query) < 2) {
            return response()->json([
                'customers' => []
            ]);
        }

        $results = [
            'customers' => []
        ];

        $currentUser = auth()->user();
        
        // Müşteri araması ve rezervasyonları
        $customerQuery = Customer::with(['reservations' => function($q) use ($currentUser) {
            $q->with('salon')->orderBy('date', 'desc')->limit(5);
            if ($currentUser->role === 'rezervasyoncu') {
                $q->where('created_by', $currentUser->id);
            }
        }]);
        
        // Rezervasyoncu ise sadece kendi müşterilerini göster
        if ($currentUser->role === 'rezervasyoncu') {
            $customerQuery->where('created_by', $currentUser->id);
        } elseif (!$currentUser->isSuperAdmin()) {
            $companyId = $currentUser->companies->first()?->id;
            if ($companyId) {
                $companyUserIds = \App\Models\User::whereHas('companies', function($q) use ($companyId) {
                    $q->where('companies.id', $companyId);
                })->pluck('id');
                $customerQuery->whereIn('created_by', $companyUserIds);
            } else {
                $customerQuery->where('created_by', $currentUser->id);
            }
        }
        
        $customers = $customerQuery
            ->where(function($q) use ($query) {
                $q->where('name', 'like', "%{$query}%")
                  ->orWhere('phone', 'like', "%{$query}%")
                  ->orWhere('email', 'like', "%{$query}%");
            })
            ->orderBy('name')
            ->limit(20)
            ->get();
        
        // Gelin/damat ismine göre rezervasyon araması
        $reservationQuery = Reservation::with(['customer', 'salon'])
            ->where(function($q) use ($query) {
                // items_data JSON içinde gelin/damat isimlerini ara
                // MySQL JSON_EXTRACT kullanarak arama yap
                $q->whereRaw("JSON_UNQUOTE(JSON_EXTRACT(items_data, '$.bride_name')) LIKE ?", ["%{$query}%"])
                  ->orWhereRaw("JSON_UNQUOTE(JSON_EXTRACT(items_data, '$.groom_name')) LIKE ?", ["%{$query}%"])
                  ->orWhereRaw("JSON_UNQUOTE(JSON_EXTRACT(items_data, '$.bride_groom_name')) LIKE ?", ["%{$query}%"]);
            });
        
        // Rezervasyoncu ise sadece kendi rezervasyonlarını göster
        if ($currentUser->role === 'rezervasyoncu') {
            $reservationQuery->where('created_by', $currentUser->id);
        } elseif (!$currentUser->isSuperAdmin()) {
            $companyId = $currentUser->companies->first()?->id;
            if ($companyId) {
                $companyUserIds = \App\Models\User::whereHas('companies', function($q) use ($companyId) {
                    $q->where('companies.id', $companyId);
                })->pluck('id');
                $reservationQuery->whereIn('created_by', $companyUserIds);
            } else {
                $reservationQuery->where('created_by', $currentUser->id);
            }
        }
        
        $reservationsByBrideGroom = $reservationQuery
            ->orderBy('date', 'desc')
            ->limit(10)
            ->get();
        
        // Müşterileri ekle
        foreach ($customers as $customer) {
            // Müşteriyi ekle
            $results['customers'][] = [
                'id' => $customer->id,
                'title' => $customer->name,
                'url' => route('admin.customers.show', $customer->id),
                'type' => 'Müşteri',
                'icon' => 'bi-person-fill',
                'subtitle' => $customer->phone ?? '',
                'is_customer' => true
            ];
            
            // Müşterinin rezervasyonlarını ekle
            foreach ($customer->reservations as $reservation) {
                $statusText = match($reservation->status) {
                    'open' => 'Açık',
                    'confirmed' => 'Onaylı',
                    'cancelled' => 'İptal',
                    default => 'Tamamlandı'
                };
                
                $results['customers'][] = [
                    'id' => $reservation->id,
                    'title' => ($reservation->salon->name ?? 'Salon') . ' - ' . ($reservation->date ? $reservation->date->format('d.m.Y') : ''),
                    'url' => route('admin.rezervasyonlar.show', $reservation->id),
                    'type' => $statusText,
                    'icon' => 'bi-calendar-event',
                    'subtitle' => $customer->name,
                    'is_customer' => false,
                    'customer_id' => $customer->id
                ];
            }
        }
        
        // Gelin/damat ismine göre bulunan rezervasyonları ekle
        foreach ($reservationsByBrideGroom as $reservation) {
            $itemsData = json_decode($reservation->items_data ?? '{}', true);
            $brideName = $itemsData['bride_name'] ?? '';
            $groomName = $itemsData['groom_name'] ?? '';
            $brideGroomName = $itemsData['bride_groom_name'] ?? '';
            
            // Eğer bu rezervasyon zaten müşteri araması sonuçlarında varsa ekleme
            $alreadyAdded = false;
            foreach ($results['customers'] as $result) {
                if (isset($result['id']) && $result['id'] == $reservation->id && !$result['is_customer']) {
                    $alreadyAdded = true;
                    break;
                }
            }
            
            if (!$alreadyAdded) {
                $statusText = match($reservation->status) {
                    'open' => 'Açık',
                    'confirmed' => 'Onaylı',
                    'cancelled' => 'İptal',
                    default => 'Tamamlandı'
                };
                
                $subtitle = '';
                if ($brideGroomName) {
                    $subtitle = $brideGroomName;
                } elseif ($brideName && $groomName) {
                    $subtitle = $groomName . ' & ' . $brideName;
                } elseif ($groomName) {
                    $subtitle = $groomName;
                } elseif ($brideName) {
                    $subtitle = $brideName;
                } else {
                    $subtitle = $reservation->customer->name ?? 'Müşteri';
                }
                
                $results['customers'][] = [
                    'id' => $reservation->id,
                    'title' => ($reservation->salon->name ?? 'Salon') . ' - ' . ($reservation->date ? $reservation->date->format('d.m.Y') : ''),
                    'url' => route('admin.rezervasyonlar.show', $reservation->id),
                    'type' => $statusText . ' (Gelin/Damat)',
                    'icon' => 'bi-heart-fill',
                    'subtitle' => $subtitle,
                    'is_customer' => false,
                    'customer_id' => $reservation->customer_id
                ];
            }
        }

        return response()->json($results);
    }
}

