@extends('layouts.admin')

@section('title', 'Yeni Rezervasyon')

@section('content')
<div class="container-fluid">
    <div class="d-flex justify-content-between align-items-center mb-4">
        <h1 class="h3 mb-0"><i class="bi bi-plus-circle"></i> Yeni Rezervasyon Oluştur</h1>
        <a href="{{ route('admin.rezervasyonlar.index') }}" class="btn btn-secondary">
            <i class="bi bi-arrow-left"></i> Geri Dön
        </a>
    </div>

    @if($errors->any())
        <div class="alert alert-danger alert-dismissible fade show">
            <i class="bi bi-exclamation-triangle"></i> <strong>Hata!</strong> Lütfen formu kontrol edin.
            <ul class="mb-0 mt-2">
                @foreach($errors->all() as $error)
                    <li>{{ $error }}</li>
                @endforeach
            </ul>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
    @endif

    @if(session('error'))
        <div class="alert alert-danger alert-dismissible fade show">
            <i class="bi bi-exclamation-circle"></i> {{ session('error') }}
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
    @endif

    <form id="reservationForm" method="POST" action="{{ route('admin.rezervasyonlar.store') }}">
        @csrf
        
        <div class="row">
            <!-- Sol Kolon -->
            <div class="col-lg-8">
                <!-- Müşteri ve Salon -->
                <div class="card shadow-sm mb-4">
                    <div class="card-header bg-primary text-white">
                        <h5 class="mb-0"><i class="bi bi-person"></i> Müşteri ve Salon Bilgileri</h5>
                    </div>
                    <div class="card-body">
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label fw-bold">Müşteri <span class="text-danger">*</span></label>
                                <div class="input-group">
                                    <input type="text" id="customer_search" class="form-control" 
                                           placeholder="Müşteri ara..." autocomplete="off">
                                    <button type="button" class="btn btn-outline-secondary" onclick="clearCustomerSearch()">
                                        <i class="bi bi-x"></i>
                                    </button>
                                </div>
                                <select name="customer_id" id="customer_id" class="form-select mt-2" required>
                                    <option value="">Müşteri Seçiniz</option>
                                    @foreach($customers as $customer)
                                        <option value="{{ $customer->id }}" {{ old('customer_id') == $customer->id ? 'selected' : '' }}
                                            data-name="{{ $customer->name }}"
                                            data-phone="{{ $customer->phone ?? '' }}"
                                            data-email="{{ $customer->email ?? '' }}">
                                            {{ $customer->name }} @if($customer->phone)({{ $customer->phone }})@endif
                                        </option>
                                    @endforeach
                                </select>
                                <small class="text-muted">Müşteri adı veya telefon ile arayabilirsiniz</small>
                            </div>
                            
                            <div class="col-md-6 mb-3">
                                <label class="form-label fw-bold">Salon <span class="text-danger">*</span></label>
                                <select name="salon_id" id="salon_id" class="form-select" required>
                                    <option value="">Salon Seçiniz</option>
                                    @foreach($salons as $salon)
                                        <option value="{{ $salon->id }}" {{ old('salon_id') == $salon->id ? 'selected' : '' }}
                                            data-capacity="{{ $salon->capacity ?? 0 }}">
                                            {{ $salon->name }} @if($salon->capacity)(Kapasite: {{ $salon->capacity }})@endif
                                        </option>
                                    @endforeach
                                </select>
                            </div>
                        </div>
                        
                        <!-- Gelin ve Damat Bilgileri - Müşteri kutusunun altında -->
                        <div class="row mt-3">
                            <div class="col-md-5 mb-3">
                                <label class="form-label fw-bold">Gelin ve Damat İsmi</label>
                                <input type="text" name="bride_groom_name" id="bride_groom_name" 
                                       class="form-control" value="{{ old('bride_groom_name') }}" 
                                       placeholder="Örn: Ahmet Yılmaz & Ayşe Demir">
                            </div>
                            <div class="col-md-7 mb-3">
                                <label class="form-label fw-bold">İletişim Numarası</label>
                                <input type="tel" name="bride_groom_phone" id="bride_groom_phone" 
                                       class="form-control" value="{{ old('bride_groom_phone') }}" 
                                       placeholder="05XX XXX XX XX">
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Tarih ve Saat -->
                <div class="card shadow-sm mb-4">
                    <div class="card-header bg-info text-white">
                        <h5 class="mb-0"><i class="bi bi-calendar"></i> Tarih ve Saat</h5>
                    </div>
                    <div class="card-body">
                        <div class="row">
                            <div class="col-md-4 mb-3">
                                <label class="form-label fw-bold">Tarih <span class="text-danger">*</span></label>
                                <input type="date" name="date" id="reservation_date" class="form-control" 
                                       value="{{ old('date', date('Y-m-d')) }}" 
                                       min="{{ date('Y-m-d') }}" required>
                            </div>
                            
                            <div class="col-md-4 mb-3">
                                <label class="form-label fw-bold">Başlangıç Saati <span class="text-danger">*</span></label>
                                <input type="time" name="start_time" id="start_time" class="form-control" 
                                       value="{{ old('start_time') }}" required>
                            </div>
                            
                            <div class="col-md-4 mb-3">
                                <label class="form-label fw-bold">Bitiş Saati <span class="text-danger">*</span></label>
                                <input type="time" name="end_time" id="end_time" class="form-control" 
                                       value="{{ old('end_time') }}" required>
                            </div>
                        </div>
                        
                        <div class="alert alert-info mb-0" id="timeCheckResult" style="display: none;">
                            <i class="bi bi-info-circle"></i> <span id="timeCheckMessage"></span>
                        </div>
                    </div>
                </div>

                <!-- Misafir Sayısı ve Durum -->
                <div class="card shadow-sm mb-4">
                    <div class="card-header bg-success text-white">
                        <h5 class="mb-0"><i class="bi bi-people"></i> Detaylar</h5>
                    </div>
                    <div class="card-body">
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label fw-bold">Misafir Sayısı <span class="text-danger">*</span></label>
                                <input type="number" name="guest_count" id="guest_count" class="form-control" 
                                       value="{{ old('guest_count', 1) }}" min="1" required>
                            </div>
                            
                            <div class="col-md-6 mb-3">
                                <label class="form-label fw-bold">Durum <span class="text-danger">*</span></label>
                                <select name="status" id="status" class="form-select" required>
                                    <option value="open" {{ old('status', 'open') == 'open' ? 'selected' : '' }}>Açık</option>
                                    <option value="confirmed" {{ old('status') == 'confirmed' ? 'selected' : '' }}>Onaylandı</option>
                                    <option value="completed" {{ old('status') == 'completed' ? 'selected' : '' }}>Tamamlandı</option>
                                </select>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Menü/Yemekler -->
                @if(isset($menus) && $menus->count() > 0)
                <div class="card shadow-sm mb-4">
                    <div class="card-header bg-warning text-dark">
                        <h5 class="mb-0"><i class="bi bi-cup-hot"></i> Menü/Yemekler</h5>
                    </div>
                    <div class="card-body">
                        <div id="mealsContainer">
                            <div class="meal-item mb-3 p-3 border rounded">
                                <div class="row align-items-end">
                                    <div class="col-md-5">
                                        <label class="form-label">Menü Seçiniz</label>
                                        <select name="meals[0][menu_id]" class="form-select meal-menu-select" data-index="0">
                                            <option value="">Menü Seçiniz</option>
                                            @foreach($menus as $menu)
                                                <option value="{{ $menu->id }}" data-price="{{ $menu->price }}">
                                                    {{ $menu->name }} - {{ number_format($menu->price, 2) }} ₺
                                                </option>
                                            @endforeach
                                        </select>
                                    </div>
                                    <div class="col-md-3">
                                        <label class="form-label">Adet</label>
                                        <input type="number" name="meals[0][quantity]" class="form-control meal-quantity" 
                                               value="1" min="1" data-index="0">
                                    </div>
                                    <div class="col-md-3">
                                        <label class="form-label">Birim Fiyat (₺)</label>
                                        <input type="number" step="0.01" name="meals[0][price]" class="form-control meal-price" 
                                               value="0" min="0" readonly data-index="0">
                                    </div>
                                    <div class="col-md-1">
                                        <button type="button" class="btn btn-danger remove-meal" style="display: none;">
                                            <i class="bi bi-trash"></i>
                                        </button>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <button type="button" class="btn btn-sm btn-outline-primary" id="addMealBtn">
                            <i class="bi bi-plus-circle"></i> Menü Ekle
                        </button>
                    </div>
                </div>
                @endif

                <!-- Ekstra Hizmetler -->
                @if(isset($extras) && $extras->count() > 0)
                <div class="card shadow-sm mb-4">
                    <div class="card-header bg-secondary text-white">
                        <h5 class="mb-0"><i class="bi bi-star"></i> Ekstra Hizmetler</h5>
                    </div>
                    <div class="card-body">
                        <div id="extrasContainer">
                            @foreach($extras as $index => $extra)
                            <div class="form-check mb-3 p-3 border rounded extra-item">
                                <input class="form-check-input extra-checkbox" type="checkbox" 
                                       name="extras[{{ $index }}][extra_id]" 
                                       value="{{ $extra->id }}" 
                                       id="extra_{{ $extra->id }}"
                                       data-price="{{ $extra->price }}">
                                <label class="form-check-label w-100" for="extra_{{ $extra->id }}">
                                    <div class="d-flex justify-content-between align-items-center">
                                        <div>
                                            <strong>{{ $extra->name }}</strong>
                                            @if($extra->description)
                                                <br><small class="text-muted">{{ $extra->description }}</small>
                                            @endif
                                        </div>
                                        <div>
                                            <span class="badge bg-primary">{{ number_format($extra->price, 2) }} ₺</span>
                                        </div>
                                    </div>
                                </label>
                                <input type="hidden" name="extras[{{ $index }}][price]" 
                                       class="extra-price-input" value="{{ $extra->price }}">
                            </div>
                            @endforeach
                        </div>
                    </div>
                </div>
                @endif

                <!-- Notlar -->
                <div class="card shadow-sm mb-4">
                    <div class="card-header bg-dark text-white">
                        <h5 class="mb-0"><i class="bi bi-file-text"></i> Notlar</h5>
                    </div>
                    <div class="card-body">
                        <textarea name="notes" id="notes" class="form-control" rows="4" 
                                  placeholder="Rezervasyon ile ilgili notlar...">{{ old('notes') }}</textarea>
                    </div>
                </div>
            </div>

            <!-- Sağ Kolon - Fiyatlandırma -->
            <div class="col-lg-4">
                <div class="card shadow-sm sticky-top" style="top: 20px;">
                    <div class="card-header bg-primary text-white">
                        <h5 class="mb-0"><i class="bi bi-calculator"></i> Fiyatlandırma</h5>
                    </div>
                    <div class="card-body">
                        <!-- Salon Ücreti -->
                        <div class="mb-3">
                            <label class="form-label fw-bold">Salon Ücreti (₺) <span class="text-danger">*</span></label>
                            <input type="number" step="0.01" name="salon_price" id="salon_price" 
                                   class="form-control" value="{{ old('salon_price', 0) }}" min="0" required>
                        </div>

                        <!-- Organizasyon Ücreti -->
                        <div class="mb-3">
                            <label class="form-label fw-bold">Organizasyon Ücreti (₺)</label>
                            <input type="number" step="0.01" name="organization_price" id="organization_price" 
                                   class="form-control" value="{{ old('organization_price', 0) }}" min="0">
                        </div>

                        <!-- Özet -->
                        <hr>
                        <div class="mb-2 d-flex justify-content-between">
                            <span>Salon Ücreti:</span>
                            <strong id="summary_salon">0.00 ₺</strong>
                        </div>
                        <div class="mb-2 d-flex justify-content-between">
                            <span>Menü Toplamı:</span>
                            <strong id="summary_meals">0.00 ₺</strong>
                        </div>
                        <div class="mb-2 d-flex justify-content-between">
                            <span>Organizasyon:</span>
                            <strong id="summary_organization">0.00 ₺</strong>
                        </div>
                        <div class="mb-2 d-flex justify-content-between">
                            <span>Ekstra Hizmetler:</span>
                            <strong id="summary_extras">0.00 ₺</strong>
                        </div>
                        <hr>
                        <div class="mb-3 d-flex justify-content-between">
                            <span class="fs-5 fw-bold">TOPLAM:</span>
                            <span class="fs-5 fw-bold text-primary" id="summary_total">0.00 ₺</span>
                        </div>

                        <!-- Kapora -->
                        <div class="mb-3">
                            <label class="form-label fw-bold">Ödenen Kapora (₺)</label>
                            <input type="number" step="0.01" name="deposit_paid" id="deposit_paid" 
                                   class="form-control" value="{{ old('deposit_paid', 0) }}" min="0">
                        </div>

                        <!-- Kalan Tutar -->
                        <div class="mb-3">
                            <label class="form-label fw-bold">Kalan Tutar</label>
                            <div class="input-group">
                                <input type="text" id="remaining_amount" class="form-control" 
                                       value="0.00 ₺" readonly>
                            </div>
                        </div>

                        <!-- Butonlar -->
                        <div class="d-grid gap-2">
                            <button type="submit" class="btn btn-primary btn-lg">
                                <i class="bi bi-check-circle"></i> Rezervasyonu Kaydet
                            </button>
                            <a href="{{ route('admin.rezervasyonlar.index') }}" class="btn btn-secondary">
                                <i class="bi bi-x-circle"></i> İptal
                            </a>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </form>
</div>

<script>
let mealCount = 1;

// Menü ekle
const addMealBtn = document.getElementById('addMealBtn');
if (addMealBtn) {
    addMealBtn.addEventListener('click', function() {
        const container = document.getElementById('mealsContainer');
        if (!container) return;
        
        const newMeal = document.createElement('div');
        newMeal.className = 'meal-item mb-3 p-3 border rounded';
        
        // Menü seçeneklerini oluştur
        let menuOptions = '<option value="">Menü Seçiniz</option>';
        @if(isset($menus) && $menus->count() > 0)
        @foreach($menus as $menu)
        menuOptions += '<option value="{{ $menu->id }}" data-price="{{ $menu->price }}">{{ $menu->name }} - {{ number_format($menu->price, 2) }} ₺</option>';
        @endforeach
        @endif
        
        newMeal.innerHTML = `
            <div class="row align-items-end">
                <div class="col-md-5">
                    <label class="form-label">Menü Seçiniz</label>
                    <select name="meals[${mealCount}][menu_id]" class="form-select meal-menu-select" data-index="${mealCount}">
                        ${menuOptions}
                    </select>
                </div>
                <div class="col-md-3">
                    <label class="form-label">Adet</label>
                    <input type="number" name="meals[${mealCount}][quantity]" class="form-control meal-quantity" 
                           value="1" min="1" data-index="${mealCount}">
                </div>
                <div class="col-md-3">
                    <label class="form-label">Birim Fiyat (₺)</label>
                    <input type="number" step="0.01" name="meals[${mealCount}][price]" class="form-control meal-price" 
                           value="0" min="0" readonly data-index="${mealCount}">
                </div>
                <div class="col-md-1">
                    <button type="button" class="btn btn-danger remove-meal">
                        <i class="bi bi-trash"></i>
                    </button>
                </div>
            </div>
        `;
        container.appendChild(newMeal);
        mealCount++;
        
        // İlk menü item'ındaki sil butonunu göster
        if (container.children.length > 1) {
            container.querySelectorAll('.remove-meal').forEach(btn => btn.style.display = 'block');
        }
        
        attachMealEvents(newMeal);
    });
}

// Menü sil
document.addEventListener('click', function(e) {
    if (e.target.closest('.remove-meal')) {
        const mealItem = e.target.closest('.meal-item');
        mealItem.remove();
        calculateTotal();
        
        // Eğer tek menü kaldıysa sil butonunu gizle
        const container = document.getElementById('mealsContainer');
        if (container && container.children.length === 1) {
            container.querySelector('.remove-meal').style.display = 'none';
        }
    }
});

// Menü event'lerini ekle
function attachMealEvents(element) {
    const menuSelect = element.querySelector('.meal-menu-select');
    const quantityInput = element.querySelector('.meal-quantity');
    const priceInput = element.querySelector('.meal-price');
    const index = menuSelect.dataset.index;
    
    menuSelect.addEventListener('change', function() {
        const selectedOption = this.options[this.selectedIndex];
        const price = selectedOption ? parseFloat(selectedOption.dataset.price || 0) : 0;
        priceInput.value = price.toFixed(2);
        calculateTotal();
    });
    
    quantityInput.addEventListener('input', function() {
        calculateTotal();
    });
}

// İlk menü için event'leri ekle
document.querySelectorAll('.meal-item').forEach(item => {
    attachMealEvents(item);
});

// Ekstra hizmet checkbox'ları
document.querySelectorAll('.extra-checkbox').forEach(checkbox => {
    checkbox.addEventListener('change', function() {
        calculateTotal();
    });
});

// Fiyat hesaplama
function calculateTotal() {
    // Salon ücreti
    const salonPrice = parseFloat(document.getElementById('salon_price').value || 0);
    
    // Menü toplamı
    let mealTotal = 0;
    document.querySelectorAll('.meal-item').forEach(item => {
        const quantity = parseFloat(item.querySelector('.meal-quantity').value || 0);
        const price = parseFloat(item.querySelector('.meal-price').value || 0);
        mealTotal += quantity * price;
    });
    
    // Organizasyon ücreti
    const organizationPrice = parseFloat(document.getElementById('organization_price').value || 0);
    
    // Ekstra hizmetler
    let extraTotal = 0;
    document.querySelectorAll('.extra-checkbox:checked').forEach(checkbox => {
        extraTotal += parseFloat(checkbox.dataset.price || 0);
    });
    
    // Toplam
    const total = salonPrice + mealTotal + organizationPrice + extraTotal;
    
    // Kalan tutar
    const depositPaid = parseFloat(document.getElementById('deposit_paid').value || 0);
    const remaining = total - depositPaid;
    
    // Güncelle
    document.getElementById('summary_salon').textContent = salonPrice.toFixed(2) + ' ₺';
    document.getElementById('summary_meals').textContent = mealTotal.toFixed(2) + ' ₺';
    document.getElementById('summary_organization').textContent = organizationPrice.toFixed(2) + ' ₺';
    document.getElementById('summary_extras').textContent = extraTotal.toFixed(2) + ' ₺';
    document.getElementById('summary_total').textContent = total.toFixed(2) + ' ₺';
    document.getElementById('remaining_amount').value = remaining.toFixed(2) + ' ₺';
}

// İlk hesaplama
calculateTotal();

// Input değişikliklerinde hesapla
document.getElementById('salon_price').addEventListener('input', calculateTotal);
document.getElementById('organization_price').addEventListener('input', calculateTotal);
document.getElementById('deposit_paid').addEventListener('input', calculateTotal);

// Müşteri arama
const customerSearch = document.getElementById('customer_search');
const customerSelect = document.getElementById('customer_id');

if (customerSearch && customerSelect) {
    customerSearch.addEventListener('input', function() {
        const searchTerm = this.value.toLowerCase();
        const options = customerSelect.querySelectorAll('option');
        
        options.forEach(option => {
            if (option.value === '') {
                option.style.display = 'block';
                return;
            }
            
            const name = (option.dataset.name || '').toLowerCase();
            const phone = (option.dataset.phone || '').toLowerCase();
            const text = option.textContent.toLowerCase();
            
            if (name.includes(searchTerm) || phone.includes(searchTerm) || text.includes(searchTerm)) {
                option.style.display = 'block';
            } else {
                option.style.display = 'none';
            }
        });
        
        // Eğer tek seçenek kaldıysa otomatik seç
        const visibleOptions = Array.from(options).filter(opt => opt.style.display !== 'none' && opt.value !== '');
        if (visibleOptions.length === 1) {
            customerSelect.value = visibleOptions[0].value;
            customerSearch.value = visibleOptions[0].textContent;
        }
    });
}

function clearCustomerSearch() {
    if (customerSearch) customerSearch.value = '';
    if (customerSelect) {
        customerSelect.value = '';
        customerSelect.querySelectorAll('option').forEach(opt => opt.style.display = 'block');
    }
}

// Saat kontrolü
function checkAvailableTimes() {
    const date = document.getElementById('reservation_date');
    const salonId = document.getElementById('salon_id');
    const startTime = document.getElementById('start_time');
    const endTime = document.getElementById('end_time');
    
    if (!date || !salonId || !startTime || !endTime) return;
    
    const dateVal = date.value;
    const salonIdVal = salonId.value;
    const startTimeVal = startTime.value;
    const endTimeVal = endTime.value;
    
    if (!dateVal || !salonIdVal || !startTimeVal || !endTimeVal) {
        const resultDiv = document.getElementById('timeCheckResult');
        if (resultDiv) resultDiv.style.display = 'none';
        return;
    }
    
    fetch('{{ route("admin.reservations.check-times") }}', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'X-CSRF-TOKEN': '{{ csrf_token() }}'
        },
        body: JSON.stringify({
            date: dateVal,
            salon_id: salonIdVal
        })
    })
    .then(response => response.json())
    .then(data => {
        const resultDiv = document.getElementById('timeCheckResult');
        const messageSpan = document.getElementById('timeCheckMessage');
        
        if (!resultDiv || !messageSpan) return;
        
        if (data.success && data.booked_slots) {
            const bookedSlots = data.booked_slots;
            
            // Çakışma kontrolü
            let hasConflict = false;
            bookedSlots.forEach(slot => {
                if (slot >= startTimeVal && slot < endTimeVal) {
                    hasConflict = true;
                }
            });
            
            if (hasConflict) {
                resultDiv.className = 'alert alert-warning mb-0';
                messageSpan.textContent = '⚠️ Seçilen saat aralığında başka bir rezervasyon var!';
                resultDiv.style.display = 'block';
            } else {
                resultDiv.className = 'alert alert-success mb-0';
                messageSpan.textContent = '✅ Seçilen saat aralığı müsait!';
                resultDiv.style.display = 'block';
            }
        } else {
            resultDiv.style.display = 'none';
        }
    })
    .catch(error => {
        console.error('Saat kontrolü hatası:', error);
    });
}

// Tarih, salon veya saat değiştiğinde kontrol et
const reservationDate = document.getElementById('reservation_date');
const salonId = document.getElementById('salon_id');
const startTime = document.getElementById('start_time');
const endTime = document.getElementById('end_time');

if (reservationDate) reservationDate.addEventListener('change', checkAvailableTimes);
if (salonId) salonId.addEventListener('change', checkAvailableTimes);
if (startTime) startTime.addEventListener('change', checkAvailableTimes);
if (endTime) endTime.addEventListener('change', checkAvailableTimes);
</script>
@endsection
