<?php

if (file_exists(__DIR__.'/../app/helpers.php')) {
    require __DIR__.'/../app/helpers.php';
}

use Illuminate\Foundation\Application;
use Illuminate\Foundation\Configuration\Exceptions;
use Illuminate\Foundation\Configuration\Middleware;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;
use Illuminate\Session\TokenMismatchException;

return Application::configure(basePath: dirname(__DIR__))
    ->withRouting(
        web: __DIR__.'/../routes/web.php',
        api: __DIR__.'/../routes/api.php',
        commands: __DIR__.'/../routes/console.php',
        health: '/up',
    )
    ->withMiddleware(function (Middleware $middleware): void {
        $middleware->alias([
            'check.license' => \App\Http\Middleware\CheckLicense::class,
            'mobile.only' => \App\Http\Middleware\MobileOnly::class,
            'require.salon' => \App\Http\Middleware\RequireSalon::class,
            'check.beauty-salon' => \App\Http\Middleware\CheckBeautySalon::class,
            'package.expiration' => \App\Http\Middleware\CheckPackageExpiration::class,
            'personel' => \App\Http\Middleware\EnsurePersonel::class,
        ]);

        $middleware->append(\App\Http\Middleware\CorsMiddleware::class);

        $middleware->validateCsrfTokens(except: [
            'api/mobile/*',
            'api/ios/*',
        ]);

        $middleware->redirectGuestsTo(function ($request) {
            try {
                
                if ($request->is('admin/mobile/*')) {
                    return '/admin/mobile/login';
                }
                
                return route('login');
            } catch (\Exception $e) {
                
                if ($request->is('admin/mobile/*')) {
                    return '/admin/mobile/login';
                }
                return '/admin/login';
            }
        });
    })
    ->withExceptions(function (Exceptions $exceptions): void {
        
        $exceptions->render(function (\Illuminate\Contracts\Container\BindingResolutionException $e, $request) {
            // View service provider hatası için view kullanmadan basit HTML döndür
            if (str_contains($e->getMessage(), 'Target class [view] does not exist')) {
                // View service provider yüklenmemiş, view kullanmadan basit HTML döndür
                return new \Illuminate\Http\Response(
                    '<!DOCTYPE html><html><head><meta charset="utf-8"><title>500 Server Error</title></head><body><h1>500 Server Error</h1><p>Bir hata oluştu. Lütfen daha sonra tekrar deneyin.</p></body></html>',
                    500,
                    ['Content-Type' => 'text/html; charset=utf-8']
                );
            }
            return null;
        });

        $exceptions->render(function (NotFoundHttpException $e, $request) {
            if ($request->is('admin/mobile/*')) {
                // View service provider yüklenmemiş olabilir, direkt HTML döndür
                return new \Illuminate\Http\Response('<!DOCTYPE html><html><head><meta charset="utf-8"><title>404 Not Found</title></head><body><h1>404 - Sayfa Bulunamadı</h1></body></html>', 404, ['Content-Type' => 'text/html; charset=utf-8']);
            }
            // Admin panel için özel 404 sayfası
            if ($request->is('admin/*') || $request->is('personel/*')) {
                try {
                    return response()->view('errors.404', [], 404);
                } catch (\Exception $viewException) {
                    return null;
                }
            }
            // Public için özel 404 sayfası
            try {
                return response()->view('errors.public.404', [], 404);
            } catch (\Exception $viewException) {
                return null;
            }
        });

        $exceptions->render(function (TokenMismatchException $e, $request) {
            if ($request->is('admin/mobile/*')) {
                return new \Illuminate\Http\Response('<!DOCTYPE html><html><head><meta charset="utf-8"><title>Session Expired</title></head><body><h1>Oturum Süresi Doldu</h1></body></html>', 419, ['Content-Type' => 'text/html; charset=utf-8']);
            }

            // Admin panel için özel 419 sayfası
            if ($request->is('admin/*') || $request->is('personel/*')) {
                try {
                    return response()->view('errors.419', [], 419);
                } catch (\Exception $viewException) {
                    return null;
                }
            }
            // Public için özel 419 sayfası
            try {
                return response()->view('errors.public.419', [], 419);
            } catch (\Exception $viewException) {
                return null;
            }
        });

        $exceptions->render(function (\Illuminate\Auth\AuthenticationException $e, $request) {
            if ($request->is('admin/mobile/*')) {
                
                if (!$request->session()->has('url.intended')) {
                    $request->session()->put('url.intended', $request->fullUrl());
                }
                return redirect('/admin/mobile/login')
                    ->with('error', 'Devam etmek için giriş yapmanız gerekiyor.');
            }
            return null;
        });

        $exceptions->render(function (\Illuminate\View\ViewException $e, $request) {
            if ($request->is('admin/mobile/*')) {
                
                $message = $e->getMessage();
                try {
                    if (str_contains($message, 'Authentication') || 
                        str_contains($message, 'Unauthenticated') ||
                        !auth()->check()) {
                        if (!$request->session()->has('url.intended')) {
                            $request->session()->put('url.intended', $request->fullUrl());
                        }
                        return redirect('/admin/mobile/login')
                            ->with('error', 'Devam etmek için giriş yapmanız gerekiyor.');
                    }
                } catch (\Exception $authException) {
                    
                }
            }
            return null;
        });

        $exceptions->render(function (\Throwable $e, $request) {
            if ($request->is('admin/mobile/*')) {
                
                if (!($e instanceof NotFoundHttpException) && 
                    !($e instanceof TokenMismatchException) &&
                    !($e instanceof \Illuminate\Auth\AuthenticationException) &&
                    !($e instanceof \Illuminate\View\ViewException) &&
                    !($e instanceof \Illuminate\Contracts\Container\BindingResolutionException)) {
                    // View service provider yüklenmemiş olabilir, direkt HTML döndür
                    return new \Illuminate\Http\Response('<!DOCTYPE html><html><head><meta charset="utf-8"><title>500 Server Error</title></head><body><h1>500 Server Error</h1><p>Bir hata oluştu. Lütfen daha sonra tekrar deneyin.</p></body></html>', 500, ['Content-Type' => 'text/html; charset=utf-8']);
                }
            }
            // 500 hatası için özel sayfalar
            if (!($e instanceof NotFoundHttpException) && 
                !($e instanceof TokenMismatchException) &&
                !($e instanceof \Illuminate\Auth\AuthenticationException) &&
                !($e instanceof \Illuminate\View\ViewException) &&
                !($e instanceof \Illuminate\Contracts\Container\BindingResolutionException) &&
                !($e instanceof \Illuminate\Auth\Access\AuthorizationException)) {
                try {
                    $homeUrl = url('/');
                } catch (\Throwable $urlError) {
                    $homeUrl = '/';
                }
                $fallbackHtml = '<!DOCTYPE html><html><head><meta charset="utf-8"><title>500 Sunucu Hatası</title></head><body style="font-family:sans-serif;padding:2rem;text-align:center"><h1>500 Sunucu Hatası</h1><p>Bir şeyler ters gitti. Lütfen birkaç dakika sonra tekrar deneyin.</p><p><a href="'.htmlspecialchars($homeUrl, ENT_QUOTES, 'UTF-8').'">Ana Sayfaya Dön</a></p></body></html>';
                // Admin panel için özel 500 sayfası
                if ($request->is('admin/*') || $request->is('personel/*')) {
                    try {
                        return response()->view('errors.500', [], 500);
                    } catch (\Throwable $viewException) {
                        return new \Illuminate\Http\Response($fallbackHtml, 500, ['Content-Type' => 'text/html; charset=utf-8']);
                    }
                }
                // Public için özel 500 sayfası
                try {
                    return response()->view('errors.public.500', [], 500);
                } catch (\Throwable $viewException) {
                    return new \Illuminate\Http\Response($fallbackHtml, 500, ['Content-Type' => 'text/html; charset=utf-8']);
                }
            }
            return null;
        });

        $exceptions->render(function (\Illuminate\Auth\Access\AuthorizationException $e, $request) {
            if ($request->is('admin/mobile/*')) {
                try {
                    return response()->view('admin.mobile.errors.403', [], 403);
                } catch (\Exception $viewException) {
                    return new \Illuminate\Http\Response('<!DOCTYPE html><html><head><meta charset="utf-8"><title>403 Forbidden</title></head><body><h1>403 - Erişim Reddedildi</h1></body></html>', 403, ['Content-Type' => 'text/html; charset=utf-8']);
                }
            }
            // Admin panel için özel 403 sayfası
            if ($request->is('admin/*') || $request->is('personel/*')) {
                try {
                    return response()->view('errors.403', [], 403);
                } catch (\Exception $viewException) {
                    return null;
                }
            }
            // Public için özel 403 sayfası
            try {
                return response()->view('errors.public.403', [], 403);
            } catch (\Exception $viewException) {
                return null;
            }
        });
    })->create();
