<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        // Loyalty Tiers (Seviyeler)
        Schema::create('loyalty_tiers', function (Blueprint $table) {
            $table->id();
            $table->foreignId('beauty_salon_id')->constrained('users')->onDelete('cascade');
            $table->string('name'); // Standard, VIP, Gold, Platinum
            $table->integer('level')->default(1); // 1=Standard, 2=VIP, 3=Gold, 4=Platinum
            $table->integer('min_visits')->default(0); // Minimum ziyaret sayısı
            $table->decimal('min_spend', 10, 2)->default(0); // Minimum harcama
            $table->decimal('points_per_visit', 10, 2)->default(1); // Ziyaret başına puan
            $table->decimal('points_per_spend', 10, 2)->default(0.01); // Harcama başına puan (örn: 100 TL = 1 puan)
            $table->decimal('discount_percentage', 5, 2)->default(0); // İndirim yüzdesi
            $table->boolean('priority_booking')->default(false); // Öncelikli randevu
            $table->text('benefits')->nullable(); // JSON: ["free_service", "special_event"]
            $table->boolean('is_active')->default(true);
            $table->integer('sort_order')->default(0);
            $table->timestamps();
            
            $table->index(['beauty_salon_id', 'level']);
        });

        // Customer Loyalty Status
        Schema::create('customer_loyalty', function (Blueprint $table) {
            $table->id();
            $table->foreignId('customer_id')->constrained()->onDelete('cascade');
            $table->foreignId('beauty_salon_id')->constrained('users')->onDelete('cascade');
            $table->foreignId('loyalty_tier_id')->nullable()->constrained('loyalty_tiers')->onDelete('set null');
            $table->decimal('total_points', 10, 2)->default(0);
            $table->integer('total_visits')->default(0);
            $table->decimal('total_spend', 10, 2)->default(0);
            $table->decimal('points_used', 10, 2)->default(0);
            $table->decimal('points_available', 10, 2)->default(0); // total_points - points_used
            $table->timestamp('last_visit_at')->nullable();
            $table->timestamp('tier_upgraded_at')->nullable();
            $table->timestamps();
            
            $table->unique(['customer_id', 'beauty_salon_id']);
            $table->index(['beauty_salon_id', 'loyalty_tier_id']);
        });

        // Loyalty Points History
        Schema::create('loyalty_points_history', function (Blueprint $table) {
            $table->id();
            $table->foreignId('customer_loyalty_id')->constrained('customer_loyalty')->onDelete('cascade');
            $table->foreignId('appointment_id')->nullable()->constrained('appointments')->onDelete('set null');
            $table->enum('type', ['earned', 'used', 'expired', 'adjusted'])->default('earned');
            $table->decimal('points', 10, 2);
            $table->text('description')->nullable();
            $table->json('metadata')->nullable(); // Ek bilgiler
            $table->timestamps();
            
            $table->index(['customer_loyalty_id', 'type']);
        });

        // Loyalty Rewards (Ödül Kataloğu)
        Schema::create('loyalty_rewards', function (Blueprint $table) {
            $table->id();
            $table->foreignId('beauty_salon_id')->constrained('users')->onDelete('cascade');
            $table->foreignId('loyalty_tier_id')->nullable()->constrained('loyalty_tiers')->onDelete('set null');
            $table->string('name');
            $table->text('description')->nullable();
            $table->enum('reward_type', ['discount', 'free_service', 'points', 'gift'])->default('discount');
            $table->decimal('points_required', 10, 2)->default(0);
            $table->decimal('discount_percentage', 5, 2)->nullable();
            $table->foreignId('service_id')->nullable()->constrained('services')->onDelete('set null');
            $table->integer('max_redemptions')->nullable(); // Maksimum kullanım sayısı
            $table->integer('redemptions_count')->default(0);
            $table->date('valid_from')->nullable();
            $table->date('valid_until')->nullable();
            $table->boolean('is_active')->default(true);
            $table->timestamps();
            
            $table->index(['beauty_salon_id', 'is_active']);
        });
    }

    public function down(): void
    {
        Schema::dropIfExists('loyalty_rewards');
        Schema::dropIfExists('loyalty_points_history');
        Schema::dropIfExists('customer_loyalty');
        Schema::dropIfExists('loyalty_tiers');
    }
};
