<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        // Campaigns
        Schema::create('campaigns', function (Blueprint $table) {
            $table->id();
            $table->foreignId('beauty_salon_id')->constrained('users')->onDelete('cascade');
            $table->string('name');
            $table->text('description')->nullable();
            $table->enum('type', ['time_based', 'visit_based', 'spend_based', 'segment_based'])->default('time_based');
            $table->enum('status', ['draft', 'active', 'paused', 'completed', 'cancelled'])->default('draft');
            $table->date('start_date');
            $table->date('end_date')->nullable();
            $table->decimal('budget', 10, 2)->nullable(); // Toplam bütçe
            $table->decimal('budget_used', 10, 2)->default(0);
            $table->integer('max_redemptions')->nullable(); // Maksimum kullanım
            $table->integer('redemptions_count')->default(0);
            $table->decimal('discount_percentage', 5, 2)->nullable();
            $table->decimal('discount_amount', 10, 2)->nullable();
            $table->json('target_segments')->nullable(); // ["vip", "new", "at_risk"]
            $table->json('target_services')->nullable(); // Service ID'leri
            $table->boolean('auto_stop_on_loss')->default(false); // Zarar durumunda otomatik durdur
            $table->decimal('min_roi', 5, 2)->nullable(); // Minimum ROI yüzdesi
            $table->timestamps();
            
            $table->index(['beauty_salon_id', 'status', 'start_date']);
        });

        // Campaign Rules (Kural Motoru)
        Schema::create('campaign_rules', function (Blueprint $table) {
            $table->id();
            $table->foreignId('campaign_id')->constrained('campaigns')->onDelete('cascade');
            $table->enum('rule_type', ['visit_count', 'spend_amount', 'time_of_day', 'day_of_week', 'date_range', 'customer_segment'])->default('visit_count');
            $table->string('operator')->default('>='); // >=, <=, ==, !=
            $table->string('value'); // Kural değeri
            $table->json('conditions')->nullable(); // Ek koşullar
            $table->integer('sort_order')->default(0);
            $table->timestamps();
        });

        // Campaign Performance Tracking
        Schema::create('campaign_performance', function (Blueprint $table) {
            $table->id();
            $table->foreignId('campaign_id')->constrained('campaigns')->onDelete('cascade');
            $table->date('date');
            $table->integer('sent_count')->default(0); // Gönderilen kampanya sayısı
            $table->integer('opened_count')->default(0);
            $table->integer('clicked_count')->default(0);
            $table->integer('converted_count')->default(0); // Randevu oluşturan
            $table->integer('completed_count')->default(0); // Tamamlanan randevu
            $table->decimal('revenue', 10, 2)->default(0);
            $table->decimal('cost', 10, 2)->default(0);
            $table->decimal('roi', 5, 2)->default(0); // ROI yüzdesi
            $table->timestamps();
            
            $table->unique(['campaign_id', 'date']);
        });

        // Campaign Redemptions
        Schema::create('campaign_redemptions', function (Blueprint $table) {
            $table->id();
            $table->foreignId('campaign_id')->constrained('campaigns')->onDelete('cascade');
            $table->foreignId('customer_id')->constrained()->onDelete('cascade');
            $table->foreignId('appointment_id')->nullable()->constrained('appointments')->onDelete('set null');
            $table->decimal('discount_amount', 10, 2)->default(0);
            $table->timestamp('redeemed_at');
            $table->timestamps();
            
            $table->index(['campaign_id', 'customer_id']);
        });
    }

    public function down(): void
    {
        Schema::dropIfExists('campaign_redemptions');
        Schema::dropIfExists('campaign_performance');
        Schema::dropIfExists('campaign_rules');
        Schema::dropIfExists('campaigns');
    }
};
