<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        // Membership Packages
        Schema::create('membership_packages', function (Blueprint $table) {
            $table->id();
            $table->foreignId('beauty_salon_id')->constrained('users')->onDelete('cascade');
            $table->string('name');
            $table->text('description')->nullable();
            $table->enum('period', ['monthly', 'quarterly', 'yearly'])->default('monthly');
            $table->decimal('price', 10, 2);
            $table->integer('service_limit')->nullable(); // Paket başına hizmet sayısı
            $table->decimal('amount_limit', 10, 2)->nullable(); // Paket başına tutar limiti
            $table->boolean('allow_rollover')->default(true); // Devir hakkı
            $table->integer('rollover_percentage')->default(100); // Devir yüzdesi
            $table->boolean('priority_booking')->default(true); // Öncelikli randevu
            $table->json('included_services')->nullable(); // Dahil edilen hizmetler
            $table->json('excluded_services')->nullable(); // Hariç tutulan hizmetler
            $table->boolean('is_active')->default(true);
            $table->timestamps();
            
            $table->index(['beauty_salon_id', 'is_active']);
        });

        // Customer Memberships
        Schema::create('customer_memberships', function (Blueprint $table) {
            $table->id();
            $table->foreignId('customer_id')->constrained()->onDelete('cascade');
            $table->foreignId('beauty_salon_id')->constrained('users')->onDelete('cascade');
            $table->foreignId('membership_package_id')->constrained('membership_packages')->onDelete('cascade');
            $table->enum('status', ['active', 'paused', 'cancelled', 'expired'])->default('active');
            $table->date('start_date');
            $table->date('end_date');
            $table->date('next_renewal_date')->nullable();
            $table->integer('services_used')->default(0);
            $table->integer('services_remaining')->default(0);
            $table->decimal('amount_used', 10, 2)->default(0);
            $table->decimal('amount_remaining', 10, 2)->default(0);
            $table->boolean('auto_renew')->default(true);
            $table->timestamp('paused_at')->nullable();
            $table->timestamp('cancelled_at')->nullable();
            $table->text('cancellation_reason')->nullable();
            $table->timestamps();
            
            $table->index(['customer_id', 'beauty_salon_id', 'status']);
        });

        // Membership Usage History
        Schema::create('membership_usage_history', function (Blueprint $table) {
            $table->id();
            $table->foreignId('customer_membership_id')->constrained('customer_memberships')->onDelete('cascade');
            $table->foreignId('appointment_id')->nullable()->constrained('appointments')->onDelete('set null');
            $table->foreignId('service_id')->nullable()->constrained('services')->onDelete('set null');
            $table->enum('type', ['service_used', 'amount_used', 'renewal', 'pause', 'resume', 'cancellation'])->default('service_used');
            $table->integer('services_count')->default(0);
            $table->decimal('amount', 10, 2)->default(0);
            $table->text('description')->nullable();
            $table->timestamps();
            
            $table->index(['customer_membership_id', 'type']);
        });

        // Membership Payments
        Schema::create('membership_payments', function (Blueprint $table) {
            $table->id();
            $table->foreignId('customer_membership_id')->constrained('customer_memberships')->onDelete('cascade');
            $table->decimal('amount', 10, 2);
            $table->enum('payment_type', ['initial', 'renewal', 'upgrade'])->default('initial');
            $table->enum('payment_status', ['pending', 'completed', 'failed', 'refunded'])->default('pending');
            $table->string('payment_method')->nullable();
            $table->string('transaction_id')->nullable();
            $table->timestamp('paid_at')->nullable();
            $table->timestamps();
            
            $table->index(['customer_membership_id', 'payment_status']);
        });
    }

    public function down(): void
    {
        Schema::dropIfExists('membership_payments');
        Schema::dropIfExists('membership_usage_history');
        Schema::dropIfExists('customer_memberships');
        Schema::dropIfExists('membership_packages');
    }
};
