<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        // Branches (Şubeler)
        if (!Schema::hasTable('branches')) {
            Schema::create('branches', function (Blueprint $table) {
                $table->id();
                $table->foreignId('parent_salon_id')->constrained('users')->onDelete('cascade'); // Ana salon
                $table->foreignId('branch_manager_id')->nullable()->constrained('users')->onDelete('set null'); // Şube yöneticisi
                $table->string('name');
                $table->string('slug')->unique();
                $table->string('phone')->nullable();
                $table->text('address')->nullable();
                $table->string('city')->nullable();
                $table->string('district')->nullable();
                $table->enum('status', ['active', 'inactive', 'suspended'])->default('active');
                $table->decimal('revenue_share_percentage', 5, 2)->default(0); // Gelir paylaşım yüzdesi
                $table->boolean('can_create_campaigns')->default(false); // Şube özel kampanya oluşturabilir mi
                $table->json('settings')->nullable(); // Şube özel ayarlar
                $table->timestamps();
                
                $table->index(['parent_salon_id', 'status']);
            });
        }

        // Branch Revenue Sharing
        if (!Schema::hasTable('branch_revenue_sharing')) {
            Schema::create('branch_revenue_sharing', function (Blueprint $table) {
                $table->id();
                $table->foreignId('branch_id')->constrained('branches')->onDelete('cascade');
                $table->date('period_date'); // Dönem tarihi (ay/yıl)
                $table->decimal('total_revenue', 10, 2)->default(0);
                $table->decimal('revenue_share_percentage', 5, 2)->default(0);
                $table->decimal('branch_share', 10, 2)->default(0);
                $table->decimal('parent_share', 10, 2)->default(0);
                $table->enum('status', ['pending', 'calculated', 'paid'])->default('pending');
                $table->timestamp('paid_at')->nullable();
                $table->timestamps();
                
                $table->unique(['branch_id', 'period_date']);
            });
        }

        // Branch Staff Assignments
        if (!Schema::hasTable('branch_staff')) {
            Schema::create('branch_staff', function (Blueprint $table) {
                $table->id();
                $table->foreignId('branch_id')->constrained('branches')->onDelete('cascade');
                $table->foreignId('staff_id')->constrained('users')->onDelete('cascade');
                $table->enum('role', ['manager', 'staff', 'admin'])->default('staff');
                $table->boolean('is_active')->default(true);
                $table->timestamps();
                
                $table->unique(['branch_id', 'staff_id']);
            });
        }

        // Branch Campaigns (Şube özel kampanyalar)
        if (!Schema::hasTable('branch_campaigns')) {
            Schema::create('branch_campaigns', function (Blueprint $table) {
                $table->id();
                $table->foreignId('branch_id')->constrained('branches')->onDelete('cascade');
                $table->unsignedBigInteger('campaign_id')->nullable(); // campaigns tablosu henüz yok, sonra foreign key eklenebilir
                $table->string('name');
                $table->text('description')->nullable();
                $table->enum('scope', ['branch_only', 'all_branches'])->default('branch_only');
                $table->boolean('is_active')->default(true);
                $table->timestamps();
                
                $table->index(['branch_id', 'is_active']);
                // campaigns tablosu oluşturulduktan sonra foreign key eklenebilir:
                // $table->foreign('campaign_id')->references('id')->on('campaigns')->onDelete('set null');
            });
        }
    }

    public function down(): void
    {
        Schema::dropIfExists('branch_campaigns');
        Schema::dropIfExists('branch_staff');
        Schema::dropIfExists('branch_revenue_sharing');
        Schema::dropIfExists('branches');
    }
};
