<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        // Customer Segments
        if (!Schema::hasTable('customer_segments')) {
            Schema::create('customer_segments', function (Blueprint $table) {
                $table->id();
                $table->foreignId('beauty_salon_id')->constrained('users')->onDelete('cascade');
                $table->string('name'); // Yüksek CLV, Orta CLV, Düşük CLV, VIP, At Risk
                $table->string('slug')->unique();
                $table->text('description')->nullable();
                $table->decimal('min_clv', 10, 2)->nullable(); // Minimum CLV değeri
                $table->decimal('max_clv', 10, 2)->nullable(); // Maksimum CLV değeri
                $table->integer('min_visits')->nullable();
                $table->integer('max_visits')->nullable();
                $table->integer('days_since_last_visit')->nullable(); // Son ziyaretten bu yana geçen gün
                $table->json('criteria')->nullable(); // Ek kriterler
                $table->boolean('is_active')->default(true);
                $table->timestamps();
                
                $table->index(['beauty_salon_id', 'is_active']);
            });
        }

        // Customer CLV Scores
        if (!Schema::hasTable('customer_clv_scores')) {
            Schema::create('customer_clv_scores', function (Blueprint $table) {
                $table->id();
                $table->foreignId('customer_id')->constrained()->onDelete('cascade');
                $table->foreignId('beauty_salon_id')->constrained('users')->onDelete('cascade');
                $table->foreignId('segment_id')->nullable()->constrained('customer_segments')->onDelete('set null');
                $table->decimal('clv_score', 10, 2)->default(0); // Toplam CLV değeri
                $table->decimal('average_order_value', 10, 2)->default(0); // Ortalama sipariş değeri
                $table->decimal('purchase_frequency', 5, 2)->default(0); // Satın alma sıklığı (ayda kaç kez)
                $table->integer('total_visits')->default(0);
                $table->decimal('total_spend', 10, 2)->default(0);
                $table->integer('days_since_first_visit')->default(0);
                $table->integer('days_since_last_visit')->default(0);
                $table->decimal('churn_risk_score', 5, 2)->default(0); // 0-100 arası churn riski
                $table->enum('churn_risk_level', ['low', 'medium', 'high', 'critical'])->default('low');
                $table->boolean('is_vip')->default(false);
                $table->timestamp('last_calculated_at')->nullable();
                $table->timestamps();
                
                $table->unique(['customer_id', 'beauty_salon_id']);
                $table->index(['beauty_salon_id', 'segment_id', 'churn_risk_level'], 'clv_scores_idx');
            });
        }

        // CLV Calculation History
        if (!Schema::hasTable('clv_calculation_history')) {
            Schema::create('clv_calculation_history', function (Blueprint $table) {
                $table->id();
                $table->foreignId('customer_clv_score_id')->constrained('customer_clv_scores')->onDelete('cascade');
                $table->decimal('previous_clv', 10, 2)->default(0);
                $table->decimal('new_clv', 10, 2)->default(0);
                $table->decimal('change_amount', 10, 2)->default(0);
                $table->decimal('change_percentage', 5, 2)->default(0);
                $table->text('calculation_details')->nullable(); // JSON: hesaplama detayları
                $table->timestamps();
                
                $table->index(['customer_clv_score_id', 'created_at']);
            });
        }

        // Personalized Offers (Kişiselleştirilmiş Teklifler)
        if (!Schema::hasTable('personalized_offers')) {
            Schema::create('personalized_offers', function (Blueprint $table) {
                $table->id();
                $table->foreignId('customer_id')->constrained()->onDelete('cascade');
                $table->foreignId('beauty_salon_id')->constrained('users')->onDelete('cascade');
                $table->foreignId('segment_id')->nullable()->constrained('customer_segments')->onDelete('set null');
                $table->string('offer_name');
                $table->text('description')->nullable();
                $table->enum('offer_type', ['discount', 'free_service', 'points', 'special'])->default('discount');
                $table->decimal('discount_percentage', 5, 2)->nullable();
                $table->decimal('discount_amount', 10, 2)->nullable();
                $table->foreignId('service_id')->nullable()->constrained('services')->onDelete('set null');
                $table->date('valid_from');
                $table->date('valid_until');
                $table->enum('status', ['pending', 'sent', 'redeemed', 'expired'])->default('pending');
                $table->timestamp('sent_at')->nullable();
                $table->timestamp('redeemed_at')->nullable();
                $table->foreignId('appointment_id')->nullable()->constrained('appointments')->onDelete('set null');
                $table->timestamps();
                
                $table->index(['customer_id', 'beauty_salon_id', 'status']);
            });
        }
    }

    public function down(): void
    {
        Schema::dropIfExists('personalized_offers');
        Schema::dropIfExists('clv_calculation_history');
        Schema::dropIfExists('customer_clv_scores');
        Schema::dropIfExists('customer_segments');
    }
};
