<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        // Staff Performance Metrics
        if (!Schema::hasTable('staff_performance_metrics')) {
            Schema::create('staff_performance_metrics', function (Blueprint $table) {
            $table->id();
            $table->foreignId('staff_id')->constrained('users')->onDelete('cascade');
            $table->foreignId('beauty_salon_id')->constrained('users')->onDelete('cascade');
            $table->date('period_date'); // Dönem (ay/yıl)
            $table->integer('appointments_count')->default(0);
            $table->integer('completed_appointments')->default(0);
            $table->integer('cancelled_appointments')->default(0);
            $table->decimal('total_revenue', 10, 2)->default(0);
            $table->decimal('service_revenue', 10, 2)->default(0);
            $table->integer('customer_count')->default(0);
            $table->integer('repeat_customers')->default(0);
            $table->decimal('average_rating', 3, 2)->default(0); // 0-5 arası
            $table->integer('rating_count')->default(0);
            $table->decimal('satisfaction_score', 5, 2)->default(0); // 0-100
            $table->integer('on_time_percentage')->default(100); // Zamanında gelme yüzdesi
            $table->timestamps();
            
            $table->unique(['staff_id', 'beauty_salon_id', 'period_date'], 'staff_perf_unique');
            $table->index(['beauty_salon_id', 'period_date']);
            });
        }

        // Staff Commissions
        if (!Schema::hasTable('staff_commissions')) {
            Schema::create('staff_commissions', function (Blueprint $table) {
            $table->id();
            $table->foreignId('beauty_salon_id')->constrained('users')->onDelete('cascade');
            $table->foreignId('staff_id')->constrained('users')->onDelete('cascade');
            $table->enum('commission_type', ['percentage', 'fixed', 'tiered'])->default('percentage');
            $table->decimal('commission_percentage', 5, 2)->default(0); // Yüzde bazlı
            $table->decimal('commission_amount', 10, 2)->default(0); // Sabit tutar
            $table->json('tiered_rules')->nullable(); // Katmanlı kurallar
            $table->decimal('min_threshold', 10, 2)->default(0); // Minimum eşik
            $table->decimal('max_threshold', 10, 2)->nullable(); // Maksimum eşik
            $table->boolean('is_active')->default(true);
            $table->timestamps();
            
            $table->index(['beauty_salon_id', 'staff_id', 'is_active']);
            });
        }

        // Staff Commission History
        if (!Schema::hasTable('staff_commission_history')) {
            Schema::create('staff_commission_history', function (Blueprint $table) {
            $table->id();
            $table->foreignId('staff_id')->constrained('users')->onDelete('cascade');
            $table->foreignId('appointment_id')->nullable()->constrained('appointments')->onDelete('set null');
            $table->foreignId('service_id')->nullable()->constrained('services')->onDelete('set null');
            $table->decimal('service_amount', 10, 2)->default(0);
            $table->decimal('commission_percentage', 5, 2)->default(0);
            $table->decimal('commission_amount', 10, 2)->default(0);
            $table->date('period_date');
            $table->enum('status', ['pending', 'calculated', 'paid'])->default('pending');
            $table->timestamp('paid_at')->nullable();
            $table->timestamps();
            
            $table->index(['staff_id', 'period_date', 'status']);
            });
        }

        // Staff Skills & Certifications
        if (!Schema::hasTable('staff_skills')) {
            Schema::create('staff_skills', function (Blueprint $table) {
            $table->id();
            $table->foreignId('staff_id')->constrained('users')->onDelete('cascade');
            $table->string('skill_name');
            $table->enum('level', ['beginner', 'intermediate', 'advanced', 'expert'])->default('beginner');
            $table->string('certification')->nullable();
            $table->date('certification_date')->nullable();
            $table->date('expiry_date')->nullable();
            $table->text('notes')->nullable();
            $table->timestamps();
            
            $table->index(['staff_id', 'level']);
            });
        }

        // Staff Bonuses
        if (!Schema::hasTable('staff_bonuses')) {
            Schema::create('staff_bonuses', function (Blueprint $table) {
            $table->id();
            $table->foreignId('staff_id')->constrained('users')->onDelete('cascade');
            $table->foreignId('beauty_salon_id')->constrained('users')->onDelete('cascade');
            $table->string('bonus_name');
            $table->enum('bonus_type', ['target_based', 'performance_based', 'special'])->default('target_based');
            $table->decimal('target_amount', 10, 2)->nullable();
            $table->decimal('bonus_amount', 10, 2)->default(0);
            $table->date('period_start');
            $table->date('period_end');
            $table->enum('status', ['active', 'achieved', 'paid', 'expired'])->default('active');
            $table->timestamp('paid_at')->nullable();
            $table->timestamps();
            
            $table->index(['staff_id', 'beauty_salon_id', 'status']);
            });
        }
    }

    public function down(): void
    {
        Schema::dropIfExists('staff_bonuses');
        Schema::dropIfExists('staff_skills');
        Schema::dropIfExists('staff_commission_history');
        Schema::dropIfExists('staff_commissions');
        Schema::dropIfExists('staff_performance_metrics');
    }
};
