<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        // Pricing Rules
        Schema::create('pricing_rules', function (Blueprint $table) {
            $table->id();
            $table->foreignId('beauty_salon_id')->constrained('users')->onDelete('cascade');
            $table->string('name');
            $table->text('description')->nullable();
            $table->enum('rule_type', ['time_based', 'demand_based', 'staff_based', 'seasonal', 'day_of_week'])->default('time_based');
            $table->enum('pricing_type', ['percentage', 'fixed', 'multiplier'])->default('percentage');
            $table->decimal('price_adjustment', 10, 2)->default(0); // Yüzde veya sabit tutar
            $table->decimal('multiplier', 5, 2)->default(1.0); // Çarpan (örn: 1.2 = %20 artış)
            $table->time('start_time')->nullable(); // Zaman bazlı
            $table->time('end_time')->nullable();
            $table->json('days_of_week')->nullable(); // [1,2,3] = Pazartesi, Salı, Çarşamba
            $table->date('start_date')->nullable(); // Mevsimsel
            $table->date('end_date')->nullable();
            $table->integer('demand_threshold')->nullable(); // Talep eşiği
            $table->foreignId('staff_id')->nullable()->constrained('users')->onDelete('set null'); // Personel bazlı
            $table->foreignId('service_id')->nullable()->constrained('services')->onDelete('set null');
            $table->integer('max_bookings')->nullable(); // Maksimum randevu sayısı (talep yönetimi)
            $table->boolean('is_active')->default(true);
            $table->integer('priority')->default(0); // Öncelik (yüksek sayı = önce uygulanır)
            $table->timestamps();
            
            $table->index(['beauty_salon_id', 'is_active', 'priority']);
        });

        // Dynamic Prices (Hesaplanmış fiyatlar)
        Schema::create('dynamic_prices', function (Blueprint $table) {
            $table->id();
            $table->foreignId('service_id')->constrained('services')->onDelete('cascade');
            $table->foreignId('staff_id')->nullable()->constrained('users')->onDelete('set null');
            $table->date('date');
            $table->time('time_slot')->nullable();
            $table->decimal('base_price', 10, 2);
            $table->decimal('adjusted_price', 10, 2);
            $table->decimal('adjustment_amount', 10, 2)->default(0);
            $table->decimal('adjustment_percentage', 5, 2)->default(0);
            $table->foreignId('pricing_rule_id')->nullable()->constrained('pricing_rules')->onDelete('set null');
            $table->integer('demand_level')->default(0); // 0-100 arası talep seviyesi
            $table->integer('available_slots')->default(0);
            $table->timestamps();
            
            $table->unique(['service_id', 'staff_id', 'date', 'time_slot']);
            $table->index(['date', 'demand_level']);
        });

        // Price Testing (A/B Test)
        Schema::create('price_tests', function (Blueprint $table) {
            $table->id();
            $table->foreignId('beauty_salon_id')->constrained('users')->onDelete('cascade');
            $table->foreignId('service_id')->constrained('services')->onDelete('cascade');
            $table->string('test_name');
            $table->decimal('control_price', 10, 2); // Kontrol grubu fiyatı
            $table->decimal('test_price', 10, 2); // Test grubu fiyatı
            $table->integer('control_group_size')->default(0);
            $table->integer('test_group_size')->default(0);
            $table->date('start_date');
            $table->date('end_date');
            $table->enum('status', ['draft', 'running', 'completed', 'cancelled'])->default('draft');
            $table->integer('control_conversions')->default(0);
            $table->integer('test_conversions')->default(0);
            $table->decimal('control_revenue', 10, 2)->default(0);
            $table->decimal('test_revenue', 10, 2)->default(0);
            $table->decimal('conversion_rate_control', 5, 2)->default(0);
            $table->decimal('conversion_rate_test', 5, 2)->default(0);
            $table->text('conclusion')->nullable();
            $table->timestamps();
            
            $table->index(['beauty_salon_id', 'service_id', 'status']);
        });
    }

    public function down(): void
    {
        Schema::dropIfExists('price_tests');
        Schema::dropIfExists('dynamic_prices');
        Schema::dropIfExists('pricing_rules');
    }
};
