<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        // Staff Salaries (Personel Maaşları)
        if (!Schema::hasTable('staff_salaries')) {
            Schema::create('staff_salaries', function (Blueprint $table) {
                $table->id();
                $table->foreignId('staff_id')->constrained('users')->onDelete('cascade');
                $table->foreignId('beauty_salon_id')->constrained('users')->onDelete('cascade');
                $table->decimal('base_salary', 10, 2)->default(0); // Sabit maaş
                $table->decimal('commission_rate', 5, 2)->default(0); // Komisyon oranı (%)
                $table->decimal('bonus_amount', 10, 2)->default(0); // Bonus tutarı
                $table->date('period_start'); // Maaş dönemi başlangıcı
                $table->date('period_end'); // Maaş dönemi bitişi
                $table->decimal('total_earnings', 10, 2)->default(0); // Toplam kazanç (base + commission + bonus)
                $table->decimal('total_commission', 10, 2)->default(0); // Toplam komisyon
                $table->decimal('total_bonus', 10, 2)->default(0); // Toplam bonus
                $table->integer('total_appointments')->default(0); // Toplam randevu sayısı
                $table->decimal('total_revenue', 10, 2)->default(0); // Personelin getirdiği toplam gelir
                $table->enum('status', ['draft', 'calculated', 'paid', 'cancelled'])->default('draft');
                $table->date('paid_at')->nullable();
                $table->text('notes')->nullable();
                $table->timestamps();
                
                $table->index(['staff_id', 'beauty_salon_id', 'period_start', 'period_end']);
            });
        }

        // Staff Service Tracking (Personel Hizmet Takibi)
        if (!Schema::hasTable('staff_service_tracking')) {
            Schema::create('staff_service_tracking', function (Blueprint $table) {
                $table->id();
                $table->foreignId('reservation_id')->constrained('reservations')->onDelete('cascade');
                $table->foreignId('staff_id')->constrained('users')->onDelete('cascade');
                $table->foreignId('service_id')->constrained('services')->onDelete('cascade');
                $table->foreignId('beauty_salon_id')->constrained('users')->onDelete('cascade');
                $table->decimal('service_price', 10, 2); // Hizmet fiyatı
                $table->decimal('commission_amount', 10, 2)->default(0); // Bu hizmetten kazandığı komisyon
                $table->date('service_date'); // Hizmet tarihi
                $table->time('service_time'); // Hizmet saati
                $table->enum('status', ['completed', 'cancelled', 'no_show'])->default('completed');
                $table->timestamps();
                
                $table->index(['staff_id', 'service_date']);
                $table->index(['beauty_salon_id', 'service_date']);
                $table->index(['reservation_id']);
            });
        }

        // Salary Payment History (Maaş Ödeme Geçmişi)
        if (!Schema::hasTable('staff_salary_payments')) {
            Schema::create('staff_salary_payments', function (Blueprint $table) {
                $table->id();
                $table->foreignId('staff_salary_id')->constrained('staff_salaries')->onDelete('cascade');
                $table->foreignId('staff_id')->constrained('users')->onDelete('cascade');
                $table->foreignId('beauty_salon_id')->constrained('users')->onDelete('cascade');
                $table->decimal('amount', 10, 2); // Ödenen tutar
                $table->enum('payment_method', ['cash', 'bank_transfer', 'check', 'other'])->default('bank_transfer');
                $table->string('payment_reference')->nullable(); // Ödeme referansı
                $table->date('payment_date');
                $table->text('notes')->nullable();
                $table->timestamps();
                
                $table->index(['staff_id', 'payment_date']);
            });
        }
    }

    public function down(): void
    {
        Schema::dropIfExists('staff_salary_payments');
        Schema::dropIfExists('staff_service_tracking');
        Schema::dropIfExists('staff_salaries');
    }
};

