<?php
/**
 * Storage Setup Script - Laravel Storage Klasör Kurulumu
 * 
 * Bu script storage klasör izinlerini, gerekli klasörleri ve storage link'ini otomatik olarak oluşturur.
 * 
 * KULLANIM: Tarayıcıdan http://yourdomain.com/storagesetup.php adresine gidin
 * 
 * GÜVENLİK: Production ortamında kullanım sonrası bu dosyayı silin veya şifre koruması ekleyin!
 */

// Güvenlik: Production'da devre dışı bırakılabilir
$ALLOW_ACCESS = true; // false yaparak erişimi kapatabilirsiniz

// Basit şifre koruması (isteğe bağlı - şifre belirlemek için $PASSWORD değişkenini ayarlayın)
$PASSWORD = ''; // Boş bırakılırsa şifre kontrolü yapılmaz, şifre belirlerseniz: $PASSWORD = 'güvenlişifre123';

session_start();

// Şifre kontrolü (eğer şifre belirlendiyse)
if (!empty($PASSWORD)) {
    if (isset($_POST['password'])) {
        if ($_POST['password'] === $PASSWORD) {
            $_SESSION['storagesetup_authenticated'] = true;
        } else {
            $error = 'Yanlış şifre!';
        }
    }
    
    if (!isset($_SESSION['storagesetup_authenticated']) && !isset($_POST['password'])) {
        showPasswordForm($error ?? null);
        exit;
    }
}

// Erişim kontrolü
if (!$ALLOW_ACCESS) {
    die('Erişim devre dışı bırakılmıştır. $ALLOW_ACCESS değişkenini true yapın.');
}

// Proje kök dizini (public klasörünün bir üstü)
$basePath = dirname(__DIR__);

// Sonuçları saklamak için dizi
$results = [];
$success = false;

// Form gönderildi mi?
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['setup'])) {
    $results = performSetup($basePath);
    $success = true;
}

/**
 * Setup işlemlerini gerçekleştir
 */
function performSetup($basePath) {
    $results = [];
    
    try {
        // 1. Storage klasörü izinlerini ayarla
        $storagePath = $basePath . '/storage';
        if (is_dir($storagePath)) {
            $results[] = setDirectoryPermissions($storagePath, 0775, 'storage');
        } else {
            $results[] = ['type' => 'error', 'message' => 'storage klasörü bulunamadı: ' . $storagePath];
        }
        
        // 2. Bootstrap cache klasörü izinlerini ayarla
        $bootstrapCachePath = $basePath . '/bootstrap/cache';
        if (is_dir($bootstrapCachePath)) {
            $results[] = setDirectoryPermissions($bootstrapCachePath, 0775, 'bootstrap/cache');
        } else {
            $results[] = ['type' => 'error', 'message' => 'bootstrap/cache klasörü bulunamadı: ' . $bootstrapCachePath];
        }
        
        // 3. Storage/app/public klasörünü oluştur
        $storagePublicPath = $basePath . '/storage/app/public';
        if (!is_dir($storagePublicPath)) {
            if (@mkdir($storagePublicPath, 0775, true)) {
                $results[] = ['type' => 'success', 'message' => 'storage/app/public klasörü oluşturuldu'];
            } else {
                $results[] = ['type' => 'error', 'message' => 'storage/app/public klasörü oluşturulamadı'];
            }
        } else {
            $results[] = setDirectoryPermissions($storagePublicPath, 0775, 'storage/app/public');
        }
        
        // 4. Gerekli alt klasörleri oluştur
        $requiredDirs = [
            'salon-profiles',
            'salon-banners',
            'salon-gallery',
            'personel-profiles'
        ];
        
        foreach ($requiredDirs as $dir) {
            $dirPath = $storagePublicPath . '/' . $dir;
            if (!is_dir($dirPath)) {
                if (@mkdir($dirPath, 0775, true)) {
                    $results[] = ['type' => 'success', 'message' => "storage/app/public/{$dir} klasörü oluşturuldu"];
                } else {
                    $results[] = ['type' => 'warning', 'message' => "storage/app/public/{$dir} klasörü oluşturulamadı (izin hatası olabilir)"];
                }
            } else {
                $results[] = setDirectoryPermissions($dirPath, 0775, "storage/app/public/{$dir}");
            }
        }
        
        // 5. Storage link oluştur
        $publicStoragePath = $basePath . '/public/storage';
        $storageAppPublicPath = $basePath . '/storage/app/public';
        
        // Mevcut linki kontrol et
        if (is_link($publicStoragePath)) {
            $results[] = ['type' => 'info', 'message' => 'public/storage sembolik linki zaten mevcut'];
        } elseif (is_dir($publicStoragePath)) {
            $results[] = ['type' => 'warning', 'message' => 'public/storage klasörü zaten mevcut (sembolik link oluşturulamadı)'];
        } else {
            // Windows'ta symlink çalışmayabilir, alternatif yöntem kullan
            if (PHP_OS_FAMILY === 'Windows') {
                // Windows'ta junction kullan (veya manuel olarak yapılmalı)
                $results[] = ['type' => 'warning', 'message' => 'Windows sisteminde sembolik link otomatik oluşturulamıyor. Manuel olarak "php artisan storage:link" komutunu çalıştırın.'];
            } else {
                // Unix/Linux sistemlerde symlink oluştur
                if (@symlink($storageAppPublicPath, $publicStoragePath)) {
                    $results[] = ['type' => 'success', 'message' => 'public/storage sembolik linki oluşturuldu'];
                } else {
                    $results[] = ['type' => 'warning', 'message' => 'public/storage sembolik linki oluşturulamadı. "php artisan storage:link" komutunu manuel olarak çalıştırmanız gerekebilir.'];
                }
            }
        }
        
        // 6. Artisan komutunu çalıştır (storage:link için)
        if (file_exists($basePath . '/artisan')) {
            $artisanPath = $basePath . '/artisan';
            $command = 'cd ' . escapeshellarg($basePath) . ' && php artisan storage:link 2>&1';
            $output = [];
            $returnVar = 0;
            @exec($command, $output, $returnVar);
            
            if ($returnVar === 0) {
                $results[] = ['type' => 'success', 'message' => 'php artisan storage:link komutu başarıyla çalıştırıldı'];
            } elseif (strpos(implode(' ', $output), 'already exists') !== false) {
                $results[] = ['type' => 'info', 'message' => 'Storage link zaten mevcut'];
            } else {
                $results[] = ['type' => 'warning', 'message' => 'php artisan storage:link komutu çalıştırılamadı: ' . implode(' ', $output)];
            }
        }
        
    } catch (Exception $e) {
        $results[] = ['type' => 'error', 'message' => 'Hata: ' . $e->getMessage()];
    }
    
    return $results;
}

/**
 * Klasör izinlerini ayarla
 */
function setDirectoryPermissions($path, $mode, $name) {
    if (@chmod($path, $mode)) {
        // Alt klasörlere de uygula (recursive)
        $iterator = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($path, RecursiveDirectoryIterator::SKIP_DOTS),
            RecursiveIteratorIterator::SELF_FIRST
        );
        
        $successCount = 0;
        $totalCount = 0;
        
        foreach ($iterator as $item) {
            $totalCount++;
            if (@chmod($item->getPathname(), $mode)) {
                $successCount++;
            }
        }
        
        return [
            'type' => 'success',
            'message' => "{$name} klasörü izinleri 775 olarak ayarlandı" . ($totalCount > 0 ? " ({$successCount}/{$totalCount} alt öğe)" : '')
        ];
    } else {
        return [
            'type' => 'warning',
            'message' => "{$name} klasörü izinleri ayarlanamadı (sunucu yetkisi olmayabilir)"
        ];
    }
}

/**
 * Şifre formu göster
 */
function showPasswordForm($error = null) {
    ?>
    <!DOCTYPE html>
    <html lang="tr">
    <head>
        <meta charset="UTF-8">
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
        <title>Storage Setup - Şifre Gerekli</title>
        <style>
            body {
                font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
                background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
                display: flex;
                justify-content: center;
                align-items: center;
                min-height: 100vh;
                margin: 0;
            }
            .container {
                background: white;
                padding: 40px;
                border-radius: 16px;
                box-shadow: 0 20px 60px rgba(0,0,0,0.3);
                max-width: 400px;
                width: 90%;
            }
            h1 {
                margin: 0 0 30px 0;
                color: #333;
                text-align: center;
            }
            .form-group {
                margin-bottom: 20px;
            }
            label {
                display: block;
                margin-bottom: 8px;
                color: #555;
                font-weight: 500;
            }
            input[type="password"] {
                width: 100%;
                padding: 12px;
                border: 2px solid #e0e0e0;
                border-radius: 8px;
                font-size: 16px;
                box-sizing: border-box;
            }
            input[type="password"]:focus {
                outline: none;
                border-color: #667eea;
            }
            button {
                width: 100%;
                padding: 14px;
                background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
                color: white;
                border: none;
                border-radius: 8px;
                font-size: 16px;
                font-weight: 600;
                cursor: pointer;
                transition: transform 0.2s;
            }
            button:hover {
                transform: translateY(-2px);
            }
            .error {
                background: #fee;
                color: #c33;
                padding: 12px;
                border-radius: 8px;
                margin-bottom: 20px;
            }
        </style>
    </head>
    <body>
        <div class="container">
            <h1>🔒 Şifre Gerekli</h1>
            <?php if ($error): ?>
                <div class="error"><?php echo htmlspecialchars($error); ?></div>
            <?php endif; ?>
            <form method="POST">
                <div class="form-group">
                    <label>Şifre:</label>
                    <input type="password" name="password" required autofocus>
                </div>
                <button type="submit">Giriş Yap</button>
            </form>
        </div>
    </body>
    </html>
    <?php
}

?>
<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Storage Setup - Laravel Storage Kurulumu</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            padding: 20px;
            min-height: 100vh;
        }
        .container {
            max-width: 900px;
            margin: 0 auto;
            background: white;
            border-radius: 16px;
            box-shadow: 0 20px 60px rgba(0,0,0,0.3);
            overflow: hidden;
        }
        .header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 40px;
            text-align: center;
        }
        .header h1 {
            font-size: 2.5rem;
            margin-bottom: 10px;
        }
        .header p {
            opacity: 0.9;
            font-size: 1.1rem;
        }
        .content {
            padding: 40px;
        }
        .warning-box {
            background: #fff3cd;
            border: 2px solid #ffc107;
            border-radius: 12px;
            padding: 20px;
            margin-bottom: 30px;
        }
        .warning-box h3 {
            color: #856404;
            margin-bottom: 10px;
        }
        .warning-box p {
            color: #856404;
            margin: 5px 0;
        }
        .setup-button {
            display: block;
            width: 100%;
            padding: 20px;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border: none;
            border-radius: 12px;
            font-size: 1.2rem;
            font-weight: 600;
            cursor: pointer;
            transition: transform 0.2s, box-shadow 0.2s;
            margin-bottom: 30px;
        }
        .setup-button:hover {
            transform: translateY(-2px);
            box-shadow: 0 8px 20px rgba(102, 126, 234, 0.4);
        }
        .results {
            margin-top: 30px;
        }
        .result-item {
            padding: 15px;
            border-radius: 8px;
            margin-bottom: 10px;
            display: flex;
            align-items: center;
            gap: 12px;
        }
        .result-item.success {
            background: #d4edda;
            border-left: 4px solid #28a745;
            color: #155724;
        }
        .result-item.warning {
            background: #fff3cd;
            border-left: 4px solid #ffc107;
            color: #856404;
        }
        .result-item.error {
            background: #f8d7da;
            border-left: 4px solid #dc3545;
            color: #721c24;
        }
        .result-item.info {
            background: #d1ecf1;
            border-left: 4px solid #17a2b8;
            color: #0c5460;
        }
        .result-icon {
            font-size: 1.5rem;
            flex-shrink: 0;
        }
        .result-message {
            flex: 1;
        }
        .info-box {
            background: #e7f3ff;
            border: 2px solid #2196F3;
            border-radius: 12px;
            padding: 20px;
            margin-bottom: 30px;
        }
        .info-box h3 {
            color: #0d47a1;
            margin-bottom: 10px;
        }
        .info-box ul {
            color: #0d47a1;
            margin-left: 20px;
        }
        .info-box li {
            margin: 8px 0;
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>🔧 Storage Setup</h1>
            <p>Laravel Storage Klasör Kurulum Aracı</p>
        </div>
        <div class="content">
            <div class="warning-box">
                <h3>⚠️ Güvenlik Uyarısı</h3>
                <p><strong>Bu scripti sadece kurulum sırasında kullanın!</strong></p>
                <p>Kurulum tamamlandıktan sonra bu dosyayı (<code>public/storagesetup.php</code>) silin veya şifre koruması ekleyin.</p>
            </div>

            <div class="info-box">
                <h3>ℹ️ Bu Script Ne Yapar?</h3>
                <ul>
                    <li>✅ <code>storage</code> klasörü izinlerini 775 yapar</li>
                    <li>✅ <code>bootstrap/cache</code> klasörü izinlerini 775 yapar</li>
                    <li>✅ <code>storage/app/public</code> klasörünü oluşturur</li>
                    <li>✅ Gerekli alt klasörleri oluşturur (salon-profiles, salon-banners, salon-gallery, personel-profiles)</li>
                    <li>✅ <code>php artisan storage:link</code> komutunu çalıştırır</li>
                </ul>
            </div>

            <?php if ($success): ?>
                <div class="results">
                    <h2 style="margin-bottom: 20px;">📋 İşlem Sonuçları:</h2>
                    <?php foreach ($results as $result): ?>
                        <div class="result-item <?php echo $result['type']; ?>">
                            <span class="result-icon">
                                <?php 
                                switch($result['type']) {
                                    case 'success': echo '✅'; break;
                                    case 'warning': echo '⚠️'; break;
                                    case 'error': echo '❌'; break;
                                    case 'info': echo 'ℹ️'; break;
                                }
                                ?>
                            </span>
                            <span class="result-message"><?php echo htmlspecialchars($result['message']); ?></span>
                        </div>
                    <?php endforeach; ?>
                </div>
            <?php endif; ?>

            <form method="POST">
                <input type="hidden" name="setup" value="1">
                <button type="submit" class="setup-button">
                    🚀 Kurulumu Başlat
                </button>
            </form>

            <div class="info-box">
                <h3>📝 Notlar:</h3>
                <ul>
                    <li>Bazı sunucularda sembolik link oluşturma yetkisi olmayabilir. Bu durumda hosting sağlayıcınızdan "php artisan storage:link" komutunu çalıştırmalarını isteyin.</li>
                    <li>Windows sunucularda sembolik link otomatik oluşturulamaz, manuel olarak yapılması gerekir.</li>
                    <li>Klasör izinleri bazı sunucularda değiştirilemeyebilir. Bu durumda hosting sağlayıcınızla iletişime geçin.</li>
                </ul>
            </div>
        </div>
    </div>
</body>
</html>

