@extends('layouts.admin')

@section('title', 'Randevu Takvimi')

@section('content')
<style>
    @keyframes fadeInUp {
        from {
            opacity: 0;
            transform: translateY(20px);
        }
        to {
            opacity: 1;
            transform: translateY(0);
        }
    }
    
    @keyframes shimmer {
        0% {
            background-position: -1000px 0;
        }
        100% {
            background-position: 1000px 0;
        }
    }
    
    @keyframes pulse {
        0%, 100% {
            opacity: 1;
        }
        50% {
            opacity: 0.7;
        }
    }
    
    body {
        padding-top: var(--header-height) !important;
        background: linear-gradient(135deg, #fce7f3 0%, #fdf2f8 50%, #fce7f3 100%) !important;
        background-size: 200% 200%;
        animation: gradientShift 15s ease infinite;
    }
    
    @keyframes gradientShift {
        0%, 100% {
            background-position: 0% 50%;
        }
        50% {
            background-position: 100% 50%;
        }
    }
    
    .main-content {
        padding: 0 !important;
        margin: 0 !important;
    }
    
    .calendar-page {
        padding: 32px;
        min-height: calc(100vh - var(--header-height));
        animation: fadeInUp 0.6s ease-out;
    }
    
    .calendar-header {
        background: rgba(255, 255, 255, 0.9);
        backdrop-filter: blur(30px) saturate(200%);
        -webkit-backdrop-filter: blur(30px) saturate(200%);
        border-radius: 24px;
        padding: 28px 32px;
        margin-bottom: 32px;
        display: flex;
        justify-content: space-between;
        align-items: center;
        flex-wrap: wrap;
        gap: 24px;
        box-shadow: 
            0 20px 60px rgba(236, 72, 153, 0.15),
            0 0 0 1px rgba(255, 255, 255, 0.8),
            inset 0 1px 0 rgba(255, 255, 255, 0.9);
        border: 1px solid rgba(255, 255, 255, 0.5);
        transition: all 0.2s ease;
        position: relative;
    }
    
    .calendar-header:hover {
        box-shadow: 
            0 24px 70px rgba(236, 72, 153, 0.18),
            0 0 0 1px rgba(255, 255, 255, 0.9);
    }
    
    .calendar-header h2 {
        background: linear-gradient(135deg, #ec4899 0%, #be185d 50%, #ec4899 100%);
        background-size: 200% 200%;
        -webkit-background-clip: text;
        -webkit-text-fill-color: transparent;
        background-clip: text;
        font-weight: 800;
        font-size: 32px;
        margin: 0;
        letter-spacing: -0.5px;
        animation: gradientShift 3s ease infinite;
        position: relative;
        z-index: 1;
    }
    
    .view-buttons {
        display: flex;
        gap: 12px;
        position: relative;
        z-index: 1;
    }
    
    .view-btn {
        padding: 12px 24px;
        border: 2px solid rgba(236, 72, 153, 0.2);
        background: rgba(255, 255, 255, 0.95);
        backdrop-filter: blur(10px);
        border-radius: 16px;
        cursor: pointer;
        transition: all 0.2s ease;
        font-weight: 700;
        color: #64748b;
        position: relative;
        font-size: 14px;
        letter-spacing: 0.3px;
    }
    
    .view-btn:hover {
        border-color: #ec4899;
        color: #ec4899;
        background: rgba(255, 255, 255, 1);
    }
    
    .view-btn.active {
        background: linear-gradient(135deg, #ec4899 0%, #be185d 100%);
        color: white;
        border-color: transparent;
        box-shadow: 0 4px 12px rgba(236, 72, 153, 0.3);
    }
    
    .view-btn.active:hover {
        box-shadow: 0 6px 16px rgba(236, 72, 153, 0.4);
    }
    
    .personnel-calendars-container {
        position: relative;
        margin-bottom: 24px;
    }
    
    .personnel-calendars-nav {
        position: absolute;
        top: 50%;
        transform: translateY(-50%);
        z-index: 10;
        background: linear-gradient(135deg, #ec4899 0%, #be185d 100%);
        border: none;
        border-radius: 50%;
        width: 48px;
        height: 48px;
        display: flex;
        align-items: center;
        justify-content: center;
        color: white;
        font-size: 20px;
        cursor: pointer;
        transition: all 0.2s ease;
        box-shadow: 0 4px 15px rgba(236, 72, 153, 0.4);
    }
    
    .personnel-calendars-nav:hover {
        box-shadow: 0 6px 20px rgba(236, 72, 153, 0.5);
    }
    
    .personnel-calendars-nav.prev {
        left: -24px;
    }
    
    .personnel-calendars-nav.next {
        right: -24px;
    }
    
    .personnel-calendars-nav:disabled {
        opacity: 0.3;
        cursor: not-allowed;
        transform: translateY(-50%);
    }
    
    .personnel-calendars-nav:disabled:hover {
        transform: translateY(-50%) scale(1);
    }
    
    .personnel-calendars-wrapper {
        display: flex;
        gap: 28px;
        overflow-x: auto;
        overflow-y: hidden;
        scroll-behavior: smooth;
        padding: 8px 0;
        scrollbar-width: thin;
        scrollbar-color: rgba(236, 72, 153, 0.3) transparent;
        animation: fadeInUp 0.8s ease-out 0.2s both;
    }
    
    .personnel-calendars-wrapper::-webkit-scrollbar {
        height: 8px;
    }
    
    .personnel-calendars-wrapper::-webkit-scrollbar-track {
        background: rgba(236, 72, 153, 0.1);
        border-radius: 10px;
    }
    
    .personnel-calendars-wrapper::-webkit-scrollbar-thumb {
        background: linear-gradient(135deg, #ec4899 0%, #be185d 100%);
        border-radius: 10px;
    }
    
    .personnel-calendars-wrapper::-webkit-scrollbar-thumb:hover {
        background: linear-gradient(135deg, #be185d 0%, #9f1239 100%);
    }
    
    .personnel-calendar-card {
        background: rgba(255, 255, 255, 0.9);
        backdrop-filter: blur(30px) saturate(200%);
        -webkit-backdrop-filter: blur(30px) saturate(200%);
        border-radius: 24px;
        padding: 28px;
        box-shadow: 
            0 20px 60px rgba(236, 72, 153, 0.12),
            0 0 0 1px rgba(255, 255, 255, 0.8),
            inset 0 1px 0 rgba(255, 255, 255, 0.9);
        border: 1px solid rgba(255, 255, 255, 0.5);
        min-height: 650px;
        flex-shrink: 0;
        transition: all 0.2s ease;
        position: relative;
        overflow: hidden;
    }
    
    /* 1 personel için %100 genişlik */
    .personnel-calendars-wrapper[data-personel-count="1"] .personnel-calendar-card {
        width: 100%;
        min-width: 100%;
        max-width: 100%;
    }
    
    /* 2 personel için %50 genişlik */
    .personnel-calendars-wrapper[data-personel-count="2"] .personnel-calendar-card {
        width: calc(50% - 14px);
        min-width: calc(50% - 14px);
        max-width: calc(50% - 14px);
    }
    
    /* 3+ personel için sabit genişlik */
    .personnel-calendars-wrapper[data-personel-count="3+"] .personnel-calendar-card {
        width: 450px;
        min-width: 450px;
        max-width: 450px;
    }
    
    .personnel-calendar-card:hover {
        box-shadow: 
            0 24px 70px rgba(236, 72, 153, 0.15),
            0 0 0 1px rgba(255, 255, 255, 1);
    }
    
    .personnel-header {
        display: flex;
        align-items: center;
        gap: 16px;
        margin-bottom: 24px;
        padding-bottom: 20px;
        border-bottom: 2px solid;
        border-image: linear-gradient(90deg, rgba(236, 72, 153, 0.2), transparent) 1;
        position: relative;
    }
    
    .personnel-header::after {
        content: '';
        position: absolute;
        bottom: -2px;
        left: 0;
        width: 60px;
        height: 2px;
        background: linear-gradient(90deg, #ec4899, #be185d);
        border-radius: 2px;
    }
    
    .personnel-name {
        font-weight: 800;
        font-size: 18px;
        background: linear-gradient(135deg, #ec4899 0%, #be185d 50%, #ec4899 100%);
        background-size: 200% 200%;
        -webkit-background-clip: text;
        -webkit-text-fill-color: transparent;
        background-clip: text;
        letter-spacing: -0.3px;
        animation: gradientShift 3s ease infinite;
    }
    
    #calendar-container {
        display: block !important;
    }
    
    /* FullCalendar modern özelleştirmeleri - Excel görünümünü kaldır */
    .personnel-calendar-card .fc {
        font-family: 'Inter', sans-serif;
        border: none !important;
    }
    
    /* Tüm border'ları kaldır veya yumuşat */
    .personnel-calendar-card .fc-scrollgrid,
    .personnel-calendar-card .fc-scrollgrid-section,
    .personnel-calendar-card .fc-col-header-cell,
    .personnel-calendar-card .fc-daygrid-day,
    .personnel-calendar-card .fc-timegrid-col,
    .personnel-calendar-card .fc-timegrid-slot,
    .personnel-calendar-card .fc-timegrid-col-frame {
        border: none !important;
        border-color: transparent !important;
    }
    
    /* Sadece ince ayırıcı çizgiler */
    .personnel-calendar-card .fc-timegrid-slot {
        border-top: 1px solid rgba(236, 72, 153, 0.08) !important;
    }
    
    .personnel-calendar-card .fc-timegrid-slot-minor {
        border-top: 1px solid rgba(236, 72, 153, 0.04) !important;
    }
    
    .personnel-calendar-card .fc-col-header-cell {
        border-bottom: 2px solid rgba(236, 72, 153, 0.15) !important;
        border-right: none !important;
        border-left: none !important;
        border-top: none !important;
    }
    
    /* Header toolbar */
    .personnel-calendar-card .fc-header-toolbar {
        background: linear-gradient(135deg, rgba(236, 72, 153, 0.08) 0%, rgba(190, 24, 93, 0.08) 100%);
        backdrop-filter: blur(10px);
        padding: 12px;
        border-radius: 16px;
        margin-bottom: 20px;
        border: 1px solid rgba(236, 72, 153, 0.1);
        display: flex;
        flex-wrap: nowrap;
        align-items: center;
        gap: 8px;
    }
    
    .personnel-calendar-card .fc-header-toolbar .fc-toolbar-chunk {
        display: flex;
        align-items: center;
        flex-shrink: 0;
    }
    
    .personnel-calendar-card .fc-header-toolbar .fc-toolbar-chunk.fc-toolbar-chunk-center {
        flex: 1;
        min-width: 0;
        justify-content: center;
    }
    
    .personnel-calendar-card .fc-header-toolbar .fc-button {
        flex-shrink: 0;
        padding: 8px 12px;
        font-size: 11px;
    }
    
    .personnel-calendar-card .fc-button {
        background: linear-gradient(135deg, #ec4899 0%, #be185d 100%);
        border: none;
        border-radius: 12px;
        padding: 10px 20px;
        font-weight: 700;
        transition: all 0.2s ease;
        box-shadow: 0 4px 15px rgba(236, 72, 153, 0.3);
    }
    
    .personnel-calendar-card .fc-button:hover {
        box-shadow: 0 6px 20px rgba(236, 72, 153, 0.4);
    }
    
    .personnel-calendar-card .fc-button:disabled {
        opacity: 0.4;
        cursor: not-allowed;
        transform: none;
    }
    
    .personnel-calendar-card .fc-toolbar-title {
        font-weight: 500;
        color: #1e293b;
        font-size: 4px;
        letter-spacing: -0.1px;
        white-space: nowrap;
        overflow: hidden;
        text-overflow: ellipsis;
        max-width: 100%;
    }
    
    /* Hücre arka planları - yumuşak ve modern */
    .personnel-calendar-card .fc-daygrid-day {
        background: transparent !important;
    }
    
    .personnel-calendar-card .fc-daygrid-day.fc-day-today {
        background: rgba(236, 72, 153, 0.05) !important;
    }
    
    .personnel-calendar-card .fc-timegrid-slot {
        background: transparent !important;
        transition: background 0.2s ease;
    }
    
    .personnel-calendar-card .fc-timegrid-slot:hover {
        background: rgba(236, 72, 153, 0.06) !important;
        cursor: pointer;
    }
    
    .personnel-calendar-card .fc-timegrid-slot.fc-timegrid-slot-label {
        background: rgba(255, 255, 255, 0.5) !important;
    }
    
    /* Header hücreleri */
    .personnel-calendar-card .fc-col-header-cell {
        background: rgba(255, 255, 255, 0.6) !important;
        padding: 12px 8px;
        font-weight: 700;
        color: #1e293b;
        text-transform: capitalize;
        font-size: 14px;
    }
    
    .personnel-calendar-card .fc-col-header-cell:hover {
        background: rgba(236, 72, 153, 0.08) !important;
    }
    
    /* Event stilleri */
    .personnel-calendar-card .fc-event {
        border-radius: 12px;
        border: none !important;
        padding: 8px 12px;
        cursor: pointer;
        transition: all 0.2s ease;
        font-weight: 600;
        box-shadow: 0 4px 12px rgba(0,0,0,0.15);
        margin: 2px 4px;
    }
    
    .personnel-calendar-card .fc-event:hover {
        box-shadow: 0 6px 18px rgba(0,0,0,0.2);
        z-index: 10;
    }
    
    /* Time slot labels */
    .personnel-calendar-card .fc-timegrid-slot-label {
        color: #64748b;
        font-weight: 600;
        font-size: 13px;
    }
    
    .personnel-calendar-card .fc-timegrid-slot-label-cushion {
        font-size: 13px;
        font-weight: 600;
    }
    
    /* Scrollbar gizle veya özelleştir */
    .personnel-calendar-card .fc-scroller {
        scrollbar-width: thin;
        scrollbar-color: rgba(236, 72, 153, 0.3) transparent;
    }
    
    .personnel-calendar-card .fc-scroller::-webkit-scrollbar {
        width: 6px;
        height: 6px;
    }
    
    .personnel-calendar-card .fc-scroller::-webkit-scrollbar-track {
        background: transparent;
    }
    
    .personnel-calendar-card .fc-scroller::-webkit-scrollbar-thumb {
        background: rgba(236, 72, 153, 0.3);
        border-radius: 10px;
    }
    
    .personnel-calendar-card .fc-scroller::-webkit-scrollbar-thumb:hover {
        background: rgba(236, 72, 153, 0.5);
    }
    
    /* Day grid görünümü için (Aylık) */
    .personnel-calendar-card .fc-daygrid-day-frame {
        border: none !important;
    }
    
    .personnel-calendar-card .fc-daygrid-day {
        border: 1px solid rgba(236, 72, 153, 0.08) !important;
        transition: all 0.2s ease;
    }
    
    .personnel-calendar-card .fc-daygrid-day:hover {
        background: rgba(236, 72, 153, 0.05) !important;
    }
    
    .personnel-calendar-card .fc-day-today {
        background: rgba(236, 72, 153, 0.08) !important;
    }
    
    .personnel-calendar-card .fc-daygrid-day-top {
        padding: 10px 8px;
    }
    
    .personnel-calendar-card .fc-daygrid-day-number {
        font-weight: 700;
        color: #1e293b;
        padding: 6px 10px;
        border-radius: 10px;
        transition: all 0.2s ease;
        display: inline-block;
        min-width: 32px;
        text-align: center;
        font-size: 12px;
    }
    
    .personnel-calendar-card .fc-day-today .fc-daygrid-day-number {
        background: linear-gradient(135deg, #ec4899 0%, #be185d 100%);
        color: white;
        box-shadow: 0 2px 8px rgba(236, 72, 153, 0.3);
    }
    
    .personnel-calendar-card .fc-daygrid-day:hover .fc-daygrid-day-number {
        background: rgba(236, 72, 153, 0.1);
    }
    
    .personnel-calendar-card .fc-daygrid-event {
        border-radius: 8px;
        margin: 2px 4px;
        padding: 4px 8px;
        font-size: 12px;
        font-weight: 600;
        border: none !important;
    }
    
    .personnel-calendar-card .fc-daygrid-event:hover {
        box-shadow: 0 2px 8px rgba(0,0,0,0.15);
    }
    
    .personnel-calendar-card .fc-daygrid-more-link {
        color: #ec4899;
        font-weight: 600;
        padding: 4px 8px;
        border-radius: 6px;
        transition: all 0.2s ease;
    }
    
    .personnel-calendar-card .fc-daygrid-more-link:hover {
        background: rgba(236, 72, 153, 0.1);
    }
    
    /* Haftalık görünüm için */
    .personnel-calendar-card .fc-timeGridWeek-view .fc-col-header-cell {
        background: rgba(255, 255, 255, 0.7) !important;
        font-weight: 700;
        padding: 14px 8px;
        font-size: 14px;
    }
    
    .personnel-calendar-card .fc-timeGridWeek-view .fc-day-today {
        background: rgba(236, 72, 153, 0.05) !important;
    }
    
    .personnel-calendar-card .fc-timeGridWeek-view .fc-timegrid-col {
        border-right: 1px solid rgba(236, 72, 153, 0.08) !important;
    }
    
    .personnel-calendar-card .fc-timeGridWeek-view .fc-timegrid-col:hover {
        background: rgba(236, 72, 153, 0.03) !important;
    }
    
    .personnel-calendar-card .fc-timeGridWeek-view .fc-timegrid-slot {
        border-top: 1px solid rgba(236, 72, 153, 0.06) !important;
    }
    
    .personnel-calendar-card .fc-timeGridWeek-view .fc-timegrid-slot-minor {
        border-top: 1px solid rgba(236, 72, 153, 0.03) !important;
    }
    
    /* Aylık görünüm için özel stiller */
    .personnel-calendar-card .fc-dayGridMonth-view .fc-col-header-cell {
        background: rgba(255, 255, 255, 0.8) !important;
        font-weight: 700;
        padding: 12px 8px;
        text-transform: capitalize;
        color: #1e293b;
        font-size: 14px;
    }
    
    .personnel-calendar-card .fc-dayGridMonth-view .fc-daygrid-day {
        min-height: 100px;
    }
    
    .personnel-calendar-card .fc-dayGridMonth-view .fc-daygrid-day-frame {
        padding: 4px;
    }
    
    .personnel-calendar-card .fc-dayGridMonth-view .fc-daygrid-day-events {
        margin-top: 4px;
    }
    
    /* Haftalık ve aylık görünümde event stilleri */
    .personnel-calendar-card .fc-timeGridWeek-view .fc-event,
    .personnel-calendar-card .fc-dayGridMonth-view .fc-event {
        border-radius: 8px;
        border: none !important;
        padding: 6px 10px;
        font-weight: 600;
        font-size: 12px;
        box-shadow: 0 2px 6px rgba(0,0,0,0.12);
        margin: 2px 0;
    }
    
    .personnel-calendar-card .fc-timeGridWeek-view .fc-event:hover,
    .personnel-calendar-card .fc-dayGridMonth-view .fc-event:hover {
        box-shadow: 0 4px 12px rgba(0,0,0,0.2);
    }
    
    /* Haftalık görünümde saat etiketleri */
    .personnel-calendar-card .fc-timeGridWeek-view .fc-timegrid-slot-label {
        color: #64748b;
        font-weight: 600;
        font-size: 13px;
    }
    
    .personnel-calendar-card .fc-timeGridWeek-view .fc-timegrid-slot-label-cushion {
        font-size: 13px;
    }
    
    /* Günlük görünümde saat etiketleri */
    .personnel-calendar-card .fc-timeGridDay-view .fc-timegrid-slot-label {
        color: #64748b;
        font-weight: 600;
        font-size: 13px;
    }
    
    .personnel-calendar-card .fc-timeGridDay-view .fc-timegrid-slot-label-cushion {
        font-size: 13px;
    }
    
    /* Aylık görünümde daha kompakt */
    .personnel-calendar-card .fc-dayGridMonth-view {
        font-size: 12px;
    }
    
    .personnel-calendar-card .fc-dayGridMonth-view .fc-daygrid-day-number {
        font-size: 12px;
    }
    
    /* Loading animasyonu */
    .spinner-border {
        animation: pulse 1.5s ease-in-out infinite;
    }
    
    @media (max-width: 768px) {
        .calendar-page {
            padding: 16px;
        }
        
        .calendar-header {
            flex-direction: column;
            align-items: stretch;
            padding: 20px;
            border-radius: 20px;
            margin-bottom: 20px;
        }
        
        .calendar-header h2 {
            font-size: 22px;
            margin-bottom: 16px;
        }
        
        .view-buttons {
            width: 100%;
            justify-content: stretch;
            gap: 8px;
        }
        
        .view-btn {
            flex: 1;
            padding: 10px 12px;
            font-size: 12px;
        }
        
        .personnel-calendars-container {
            margin: 0 -16px;
            padding: 0 16px;
        }
        
        .personnel-calendars-wrapper {
            gap: 16px;
            padding: 8px 0;
        }
        
        .personnel-calendar-card {
            width: 100% !important;
            min-width: 100% !important;
            max-width: 100% !important;
            padding: 16px;
            border-radius: 16px;
            min-height: 500px;
        }
        
        .personnel-header {
            margin-bottom: 16px;
            padding-bottom: 12px;
        }
        
        .personnel-name {
            font-size: 16px;
        }
        
        .personnel-calendars-nav {
            width: 36px;
            height: 36px;
            font-size: 14px;
        }
        
        .personnel-calendars-nav.prev {
            left: 8px;
        }
        
        .personnel-calendars-nav.next {
            right: 8px;
        }
        
        /* Mobile FullCalendar özelleştirmeleri */
        .personnel-calendar-card .fc-header-toolbar {
            padding: 10px;
            margin-bottom: 12px;
            flex-wrap: nowrap;
            gap: 6px;
        }
        
        .personnel-calendar-card .fc-header-toolbar .fc-toolbar-chunk {
            flex-shrink: 0;
        }
        
        .personnel-calendar-card .fc-header-toolbar .fc-toolbar-chunk-center {
            flex: 1;
            min-width: 0;
        }
        
        .personnel-calendar-card .fc-toolbar-title {
            font-size: 4px;
            white-space: nowrap;
            overflow: hidden;
            text-overflow: ellipsis;
            max-width: 100%;
        }
        
        .personnel-calendar-card .fc-button {
            padding: 6px 10px;
            font-size: 10px;
        }
        
        .personnel-calendar-card .fc-col-header-cell {
            padding: 8px 4px;
            font-size: 12px;
        }
        
        .personnel-calendar-card .fc-timegrid-slot-label {
            font-size: 12px;
            padding: 4px;
        }
        
        .personnel-calendar-card .fc-timegrid-slot-label-cushion {
            font-size: 12px;
        }
        
        .personnel-calendar-card .fc-event {
            padding: 4px 8px;
            font-size: 11px;
        }
        
        .personnel-calendar-card .fc-daygrid-day-number {
            font-size: 11px;
            padding: 4px 6px;
        }
        
        .personnel-calendar-card .fc-dayGridMonth-view .fc-daygrid-day {
            min-height: 80px;
        }
        
        .personnel-calendar-card .fc-dayGridMonth-view {
            font-size: 11px;
        }
        
        .personnel-calendar-card .fc-dayGridMonth-view .fc-daygrid-day-number {
            font-size: 12px;
        }
        
        .personnel-calendar-card .fc-dayGridMonth-view .fc-event {
            font-size: 10px;
            padding: 2px 6px;
            margin: 1px 0;
        }
    }
    
    @media (max-width: 480px) {
        .calendar-page {
            padding: 12px;
        }
        
        .calendar-header {
            padding: 16px;
        }
        
        .calendar-header h2 {
            font-size: 20px;
        }
        
        .view-btn {
            padding: 8px 10px;
            font-size: 11px;
        }
        
        .personnel-calendar-card {
            padding: 12px;
            min-height: 450px;
        }
        
        .personnel-name {
            font-size: 14px;
        }
        
        .personnel-calendar-card .fc-toolbar-title {
            font-size: 3px;
            white-space: nowrap;
            overflow: hidden;
            text-overflow: ellipsis;
        }
        
        .personnel-calendar-card .fc-header-toolbar {
            padding: 8px;
            gap: 4px;
        }
        
        .personnel-calendar-card .fc-header-toolbar .fc-button {
            padding: 5px 8px;
            font-size: 9px;
        }
        
        .personnel-calendar-card .fc-button {
            padding: 6px 10px;
            font-size: 11px;
        }
        
        .personnel-calendar-card .fc-col-header-cell {
            padding: 6px 2px;
            font-size: 11px;
        }
        
        .personnel-calendar-card .fc-timegrid-slot-label {
            font-size: 11px;
        }
        
        .personnel-calendar-card .fc-timegrid-slot-label-cushion {
            font-size: 11px;
        }
        
        .personnel-calendar-card .fc-dayGridMonth-view .fc-daygrid-day {
            min-height: 70px;
        }
    }
</style>

<div class="calendar-page">
    <div class="calendar-header">
        <h2 class="mb-0">Randevu Takvimi</h2>
        <div class="view-buttons">
            <button class="view-btn" id="viewDay" onclick="changeView('day')">
                <i class="bi bi-calendar-day me-1"></i> Gün
            </button>
            <button class="view-btn" id="viewWeek" onclick="changeView('week')">
                <i class="bi bi-calendar-week me-1"></i> Hafta
            </button>
            <button class="view-btn" id="viewMonth" onclick="changeView('month')">
                <i class="bi bi-calendar-month me-1"></i> Ay
            </button>
        </div>
    </div>
    
    <div id="calendar-container">
        <div class="personnel-calendars-container">
            <button class="personnel-calendars-nav prev" id="calendarsPrev" onclick="scrollCalendars('prev')" style="display: none;">
                <i class="bi bi-chevron-left"></i>
            </button>
            <div class="personnel-calendars-wrapper" id="calendarsWrapper">
                <div class="text-center py-5" style="min-width: 100%;">
                    <div class="spinner-border text-primary" role="status"></div>
                    <p class="mt-3">Takvimler yükleniyor...</p>
                </div>
            </div>
            <button class="personnel-calendars-nav next" id="calendarsNext" onclick="scrollCalendars('next')" style="display: none;">
                <i class="bi bi-chevron-right"></i>
            </button>
        </div>
    </div>
</div>

<!-- Randevu Detay Modal -->
<div class="modal fade" id="reservationDetailModal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog modal-xl modal-dialog-scrollable">
        <div class="modal-content border-0 shadow-lg" style="border-radius: 16px;">
            <div class="modal-header" style="background: linear-gradient(135deg, #ec4899 0%, #be185d 100%);">
                <h5 class="modal-title text-white fw-bold">
                    <i class="bi bi-calendar-check me-2"></i>Randevu Detayı
                </h5>
                <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body" id="reservationDetailContent">
                <div class="text-center py-5">
                    <div class="spinner-border text-primary" role="status"></div>
                    <p class="mt-3">Yükleniyor...</p>
                </div>
            </div>
        </div>
    </div>
</div>

@push('scripts')
<link href="https://cdn.jsdelivr.net/npm/fullcalendar@5.11.5/main.min.css" rel="stylesheet">
<script src="https://cdn.jsdelivr.net/npm/fullcalendar@5.11.5/main.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/fullcalendar@5.11.5/locales/tr.js"></script>

<script>
let currentView = 'day';
let personnelCalendars = [];
let personnelData = [];

function showReservationDetail(reservationId) {
    const modal = document.getElementById('reservationDetailModal');
    const content = document.getElementById('reservationDetailContent');
    if (!modal || !content) return;

    content.innerHTML = '<div class="text-center py-5"><div class="spinner-border text-primary" role="status"></div><p class="mt-3">Yükleniyor...</p></div>';
    const bsModal = new bootstrap.Modal(modal);
    bsModal.show();

    const showUrl = '{{ route("admin.appointments.show", ["id" => ":id"]) }}'.replace(':id', reservationId);
    fetch(showUrl, {
        headers: {
            'X-Requested-With': 'XMLHttpRequest',
            'Accept': 'application/json'
        }
    })
    .then(async response => {
        if (!response.ok) throw new Error('HTTP ' + response.status);
        return response.json();
    })
    .then(data => {
        const reservation = data.reservation || data;
        const customer = data.customer;
        const service = reservation.appointment_request?.service || (data.service ? { name: data.service.name, price: data.service.price } : null);

        let html = '<div class="row g-3">';
        html += '<div class="col-md-6">';
        html += '<div class="card shadow-sm mb-3" style="border-radius: 12px; border: none;">';
        html += '<div class="card-header" style="background: linear-gradient(135deg, #ec4899 0%, #be185d 100%); border-radius: 12px 12px 0 0; padding: 12px 16px;">';
        html += '<h6 class="mb-0 text-white fw-bold"><i class="bi bi-person-circle me-2"></i>Müşteri Bilgileri</h6></div><div class="card-body p-3">';
        if (customer) {
            html += '<p class="mb-2"><strong>Ad Soyad:</strong> ' + (customer.name || '-') + ' ' + (customer.surname || '') + '</p>';
            html += '<p class="mb-2"><strong>Telefon:</strong> ' + (customer.phone || '-') + '</p>';
            if (customer.email) html += '<p class="mb-2"><strong>E-posta:</strong> ' + customer.email + '</p>';
        } else {
            html += '<p class="text-muted">Müşteri bilgisi bulunamadı.</p>';
        }
        html += '</div></div>';

        html += '<div class="card shadow-sm" style="border-radius: 12px; border: none;">';
        html += '<div class="card-header" style="background: linear-gradient(135deg, #f472b6 0%, #ec4899 100%); border-radius: 12px 12px 0 0; padding: 12px 16px;">';
        html += '<h6 class="mb-0 text-white fw-bold"><i class="bi bi-calendar3 me-2"></i>Randevu Bilgileri</h6></div><div class="card-body p-3">';
        html += '<p class="mb-2"><strong>Tarih:</strong> ' + (reservation.date ? new Date(reservation.date).toLocaleDateString('tr-TR') : '-') + '</p>';
        html += '<p class="mb-2"><strong>Saat:</strong> ' + (reservation.start_time || '-') + '</p>';
        if (service) html += '<p class="mb-2"><strong>Hizmet:</strong> ' + service.name + '</p>';
        if (reservation.personel) html += '<p class="mb-2"><strong>Personel:</strong> ' + reservation.personel.name + '</p>';
        if (reservation.room) html += '<p class="mb-2"><strong>Oda:</strong> ' + reservation.room.name + '</p>';
        html += '</div></div></div>';

        html += '<div class="col-md-6">';
        html += '<div class="card shadow-sm mb-3" style="border-radius: 12px; border: none;">';
        html += '<div class="card-header" style="background: linear-gradient(135deg, #ec4899 0%, #be185d 100%); border-radius: 12px 12px 0 0; padding: 12px 16px;">';
        html += '<h6 class="mb-0 text-white fw-bold"><i class="bi bi-cash-coin me-2"></i>Ödeme Bilgileri</h6></div><div class="card-body p-3">';
        html += '<p class="mb-2"><strong>Toplam Tutar:</strong> ' + parseFloat(reservation.total_price || 0).toLocaleString('tr-TR', { minimumFractionDigits: 2, maximumFractionDigits: 2 }) + ' ₺</p>';
        html += '<p class="mb-2"><strong>Ödenen:</strong> ' + parseFloat(reservation.deposit_paid || 0).toLocaleString('tr-TR', { minimumFractionDigits: 2, maximumFractionDigits: 2 }) + ' ₺</p>';
        html += '<p class="mb-0"><strong>Kalan:</strong> ' + parseFloat(reservation.remaining_amount || 0).toLocaleString('tr-TR', { minimumFractionDigits: 2, maximumFractionDigits: 2 }) + ' ₺</p>';
        html += '</div></div>';

        html += '<div class="card shadow-sm" style="border-radius: 12px; border: none;">';
        html += '<div class="card-header" style="background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%); border-radius: 12px 12px 0 0; padding: 12px 16px;">';
        html += '<h6 class="mb-0 text-white fw-bold"><i class="bi bi-gear me-2"></i>İşlemler</h6></div><div class="card-body p-3"><div class="d-grid gap-2">';
        if (reservation.status === 'pending' || reservation.status === 'open') {
            html += '<button type="button" class="btn btn-success fw-bold" onclick="confirmReservation(' + reservation.id + ')"><i class="bi bi-check-circle me-2"></i>Onayla</button>';
        }
        if (reservation.status === 'confirmed') {
            html += '<button type="button" class="btn btn-primary fw-bold" onclick="completeReservation(' + reservation.id + ')"><i class="bi bi-check2-all me-2"></i>Bitir ve Kasaya Ekle</button>';
        }
        html += '<button type="button" class="btn btn-secondary fw-bold" data-bs-dismiss="modal"><i class="bi bi-x-circle me-2"></i>Kapat</button>';
        html += '</div></div></div></div></div>';

        content.innerHTML = html;
    })
    .catch(error => {
        content.innerHTML = '<div class="alert alert-danger"><i class="bi bi-exclamation-triangle me-2"></i>Randevu detayı yüklenirken bir hata oluştu: ' + error.message + '</div>';
    });
}

function confirmReservation(reservationId) {
    if (!confirm('Bu randevuyu onaylamak istediğinize emin misiniz?')) return;
    const url = '{{ route("admin.appointments.confirm", ["id" => ":id"]) }}'.replace(':id', reservationId);
    fetch(url, {
        method: 'POST',
        headers: {
            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.content || '',
            'X-Requested-With': 'XMLHttpRequest',
            'Accept': 'application/json'
        }
    }).then(async response => {
        const data = await response.json().catch(() => ({}));
        if (!response.ok || !data.success) throw new Error(data.message || 'Onaylama başarısız');
        const modal = bootstrap.Modal.getInstance(document.getElementById('reservationDetailModal'));
        if (modal) modal.hide();
        setTimeout(() => location.reload(), 400);
    }).catch(error => alert(error.message || 'Randevu onaylanamadı.'));
}

function completeReservation(reservationId) {
    if (!confirm('Bu randevuyu tamamlayıp kasaya eklemek istiyor musunuz?')) return;
    const url = '{{ route("admin.appointments.complete", ["id" => ":id"]) }}'.replace(':id', reservationId);
    fetch(url, {
        method: 'POST',
        headers: {
            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.content || '',
            'X-Requested-With': 'XMLHttpRequest',
            'Accept': 'application/json'
        }
    }).then(async response => {
        const data = await response.json().catch(() => ({}));
        if (!response.ok || !data.success) throw new Error(data.message || 'Tamamlama başarısız');
        const modal = bootstrap.Modal.getInstance(document.getElementById('reservationDetailModal'));
        if (modal) modal.hide();
        setTimeout(() => location.reload(), 400);
    }).catch(error => alert(error.message || 'Randevu tamamlanamadı.'));
}

// Personel verilerini yükle
async function loadPersonnelData() {
    try {
        const response = await fetch('{{ route("admin.beauty-salon.appointment-requests.data") }}', {
            headers: { 'X-Requested-With': 'XMLHttpRequest', 'Accept': 'application/json' }
        });
        
        if (!response.ok) {
            throw new Error('API hatası');
        }
        
        const data = await response.json();
        personnelData = data.personels || [];
        return personnelData;
    } catch (error) {
        console.error('Personel verileri yüklenemedi:', error);
        return [];
    }
}

// Takvimleri kaydır
function scrollCalendars(direction) {
    const wrapper = document.getElementById('calendarsWrapper');
    if (!wrapper) return;
    
    const scrollAmount = 480; // Kart genişliği + gap
    const currentScroll = wrapper.scrollLeft;
    
    if (direction === 'prev') {
        wrapper.scrollTo({
            left: currentScroll - scrollAmount,
            behavior: 'smooth'
        });
    } else {
        wrapper.scrollTo({
            left: currentScroll + scrollAmount,
            behavior: 'smooth'
        });
    }
    
    updateNavButtons();
}

// Navigasyon butonlarını güncelle
function updateNavButtons() {
    const wrapper = document.getElementById('calendarsWrapper');
    const prevBtn = document.getElementById('calendarsPrev');
    const nextBtn = document.getElementById('calendarsNext');
    
    if (!wrapper || !prevBtn || !nextBtn) return;
    
    const isAtStart = wrapper.scrollLeft <= 10;
    const isAtEnd = wrapper.scrollLeft >= wrapper.scrollWidth - wrapper.clientWidth - 10;
    
    prevBtn.disabled = isAtStart;
    nextBtn.disabled = isAtEnd;
    
    prevBtn.style.display = isAtStart && isAtEnd ? 'none' : 'flex';
    nextBtn.style.display = isAtStart && isAtEnd ? 'none' : 'flex';
}

// Takvimleri oluştur
function createCalendars(personels) {
    const wrapper = document.getElementById('calendarsWrapper');
    if (!wrapper) return;
    
    wrapper.innerHTML = '';
    
    if (personels.length === 0) {
        const personelIndexUrl = '{{ route("admin.beauty-salon.personel.index") }}';
        wrapper.innerHTML = `
            <div style="min-width: 100%; text-align: center; padding: 60px 20px;">
                <div style="max-width: 500px; margin: 0 auto;">
                    <i class="bi bi-calendar-x" style="font-size: 64px; color: #ec4899; margin-bottom: 20px; opacity: 0.5;"></i>
                    <h3 style="color: #1e293b; margin-bottom: 12px; font-weight: 700;">Henüz Personel Eklenmemiş</h3>
                    <p style="color: #64748b; margin-bottom: 24px;">Takvimi görüntülemek için önce personel eklemeniz gerekmektedir.</p>
                    <a href="${personelIndexUrl}" class="btn btn-primary" style="background: linear-gradient(135deg, #ec4899 0%, #be185d 100%); border: none; padding: 12px 24px; border-radius: 12px; font-weight: 600; text-decoration: none; display: inline-block;">
                        <i class="bi bi-person-plus me-2"></i>Personel Ekle
                    </a>
                </div>
            </div>
        `;
        return;
    }
    
    // Personel sayısına göre data attribute ekle
    if (personels.length === 1) {
        wrapper.setAttribute('data-personel-count', '1');
    } else if (personels.length === 2) {
        wrapper.setAttribute('data-personel-count', '2');
    } else {
        wrapper.setAttribute('data-personel-count', '3+');
    }
    
    personnelCalendars = [];
    
    personels.forEach((personel, index) => {
        const card = document.createElement('div');
        card.className = 'personnel-calendar-card';
        card.id = `calendar-card-${personel.id}`;
        
        // Header
        const header = document.createElement('div');
        header.className = 'personnel-header';
        header.innerHTML = `
            <div class="personnel-name">${personel.name}</div>
        `;
        card.appendChild(header);
        
        // Calendar container
        const calendarEl = document.createElement('div');
        calendarEl.id = `calendar-${personel.id}`;
        calendarEl.style.minHeight = '500px';
        card.appendChild(calendarEl);
        
        wrapper.appendChild(card);
        
        // FullCalendar oluştur
        const calendar = new FullCalendar.Calendar(calendarEl, {
            initialView: currentView === 'day' ? 'timeGridDay' : currentView === 'week' ? 'timeGridWeek' : 'dayGridMonth',
            locale: 'tr',
            headerToolbar: {
                left: 'prev,next today',
                center: 'title',
                right: ''
            },
            height: 'auto',
            slotDuration: '00:30:00',
            slotMinTime: '08:00:00',
            slotMaxTime: '23:00:00',
            allDaySlot: false,
            events: function(fetchInfo, successCallback, failureCallback) {
                const start = fetchInfo.startStr;
                const end = fetchInfo.endStr;
                const feedRoute = '{{ route("admin.reservations.calendar.feed") }}';
                
                fetch(`${feedRoute}?start=${start}&end=${end}&personel_id=${personel.id}`, {
                    headers: {
                        'X-Requested-With': 'XMLHttpRequest',
                        'Accept': 'application/json',
                        'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.getAttribute('content') || ''
                    },
                    credentials: 'same-origin'
                })
                    .then(response => {
                        if (!response.ok) {
                            throw new Error('HTTP error! status: ' + response.status);
                        }
                        return response.json();
                    })
                    .then(data => {
                        const events = Array.isArray(data) ? data.map(event => ({
                            ...event,
                            backgroundColor: event.extendedProps?.status === 'leave' ? '#94a3b8' :
                                           event.extendedProps?.status === 'confirmed' ? '#ec4899' : 
                                           event.extendedProps?.status === 'completed' ? '#22c55e' : 
                                           event.extendedProps?.status === 'cancelled' ? '#ef4444' : '#f59e0b',
                            borderColor: event.extendedProps?.status === 'leave' ? '#64748b' :
                                        event.extendedProps?.status === 'confirmed' ? '#be185d' : 
                                        event.extendedProps?.status === 'completed' ? '#16a34a' : 
                                        event.extendedProps?.status === 'cancelled' ? '#dc2626' : '#d97706',
                            textColor: '#ffffff'
                        })) : [];
                        successCallback(events);
                    })
                    .catch(error => {
                        console.error('Takvim yükleme hatası:', error);
                        failureCallback(error);
                    });
            },
            eventClick: function(info) {
                info.jsEvent.preventDefault();
                if (info.event.extendedProps?.status === 'leave') {
                    return;
                }
                const reservationId = info.event.extendedProps?.reservation_id || info.event.id;
                if (reservationId) {
                    showReservationDetail(reservationId);
                }
            },
            dateClick: function(info) {
                // Tarih tıklama işlemi
                console.log('Tarih tıklandı:', info.dateStr);
            }
        });
        
        calendar.render();
        personnelCalendars.push({ calendar, personelId: personel.id });
    });
    
    // Personel sayısına göre navigasyon butonlarını ayarla
    const prevBtn = document.getElementById('calendarsPrev');
    const nextBtn = document.getElementById('calendarsNext');
    
    if (personels.length <= 2) {
        // 1-2 personel varsa navigasyon butonlarını gizle
        if (prevBtn) prevBtn.style.display = 'none';
        if (nextBtn) nextBtn.style.display = 'none';
    } else {
        // 3+ personel varsa navigasyon butonlarını göster
        if (prevBtn) prevBtn.style.display = 'flex';
        if (nextBtn) nextBtn.style.display = 'flex';
        // Scroll event listener ekle
        wrapper.addEventListener('scroll', updateNavButtons);
        // İlk yüklemede butonları kontrol et
        setTimeout(() => {
            updateNavButtons();
        }, 100);
    }
}

// Görünüm değiştir
function changeView(view) {
    currentView = view;
    
    // Buton stillerini güncelle
    document.querySelectorAll('.view-btn').forEach(btn => btn.classList.remove('active'));
    document.getElementById(`view${view.charAt(0).toUpperCase() + view.slice(1)}`).classList.add('active');
    
    // Takvim görünümlerini güncelle
    const viewMap = {
        'day': 'timeGridDay',
        'week': 'timeGridWeek',
        'month': 'dayGridMonth'
    };
    
    personnelCalendars.forEach(({ calendar }) => {
        calendar.changeView(viewMap[view]);
    });
}

// Sayfa yüklendiğinde başlat
document.addEventListener('DOMContentLoaded', async function() {
    // FullCalendar yüklenene kadar bekle
    if (typeof FullCalendar === 'undefined') {
        setTimeout(arguments.callee, 100);
        return;
    }
    
    const personels = await loadPersonnelData();
    createCalendars(personels);
    
    // Varsayılan görünümü ayarla
    document.getElementById('viewDay').classList.add('active');
});
</script>
@endpush

@endsection
