@extends('layouts.admin')

@section('title', 'Randevu Yönetimi')

@section('breadcrumb')
    <li class="breadcrumb-item"><a href="{{ route('admin.dashboard') }}">Anasayfa</a></li>
    <li class="breadcrumb-item active">Randevu Yönetimi</li>
@endsection

@section('content')
    <div style="padding-top: 1.5rem;">
    <div class="d-flex justify-content-between align-items-center mb-4 flex-wrap gap-3">
        <div>
            <h2 class="fw-bold mb-2" style="background: linear-gradient(135deg, #ec4899 0%, #be185d 100%); -webkit-background-clip: text; -webkit-text-fill-color: transparent; background-clip: text;">
                Randevu Yönetimi
            </h2>
            <small class="text-muted">
                <i class="bi bi-keyboard me-1"></i>Kısayollar: <kbd>Ctrl+N</kbd> Yeni Randevu, <kbd>Ctrl+F</kbd> Arama, <kbd>Ctrl+R</kbd> Yenile
            </small>
        </div>
        <div class="d-flex gap-2 flex-wrap">
            @if(auth()->user()->isBeautySalon())
                <button type="button" class="btn btn-warning" data-bs-toggle="modal" data-bs-target="#pendingRequestsModal">
                    <i class="bi bi-bell-fill me-2"></i>Randevu Talepleri
                    @php
                        $pendingCount = \App\Models\AppointmentRequest::where('beauty_salon_id', auth()->id())
                            ->where('status', 'pending')
                            ->count();
                    @endphp
                    @if($pendingCount > 0)
                        <span class="badge bg-danger ms-2">{{ $pendingCount }}</span>
                    @endif
                </button>
            @endif
            <button type="button" class="btn btn-success" onclick="quickCreateReservation()" title="Hızlı Randevu Oluştur (Ctrl+N)">
                <i class="bi bi-lightning-fill me-2"></i>Hızlı Randevu
            </button>
        </div>
    </div>
    
    <div style="margin-top: 2rem;"></div>
    
    <div class="card">
        <div class="card-body">
            <div class="table-responsive">
                <table class="table table-hover">
                    <thead>
                        <tr>
                            <th>Tarih</th>
                            <th>Saat</th>
                            <th>Konu</th>
                            <th>Müşteri</th>
                            <th>Durum</th>
                            <th>İşlemler</th>
                        </tr>
                    </thead>
                    <tbody>
                        @forelse($appointments as $appointment)
                            <tr>
                                <td>{{ $appointment->date->format('d.m.Y') }}</td>
                                <td>{{ $appointment->start_time }}</td>
                                <td>
                                    @if($appointment->appointmentRequest && $appointment->appointmentRequest->service)
                                        {{ $appointment->appointmentRequest->service->name }}
                                    @else
                                        <span class="text-muted">-</span>
                                    @endif
                                </td>
                                <td>{{ $appointment->customer->name ?? '-' }}</td>
                                <td>
                                    @if($appointment->status == 'pending' || $appointment->status == 'open')
                                        <span class="badge bg-warning">Beklemede</span>
                                    @elseif($appointment->status == 'approved' || $appointment->status == 'confirmed')
                                        <span class="badge bg-success">Onaylandı</span>
                                    @elseif($appointment->status == 'completed')
                                        <span class="badge bg-info">Tamamlandı</span>
                                    @else
                                        <span class="badge bg-danger">İptal</span>
                                    @endif
                                </td>
                                <td>
                                    <div class="d-flex gap-1 flex-wrap">
                                        <button type="button" class="btn btn-action btn-action-info" onclick="showReservationDetail({{ $appointment->id }})" title="Detayları Görüntüle">
                                            <i class="bi bi-eye"></i>
                                        </button>
                                        <a href="{{ route('admin.appointments.edit', $appointment->id) }}" class="btn btn-action btn-action-warning" title="Düzenle">
                                            <i class="bi bi-pencil"></i>
                                        </a>
                                        @if($appointment->status !== 'completed' && $appointment->status !== 'cancelled')
                                            <button type="button" class="btn btn-action btn-action-success" onclick="showQuickActionsModal({{ $appointment->id }}, {{ $appointment->total_price ?? 0 }}, {{ $appointment->deposit_paid ?? 0 }}, '{{ $appointment->status }}', '{{ $appointment->customer->phone ?? '' }}')" title="Hızlı İşlemler">
                                                <i class="bi bi-lightning-fill"></i>
                                            </button>
                                        @endif
                                        <form action="{{ route('admin.appointments.destroy', $appointment->id) }}" method="POST" class="d-inline delete-form" data-message="Bu randevuyu silmek istediğinize emin misiniz?">
                                            @csrf
                                            @method('DELETE')
                                            <button type="submit" class="btn btn-action btn-action-danger" title="Sil">
                                                <i class="bi bi-trash"></i>
                                            </button>
                                        </form>
                                    </div>
                                </td>
                            </tr>
                        @empty
                            <tr>
                                <td colspan="6" class="text-center">Randevu bulunamadı.</td>
                            </tr>
                        @endforelse
                    </tbody>
                </table>
            </div>
            
            <div class="mt-3">
                {{ $appointments->links() }}
            </div>
        </div>
    </div>

    <!-- Bekleyen Talepler Modal -->
    <div class="modal fade" id="pendingRequestsModal" tabindex="-1" aria-labelledby="pendingRequestsModalLabel" aria-hidden="true" role="dialog">
        <div class="modal-dialog modal-xl">
            <div class="modal-content" style="border-radius: 16px; border: none;">
                <div class="modal-header" style="background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%); border-radius: 16px 16px 0 0;">
                    <h5 class="modal-title text-white fw-bold" id="pendingRequestsModalLabel">
                        <i class="bi bi-bell-fill me-2"></i>Bekleyen Randevu Talepleri
                    </h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body" style="padding: 1.5rem;">
                    <!-- Desktop Table View -->
                    <div id="pendingRequestsList" class="table-responsive d-none d-md-block">
                        <table class="table table-hover">
                            <thead>
                                <tr>
                                    <th>Müşteri</th>
                                    <th>Telefon</th>
                                    <th>Hizmet</th>
                                    <th>Tarih</th>
                                    <th>İşlemler</th>
                                </tr>
                            </thead>
                            <tbody id="pendingRequestsTableBody">
                                <tr>
                                    <td colspan="5" class="text-center py-4">
                                        <div class="spinner-border text-warning" role="status">
                                            <span class="visually-hidden">Yükleniyor...</span>
                                        </div>
                                    </td>
                                </tr>
                            </tbody>
                        </table>
                    </div>
                    <!-- Mobile Card View -->
                    <div id="pendingRequestsMobileList" class="d-block d-md-none">
                        <div id="pendingRequestsMobileBody">
                            <div class="text-center py-4">
                                <div class="spinner-border text-warning" role="status">
                                    <span class="visually-hidden">Yükleniyor...</span>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Randevu Detay Modal -->
    <div class="modal fade" id="reservationDetailModal" tabindex="-1" aria-labelledby="reservationDetailModalLabel" aria-hidden="true" role="dialog">
        <div class="modal-dialog modal-xl" style="max-width: 90%;">
            <div class="modal-content" style="border-radius: 16px; border: none; max-height: 90vh; display: flex; flex-direction: column;">
                <div class="modal-header" style="background: linear-gradient(135deg, #ec4899 0%, #be185d 100%); border-radius: 16px 16px 0 0; flex-shrink: 0;">
                    <h5 class="modal-title text-white fw-bold" id="reservationDetailModalLabel">
                        <i class="bi bi-calendar-event me-2"></i>Randevu Detayı
                    </h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body" style="padding: 1.5rem; overflow-y: auto; flex: 1;" id="reservationDetailContent">
                    <div class="text-center py-5">
                        <div class="spinner-border text-primary" role="status"></div>
                        <p class="mt-3">Yükleniyor...</p>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Randevu Talebi Detay Modal -->
    <div class="modal fade" id="requestDetailModal" tabindex="-1" aria-labelledby="requestDetailModalLabel" aria-hidden="true" role="dialog">
        <div class="modal-dialog modal-xl" style="max-width: 90%;">
            <div class="modal-content" style="border-radius: 16px; border: none; max-height: 90vh; display: flex; flex-direction: column;">
                <div class="modal-header" style="background: linear-gradient(135deg, #ec4899 0%, #be185d 100%); border-radius: 16px 16px 0 0; flex-shrink: 0;">
                    <h5 class="modal-title text-white fw-bold" id="requestDetailModalLabel">
                        <i class="bi bi-eye me-2"></i>Randevu Talebi Detayı
                    </h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body" style="padding: 1.5rem; overflow-y: auto; flex: 1;" id="requestDetailContent">
                    <!-- İçerik JavaScript ile doldurulacak -->
                </div>
            </div>
        </div>
    </div>

    <!-- Hızlı İşlemler Modal -->
    <div class="modal fade" id="quickActionsModal" tabindex="-1" aria-labelledby="quickActionsModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-dialog-centered">
            <div class="modal-content" style="border-radius: 16px; border: none;">
                <div class="modal-header" style="background: linear-gradient(135deg, #10b981 0%, #059669 100%); border-radius: 16px 16px 0 0;">
                    <h5 class="modal-title text-white fw-bold" id="quickActionsModalLabel">
                        <i class="bi bi-lightning-fill me-2"></i>Hızlı İşlemler
                    </h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body p-4" id="quickActionsContent">
                    <!-- İçerik JavaScript ile doldurulacak -->
                </div>
            </div>
        </div>
    </div>

    <!-- Yeni Randevu Oluştur Modal -->
    <div class="modal fade" id="newReservationModal" tabindex="-1" aria-labelledby="newReservationModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-xl" style="max-width: 90%;">
            <div class="modal-content" style="border-radius: 16px; border: none; max-height: 90vh; display: flex; flex-direction: column;">
                <div class="modal-header" style="background: linear-gradient(135deg, #ec4899 0%, #be185d 100%); border-radius: 16px 16px 0 0; flex-shrink: 0;">
                    <h5 class="modal-title text-white fw-bold" id="newReservationModalLabel">
                        <i class="bi bi-calendar-plus me-2"></i>Yeni Randevu Oluştur
                    </h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body" style="padding: 1.5rem; overflow-y: auto; flex: 1;" id="newReservationContent">
                    <div id="newReservationLoading" class="text-center py-5">
                        <div class="spinner-border text-primary" role="status"></div>
                        <p class="mt-3">Yükleniyor...</p>
                    </div>
                    <div id="newReservationFormContainer" style="display: none;">
                        <!-- İçerik JavaScript ile doldurulacak -->
                    </div>
                </div>
            </div>
        </div>
    </div>
    </div>
@endsection

@push('styles')
<style>
    /* Modern Service Card Styles */
    #servicesCheckboxContainer {
        display: grid;
        grid-template-columns: repeat(2, 1fr);
        gap: 12px;
    }
    
    @media (max-width: 768px) {
        #servicesCheckboxContainer {
            grid-template-columns: 1fr;
        }
    }
    
    .service-card-wrapper {
        position: relative;
    }
    
    .service-checkbox {
        position: absolute;
        opacity: 0;
        width: 0;
        height: 0;
    }
    
    .service-card {
        display: block;
        background: #ffffff;
        border: 2px solid #e5e7eb;
        border-radius: 12px;
        padding: 14px;
        cursor: pointer;
        transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
        position: relative;
        overflow: hidden;
        height: 100%;
    }
    
    .service-card::before {
        content: '';
        position: absolute;
        top: 0;
        left: 0;
        right: 0;
        height: 4px;
        background: linear-gradient(90deg, #ec4899 0%, #be185d 100%);
        transform: scaleX(0);
        transform-origin: left;
        transition: transform 0.3s ease;
    }
    
    .service-card:hover {
        border-color: #ec4899;
        box-shadow: 0 4px 12px rgba(236, 72, 153, 0.15);
        transform: translateY(-2px);
    }
    
    .service-checkbox:checked + .service-card {
        background: linear-gradient(135deg, #fdf2f8 0%, #fce7f3 100%);
        border-color: #ec4899;
        border-width: 2px;
        box-shadow: 0 4px 16px rgba(236, 72, 153, 0.2);
    }
    
    .service-checkbox:checked + .service-card::before {
        transform: scaleX(1);
    }
    
    .service-card-header {
        display: flex;
        align-items: flex-start;
        gap: 10px;
        margin-bottom: 10px;
    }
    
    .service-card-icon {
        width: 40px;
        height: 40px;
        border-radius: 8px;
        background: linear-gradient(135deg, #fce7f3 0%, #fbcfe8 100%);
        display: flex;
        align-items: center;
        justify-content: center;
        flex-shrink: 0;
        transition: all 0.3s ease;
    }
    
    .service-checkbox:checked + .service-card .service-card-icon {
        background: linear-gradient(135deg, #ec4899 0%, #be185d 100%);
        transform: scale(1.1);
    }
    
    .service-card-icon i {
        font-size: 18px;
        color: #ec4899;
        transition: all 0.3s ease;
    }
    
    .service-checkbox:checked + .service-card .service-card-icon i {
        color: #ffffff;
    }
    
    .service-card-content {
        flex: 1;
        min-width: 0;
    }
    
    .service-card-title {
        font-size: 0.875rem;
        font-weight: 600;
        color: #1f2937;
        margin: 0 0 3px 0;
        line-height: 1.3;
    }
    
    .service-card-description {
        font-size: 0.75rem;
        color: #6b7280;
        margin: 0;
        line-height: 1.3;
        display: -webkit-box;
        -webkit-line-clamp: 2;
        -webkit-box-orient: vertical;
        overflow: hidden;
    }
    
    .service-card-check {
        width: 20px;
        height: 20px;
        border-radius: 50%;
        background: #e5e7eb;
        display: flex;
        align-items: center;
        justify-content: center;
        flex-shrink: 0;
        opacity: 0;
        transform: scale(0);
        transition: all 0.3s cubic-bezier(0.68, -0.55, 0.265, 1.55);
    }
    
    .service-checkbox:checked + .service-card .service-card-check {
        opacity: 1;
        transform: scale(1);
        background: #ec4899;
    }
    
    .service-card-check i {
        font-size: 12px;
        color: #ffffff;
    }
    
    .service-card-footer {
        display: flex;
        justify-content: space-between;
        align-items: center;
        padding-top: 10px;
        border-top: 1px solid #f3f4f6;
        gap: 8px;
    }
    
    .service-card-info {
        display: flex;
        gap: 6px;
        align-items: center;
        flex: 1;
        min-width: 0;
    }
    
    .service-card-badge {
        display: inline-flex;
        align-items: center;
        gap: 3px;
        padding: 3px 8px;
        background: #f3f4f6;
        border-radius: 5px;
        font-size: 0.7rem;
        color: #6b7280;
        font-weight: 500;
        white-space: nowrap;
    }
    
    .service-checkbox:checked + .service-card .service-card-badge {
        background: rgba(236, 72, 153, 0.1);
        color: #ec4899;
    }
    
    .service-card-badge i {
        font-size: 0.65rem;
    }
    
    .service-card-price {
        display: flex;
        align-items: center;
        flex-shrink: 0;
    }
    
    .service-card-price-amount {
        font-size: 1rem;
        font-weight: 700;
        color: #ec4899;
        background: linear-gradient(135deg, #ec4899 0%, #be185d 100%);
        -webkit-background-clip: text;
        -webkit-text-fill-color: transparent;
        background-clip: text;
        white-space: nowrap;
    }
    
    /* Service Category Tabs Styles */
    .service-category-tabs {
        margin-bottom: 12px;
    }
    
    .service-category-tab {
        padding: 8px 16px;
        border: none;
        background: #f3f4f6;
        color: #000000;
        border-radius: 8px;
        cursor: pointer;
        white-space: nowrap;
        /* transition kaldırıldı - animasyon yok */
        font-weight: 500;
        font-size: 0.85rem;
        position: relative;
    }
    
    .service-category-tab:hover {
        background: #e5e7eb;
        /* transform kaldırıldı - animasyon yok */
    }
    
    .service-category-tab.active {
        background: linear-gradient(135deg, #ec4899 0%, #be185d 100%) !important;
        color: white !important;
        box-shadow: 0 2px 8px rgba(236, 72, 153, 0.3);
    }
    
    .service-category-tab .badge {
        font-size: 0.7rem;
        padding: 2px 6px;
        border-radius: 10px;
    }
    
    .service-category-tab.active .badge {
        background: rgba(255, 255, 255, 0.2) !important;
        color: white !important;
    }
    
    #servicesCheckboxContainer {
        display: grid;
        grid-template-columns: repeat(2, 1fr);
        gap: 12px;
        scrollbar-width: thin;
        scrollbar-color: #ec4899 #f3f4f6;
        padding: 12px !important;
        background: #fafafa !important;
        border-radius: 12px !important;
    }
    
    @media (max-width: 768px) {
        #servicesCheckboxContainer {
            grid-template-columns: 1fr;
        }
    }
    
    #servicesCheckboxContainer::-webkit-scrollbar {
        width: 8px;
    }
    
    #servicesCheckboxContainer::-webkit-scrollbar-track {
        background: #f3f4f6;
        border-radius: 10px;
    }
    
    #servicesCheckboxContainer::-webkit-scrollbar-thumb {
        background: linear-gradient(135deg, #ec4899 0%, #be185d 100%);
        border-radius: 10px;
    }
    
    #servicesCheckboxContainer::-webkit-scrollbar-thumb:hover {
        background: linear-gradient(135deg, #be185d 0%, #9f1239 100%);
    }
    
    /* İşlem Butonları - Modern ve Güzel Tasarım */
    .btn-action {
        min-width: 38px;
        height: 38px;
        padding: 0;
        display: inline-flex;
        align-items: center;
        justify-content: center;
        border-radius: 10px;
        border: none;
        font-size: 14px;
        font-weight: 600;
        transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
        box-shadow: 0 2px 8px rgba(0, 0, 0, 0.1);
        position: relative;
        overflow: hidden;
    }
    
    .btn-action::before {
        content: '';
        position: absolute;
        top: 50%;
        left: 50%;
        width: 0;
        height: 0;
        border-radius: 50%;
        background: rgba(255, 255, 255, 0.3);
        transform: translate(-50%, -50%);
        transition: width 0.6s, height 0.6s;
    }
    
    .btn-action:hover::before {
        width: 300px;
        height: 300px;
    }
    
    .btn-action:hover {
        transform: translateY(-2px);
        box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
    }
    
    .btn-action:active {
        transform: translateY(0);
        box-shadow: 0 2px 6px rgba(0, 0, 0, 0.1);
    }
    
    .btn-action-info {
        background: linear-gradient(135deg, #3b82f6 0%, #2563eb 100%);
        color: white;
    }
    
    .btn-action-info:hover {
        background: linear-gradient(135deg, #2563eb 0%, #1d4ed8 100%);
        color: white;
    }
    
    .btn-action-warning {
        background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%);
        color: white;
    }
    
    .btn-action-warning:hover {
        background: linear-gradient(135deg, #d97706 0%, #b45309 100%);
        color: white;
    }
    
    .btn-action-success {
        background: linear-gradient(135deg, #10b981 0%, #059669 100%);
        color: white;
    }
    
    .btn-action-success:hover {
        background: linear-gradient(135deg, #059669 0%, #047857 100%);
        color: white;
    }
    
    .btn-action-danger {
        background: linear-gradient(135deg, #ef4444 0%, #dc2626 100%);
        color: white;
    }
    
    .btn-action-danger:hover {
        background: linear-gradient(135deg, #dc2626 0%, #b91c1c 100%);
        color: white;
    }
    
    /* Hızlı İşlemler Modal Butonları */
    .btn-quick-action {
        padding: 16px 24px;
        border-radius: 12px;
        border: none;
        font-size: 16px;
        font-weight: 600;
        transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
        box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
        position: relative;
        overflow: hidden;
        color: white;
        display: flex;
        align-items: center;
        justify-content: center;
    }
    
    .btn-quick-action::before {
        content: '';
        position: absolute;
        top: 50%;
        left: 50%;
        width: 0;
        height: 0;
        border-radius: 50%;
        background: rgba(255, 255, 255, 0.2);
        transform: translate(-50%, -50%);
        transition: width 0.6s, height 0.6s;
    }
    
    .btn-quick-action:hover::before {
        width: 400px;
        height: 400px;
    }
    
    .btn-quick-action:hover {
        transform: translateY(-3px);
        box-shadow: 0 8px 20px rgba(0, 0, 0, 0.15);
    }
    
    .btn-quick-action:active {
        transform: translateY(-1px);
        box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
    }
    
    .btn-quick-payment {
        background: linear-gradient(135deg, #10b981 0%, #059669 100%);
    }
    
    .btn-quick-payment:hover {
        background: linear-gradient(135deg, #059669 0%, #047857 100%);
    }
    
    .btn-quick-duplicate {
        background: linear-gradient(135deg, #3b82f6 0%, #2563eb 100%);
    }
    
    .btn-quick-duplicate:hover {
        background: linear-gradient(135deg, #2563eb 0%, #1d4ed8 100%);
    }
    
    .btn-quick-status {
        background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%);
    }
    
    .btn-quick-status:hover {
        background: linear-gradient(135deg, #d97706 0%, #b45309 100%);
    }
    
    .btn-quick-call {
        background: linear-gradient(135deg, #10b981 0%, #059669 100%);
    }
    
    .btn-quick-call:hover {
        background: linear-gradient(135deg, #059669 0%, #047857 100%);
    }
    
    .btn-quick-whatsapp {
        background: linear-gradient(135deg, #25D366 0%, #128C7E 100%);
    }
    
    .btn-quick-whatsapp:hover {
        background: linear-gradient(135deg, #128C7E 0%, #075E54 100%);
    }
    
    /* Mobil Görünüm */
    @media (max-width: 768px) {
        .btn-action {
            min-width: 36px;
            height: 36px;
            font-size: 13px;
        }
        
        .btn-quick-action {
            padding: 14px 20px;
            font-size: 15px;
        }
    }
</style>
@endpush

@push('scripts')
<script>
// Bekleyen Talepler Modal JavaScript
document.addEventListener('DOMContentLoaded', function() {
    const pendingModal = document.getElementById('pendingRequestsModal');
    if (pendingModal) {
        // Modal açıldığında
        pendingModal.addEventListener('show.bs.modal', function() {
            loadPendingRequests();
        });
        
        // Modal gösterildiğinde - aria-hidden'ı kaldır
        pendingModal.addEventListener('shown.bs.modal', function() {
            // Bootstrap modal açıldığında aria-hidden'ı false yapmalı ama bazen yapmıyor
            // Modal açıkken aria-hidden olmamalı
            if (pendingModal.classList.contains('show')) {
                pendingModal.removeAttribute('aria-hidden');
            }
        });
    }
});

function loadPendingRequests() {
    const tbody = document.getElementById('pendingRequestsTableBody');
    const mobileBody = document.getElementById('pendingRequestsMobileBody');
    
    if (tbody) {
        tbody.innerHTML = '<tr><td colspan="5" class="text-center py-4"><div class="spinner-border text-warning" role="status"></div></td></tr>';
    }
    if (mobileBody) {
        mobileBody.innerHTML = '<div class="text-center py-4"><div class="spinner-border text-warning" role="status"></div></div>';
    }
    
    fetch('{{ route("admin.beauty-salon.appointment-requests.index") }}', {
        headers: {
            'X-Requested-With': 'XMLHttpRequest',
            'Accept': 'application/json',
            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.content || ''
        }
    })
    .then(async response => {
        if (!response.ok) {
            const errorText = await response.text();
            console.error('HTTP Error:', response.status, errorText);
            throw new Error('HTTP ' + response.status + ': ' + errorText.substring(0, 100));
        }
        
        const contentType = response.headers.get('content-type');
        if (contentType && contentType.includes('application/json')) {
            return response.json();
        } else {
            const text = await response.text();
            console.error('Beklenmeyen response formatı:', text.substring(0, 200));
            throw new Error('Beklenmeyen response formatı: ' + contentType);
        }
    })
    .then(data => {
        console.log('Randevu talepleri verisi:', data);
        if (tbody) tbody.innerHTML = '';
        if (mobileBody) mobileBody.innerHTML = '';
        
        if (!data) {
            console.error('Data boş veya undefined');
            if (tbody) tbody.innerHTML = '<tr><td colspan="5" class="text-center py-4 text-danger">Veri alınamadı.</td></tr>';
            if (mobileBody) mobileBody.innerHTML = '<div class="alert alert-danger text-center">Veri alınamadı.</div>';
            return;
        }
        
        const requests = data.requests || [];
        console.log('Randevu talepleri sayısı:', requests.length);
        
        if (requests.length > 0) {
            requests.forEach(request => {
                const createdDate = new Date(request.created_at);
                const formattedDate = createdDate.toLocaleDateString('tr-TR', { day: '2-digit', month: '2-digit', year: 'numeric' });
                const formattedTime = createdDate.toLocaleTimeString('tr-TR', { hour: '2-digit', minute: '2-digit' });
                
                const customerHtml = request.customer ? 
                    '<br><small class="text-muted">Kayıtlı Müşteri</small>' : '';
                
                const serviceHtml = request.service ? 
                    '<span class="badge bg-primary">' + request.service.name + '</span><br><small class="text-muted">' + 
                    parseFloat(request.service.price).toLocaleString('tr-TR', {minimumFractionDigits: 2, maximumFractionDigits: 2}) + ' ₺</small>' : 
                    '<span class="text-muted">-</span>';
                
                // Desktop Table Row
                if (tbody) {
                    const row = document.createElement('tr');
                    row.innerHTML = 
                        '<td>' +
                            (request.name || '') + ' ' + (request.surname || '') + customerHtml +
                        '</td>' +
                        '<td>' + (request.phone || '') + '</td>' +
                        '<td>' + serviceHtml + '</td>' +
                        '<td><small>' + formattedDate + ' ' + formattedTime + '</small></td>' +
                        '<td>' +
                            '<div class="btn-group" role="group">' +
                                '<button class="btn btn-sm btn-primary" onclick="showRequestDetail(' + request.id + ')">' +
                                    '<i class="bi bi-eye"></i> İncele' +
                                '</button>' +
                                '<button class="btn btn-sm btn-danger" onclick="rejectAppointmentRequest(' + request.id + ')">' +
                                    '<i class="bi bi-x-circle"></i> İptal Et' +
                                '</button>' +
                            '</div>' +
                        '</td>';
                    tbody.appendChild(row);
                }
                
                // Mobile Card
                if (mobileBody) {
                    const card = document.createElement('div');
                    card.className = 'card mb-3 shadow-sm';
                    card.innerHTML = 
                        '<div class="card-body">' +
                            '<div class="d-flex justify-content-between align-items-start mb-2">' +
                                '<div>' +
                                    '<h6 class="mb-1 fw-bold">' + (request.name || '') + ' ' + (request.surname || '') + '</h6>' +
                                    (request.customer ? '<small class="text-muted">Kayıtlı Müşteri</small>' : '') +
                                '</div>' +
                                '<small class="text-muted">' + formattedDate + '<br>' + formattedTime + '</small>' +
                            '</div>' +
                            '<div class="mb-2">' +
                                '<small class="text-muted d-block mb-1">Telefon</small>' +
                                '<strong>' + (request.phone || '-') + '</strong>' +
                            '</div>' +
                            '<div class="mb-3">' +
                                '<small class="text-muted d-block mb-1">Hizmet</small>' +
                                serviceHtml +
                            '</div>' +
                            '<div class="d-grid gap-2">' +
                                '<button class="btn btn-primary btn-sm" onclick="showRequestDetail(' + request.id + ')">' +
                                    '<i class="bi bi-eye me-1"></i>İncele' +
                                '</button>' +
                                '<button class="btn btn-danger btn-sm" onclick="rejectAppointmentRequest(' + request.id + ')">' +
                                    '<i class="bi bi-x-circle me-1"></i>İptal Et' +
                                '</button>' +
                            '</div>' +
                        '</div>';
                    mobileBody.appendChild(card);
                }
            });
        } else {
            if (tbody) tbody.innerHTML = '<tr><td colspan="5" class="text-center py-4 text-muted">Bekleyen randevu talebi bulunmamaktadır.</td></tr>';
            if (mobileBody) mobileBody.innerHTML = '<div class="alert alert-info text-center">Bekleyen randevu talebi bulunmamaktadır.</div>';
        }
    })
        .catch(error => {
            console.error('Randevu talepleri yüklenirken hata:', error);
            const tbody = document.getElementById('pendingRequestsTableBody');
            const mobileBody = document.getElementById('pendingRequestsMobileBody');
            if (tbody) {
                tbody.innerHTML = 
                    '<tr><td colspan="5" class="text-center py-4 text-danger">' +
                    '<i class="bi bi-exclamation-triangle me-2"></i>Talepler yüklenirken bir hata oluştu.<br>' +
                    '<small>' + error.message + '</small>' +
                    '</td></tr>';
            }
            if (mobileBody) {
                mobileBody.innerHTML = 
                    '<div class="alert alert-danger text-center">' +
                    '<i class="bi bi-exclamation-triangle me-2"></i>Talepler yüklenirken bir hata oluştu.<br>' +
                    '<small>' + error.message + '</small>' +
                    '</div>';
            }
        });
}

// Randevu detayını göster
function showReservationDetail(reservationId) {
    const modal = document.getElementById('reservationDetailModal');
    const content = document.getElementById('reservationDetailContent');
    
    if (!modal || !content) {
        console.error('Modal elementleri bulunamadı');
        return;
    }
    
    content.innerHTML = '<div class="text-center py-5"><div class="spinner-border text-primary" role="status"></div><p class="mt-3">Yükleniyor...</p></div>';
    
    const bootstrapModal = new bootstrap.Modal(modal);
    bootstrapModal.show();
    
    // Modal açıldığında aria-hidden'ı kaldır
    modal.addEventListener('shown.bs.modal', function() {
        if (modal.classList.contains('show')) {
            modal.removeAttribute('aria-hidden');
        }
    }, { once: true });
    
    const showUrl = '{{ route("admin.appointments.show", ["id" => ":id"]) }}'.replace(':id', reservationId);
    
    fetch(showUrl, {
        headers: {
            'X-Requested-With': 'XMLHttpRequest',
            'Accept': 'application/json'
        }
    })
    .then(async response => {
        if (!response.ok) {
            throw new Error('HTTP ' + response.status);
        }
        const contentType = response.headers.get('content-type');
        if (contentType && contentType.includes('application/json')) {
            return response.json();
        } else {
            const text = await response.text();
            throw new Error('Beklenmeyen response formatı');
        }
    })
    .then(data => {
        if (data.error) {
            content.innerHTML = '<div class="alert alert-danger">' + data.error + '</div>';
            return;
        }
        
        const reservation = data.reservation || data;
        const customer = data.customer;
        const service = reservation.appointment_request?.service || (data.service ? { name: data.service.name, price: data.service.price } : null);
        
        let html = '<div class="row g-3">';
        
        // Sol taraf: Müşteri ve Randevu Bilgileri
        html += '<div class="col-md-6">';
        html += '<div class="card shadow-sm mb-3" style="border-radius: 12px; border: none;">';
        html += '<div class="card-header" style="background: linear-gradient(135deg, #ec4899 0%, #be185d 100%); border-radius: 12px 12px 0 0; padding: 12px 16px;">';
        html += '<h6 class="mb-0 text-white fw-bold"><i class="bi bi-person-circle me-2"></i>Müşteri Bilgileri</h6>';
        html += '</div>';
        html += '<div class="card-body p-3">';
        if (customer) {
            html += '<p class="mb-2"><strong>Ad Soyad:</strong> ' + (customer.name || '-') + ' ' + (customer.surname || '') + '</p>';
            html += '<p class="mb-2"><strong>Telefon:</strong> ' + (customer.phone || '-') + '</p>';
            if (customer.email) {
                html += '<p class="mb-2"><strong>E-posta:</strong> ' + customer.email + '</p>';
            }
        } else {
            html += '<p class="text-muted">Müşteri bilgisi bulunamadı.</p>';
        }
        html += '</div></div>';
        
        html += '<div class="card shadow-sm" style="border-radius: 12px; border: none;">';
        html += '<div class="card-header" style="background: linear-gradient(135deg, #f472b6 0%, #ec4899 100%); border-radius: 12px 12px 0 0; padding: 12px 16px;">';
        html += '<h6 class="mb-0 text-white fw-bold"><i class="bi bi-calendar3 me-2"></i>Randevu Bilgileri</h6>';
        html += '</div>';
        html += '<div class="card-body p-3">';
        html += '<p class="mb-2"><strong>Tarih:</strong> ' + (reservation.date ? new Date(reservation.date).toLocaleDateString('tr-TR') : '-') + '</p>';
        html += '<p class="mb-2"><strong>Saat:</strong> ' + (reservation.start_time || '-') + '</p>';
        if (service) {
            html += '<p class="mb-2"><strong>Hizmet:</strong> ' + service.name + '</p>';
            if (service.price) {
                html += '<p class="mb-2"><strong>Fiyat:</strong> ' + parseFloat(service.price).toLocaleString('tr-TR', {minimumFractionDigits: 2, maximumFractionDigits: 2}) + ' ₺</p>';
            }
        }
        if (reservation.personel) {
            html += '<p class="mb-2"><strong>Personel:</strong> ' + reservation.personel.name + '</p>';
        }
        if (reservation.room) {
            html += '<p class="mb-2"><strong>Oda:</strong> ' + reservation.room.name + '</p>';
        }
        html += '<p class="mb-2"><strong>Durum:</strong> ';
        if (reservation.status === 'pending' || reservation.status === 'open') {
            html += '<span class="badge bg-warning">Beklemede</span>';
        } else if (reservation.status === 'confirmed') {
            html += '<span class="badge bg-success">Onaylandı</span>';
        } else if (reservation.status === 'completed') {
            html += '<span class="badge bg-info">Tamamlandı</span>';
        } else {
            html += '<span class="badge bg-danger">İptal</span>';
        }
        html += '</p>';
        if (reservation.notes) {
            html += '<p class="mb-0"><strong>Notlar:</strong> ' + reservation.notes + '</p>';
        }
        html += '</div></div>';
        html += '</div>';
        
        // Sağ taraf: Ödeme Bilgileri ve İşlemler
        html += '<div class="col-md-6">';
        html += '<div class="card shadow-sm mb-3" style="border-radius: 12px; border: none;">';
        html += '<div class="card-header" style="background: linear-gradient(135deg, #ec4899 0%, #be185d 100%); border-radius: 12px 12px 0 0; padding: 12px 16px;">';
        html += '<h6 class="mb-0 text-white fw-bold"><i class="bi bi-cash-coin me-2"></i>Ödeme Bilgileri</h6>';
        html += '</div>';
        html += '<div class="card-body p-3">';
        html += '<p class="mb-2"><strong>Toplam Tutar:</strong> <span class="text-success fw-bold">' + parseFloat(reservation.total_price || 0).toLocaleString('tr-TR', {minimumFractionDigits: 2, maximumFractionDigits: 2}) + ' ₺</span></p>';
        html += '<p class="mb-2"><strong>Ödenen:</strong> ' + parseFloat(reservation.deposit_paid || 0).toLocaleString('tr-TR', {minimumFractionDigits: 2, maximumFractionDigits: 2}) + ' ₺</p>';
        const remaining = parseFloat(reservation.remaining_amount || 0);
        html += '<p class="mb-0"><strong>Kalan:</strong> <span class="' + (remaining > 0 ? 'text-danger' : 'text-success') + ' fw-bold">' + remaining.toLocaleString('tr-TR', {minimumFractionDigits: 2, maximumFractionDigits: 2}) + ' ₺</span></p>';
        html += '</div></div>';
        
        html += '<div class="card shadow-sm" style="border-radius: 12px; border: none;">';
        html += '<div class="card-header" style="background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%); border-radius: 12px 12px 0 0; padding: 12px 16px;">';
        html += '<h6 class="mb-0 text-white fw-bold"><i class="bi bi-gear me-2"></i>İşlemler</h6>';
        html += '</div>';
        html += '<div class="card-body p-3">';
        html += '<div class="d-grid gap-2">';
        
        // Onayla butonu (sadece pending/open durumunda)
        if (reservation.status === 'pending' || reservation.status === 'open') {
            html += '<button type="button" class="btn btn-success fw-bold" onclick="confirmReservation(' + reservation.id + ')">';
            html += '<i class="bi bi-check-circle me-2"></i>Onayla';
            html += '</button>';
        }
        
        // Bitir butonu (sadece confirmed durumunda)
        if (reservation.status === 'confirmed') {
            html += '<button type="button" class="btn btn-primary fw-bold" onclick="completeReservation(' + reservation.id + ')">';
            html += '<i class="bi bi-check2-all me-2"></i>Bitir ve Kasaya Ekle';
            html += '</button>';
        }
        
        html += '<button type="button" class="btn btn-secondary fw-bold" data-bs-dismiss="modal">';
        html += '<i class="bi bi-x-circle me-2"></i>Kapat';
        html += '</button>';
        html += '</div></div></div>';
        html += '</div></div>';
        
        content.innerHTML = html;
    })
    .catch(error => {
        console.error('Randevu detayı yüklenirken hata:', error);
        content.innerHTML = '<div class="alert alert-danger"><i class="bi bi-exclamation-triangle me-2"></i>Randevu detayı yüklenirken bir hata oluştu: ' + error.message + '</div>';
    });
}

// Randevuyu onayla
function confirmReservation(reservationId) {
    if (!confirm('Bu randevuyu onaylamak istediğinize emin misiniz?')) return;
    
    const confirmUrl = '{{ route("admin.appointments.confirm", ["id" => ":id"]) }}'.replace(':id', reservationId);
    
    fetch(confirmUrl, {
        method: 'POST',
        headers: {
            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.content || '',
            'X-Requested-With': 'XMLHttpRequest',
            'Accept': 'application/json'
        }
    })
    .then(async response => {
        const data = await response.json();
        if (!response.ok) {
            throw new Error(data.message || 'HTTP ' + response.status);
        }
        return data;
    })
    .then(data => {
        if (data.success) {
            alert('Randevu başarıyla onaylandı!');
            const modal = bootstrap.Modal.getInstance(document.getElementById('reservationDetailModal'));
            if (modal) modal.hide();
            setTimeout(() => window.location.reload(), 1000);
        } else {
            alert(data.message || 'Hata oluştu');
        }
    })
    .catch(error => {
        console.error('Randevu onaylama hatası:', error);
        alert(error.message || 'Randevu onaylanırken bir hata oluştu.');
    });
}

// Randevuyu bitir ve kasaya ekle
function completeReservation(reservationId) {
    if (!confirm('Bu randevuyu bitirmek ve kalan ödemeyi kasaya eklemek istediğinize emin misiniz?')) return;
    
    const completeUrl = '{{ route("admin.appointments.complete", ["id" => ":id"]) }}'.replace(':id', reservationId);
    
    fetch(completeUrl, {
        method: 'POST',
        headers: {
            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.content || '',
            'X-Requested-With': 'XMLHttpRequest',
            'Accept': 'application/json'
        }
    })
    .then(async response => {
        const data = await response.json();
        if (!response.ok) {
            throw new Error(data.message || 'HTTP ' + response.status);
        }
        return data;
    })
    .then(data => {
        if (data.success) {
            alert(data.message || 'Randevu başarıyla tamamlandı!');
            const modal = bootstrap.Modal.getInstance(document.getElementById('reservationDetailModal'));
            if (modal) modal.hide();
            setTimeout(() => window.location.reload(), 1000);
        } else {
            alert(data.message || 'Hata oluştu');
        }
    })
    .catch(error => {
        console.error('Randevu tamamlama hatası:', error);
        alert(error.message || 'Randevu tamamlanırken bir hata oluştu.');
    });
}

function showRequestDetail(requestId) {
    const requestDetailModal = document.getElementById('requestDetailModal');
    if (!requestDetailModal) {
        console.error('requestDetailModal bulunamadı');
        return;
    }
    
    document.getElementById('requestDetailContent').innerHTML = '<div class="text-center py-5"><div class="spinner-border text-primary" role="status"></div><p class="mt-3">Yükleniyor...</p></div>';
    
    // Modal event listener'ları ekle (sadece bir kez)
    if (!requestDetailModal.hasAttribute('data-listeners-added')) {
        requestDetailModal.setAttribute('data-listeners-added', 'true');
        
        // Modal gösterildiğinde - aria-hidden'ı kaldır
        requestDetailModal.addEventListener('shown.bs.modal', function() {
            // Bootstrap modal açıldığında aria-hidden'ı false yapmalı ama bazen yapmıyor
            if (requestDetailModal.classList.contains('show')) {
                requestDetailModal.removeAttribute('aria-hidden');
            }
        });
    }
    
    const modal = new bootstrap.Modal(requestDetailModal);
    modal.show();
    
    const showUrlBase = '{{ route("admin.beauty-salon.appointment-requests.show", ["id" => ":id"]) }}';
    const showUrl = showUrlBase.replace(':id', requestId);
    
    fetch(showUrl, {
        headers: {
            'X-Requested-With': 'XMLHttpRequest',
            'Accept': 'text/html, application/json'
        }
    })
    .then(response => {
        if (!response.ok) {
            if (response.status === 404) {
                throw new Error('Randevu talebi bulunamadı.');
            }
            throw new Error('HTTP ' + response.status);
        }
        return response.text();
    })
    .then(html => {
        // Önce JSON olup olmadığını kontrol et
        let jsonData = null;
        try {
            jsonData = JSON.parse(html);
            if (jsonData.request) {
                // JSON response - bu durumda HTML view render etmek için tekrar istek yap
                console.warn('JSON response alındı, HTML view için tekrar istek yapılıyor...');
                // Accept header'ını sadece text/html olarak değiştir
                fetch(showUrl, {
                    headers: {
                        'X-Requested-With': 'XMLHttpRequest',
                        'Accept': 'text/html'
                    }
                })
                .then(response => response.text())
                .then(htmlContent => {
                    const parser = new DOMParser();
                    const doc = parser.parseFromString(htmlContent, 'text/html');
                    const containerElement = doc.querySelector('.container-fluid');
                    
                    if (containerElement) {
                        document.getElementById('requestDetailContent').innerHTML = containerElement.innerHTML;
                        // Form ve event listener'ları başlat
                        setTimeout(function() {
                            initializeRequestDetailForm(requestId);
                        }, 300);
                    } else {
                        document.getElementById('requestDetailContent').innerHTML = '<div class="alert alert-danger">İçerik bulunamadı! (container-fluid bulunamadı)</div>';
                    }
                })
                .catch(error => {
                    console.error('HTML view yüklenirken hata:', error);
                    document.getElementById('requestDetailContent').innerHTML = '<div class="alert alert-danger">Randevu talebi yüklenirken bir hata oluştu: ' + error.message + '</div>';
                });
                return;
            }
        } catch (e) {
            // JSON değil, HTML olarak devam et
        }
        
        // HTML response
        const parser = new DOMParser();
        const doc = parser.parseFromString(html, 'text/html');
        const containerElement = doc.querySelector('.container-fluid');
        
        if (!containerElement) {
            // Başka bir selector dene
            const rowElement = doc.querySelector('.row');
            if (rowElement) {
                document.getElementById('requestDetailContent').innerHTML = rowElement.outerHTML;
                // Form ve event listener'ları başlat
                setTimeout(function() {
                    initializeRequestDetailForm(requestId);
                }, 300);
                return;
            }
            
            // Hiçbir şey bulunamadı, tüm HTML'i göster
            console.error('Container bulunamadı, tüm HTML:', html.substring(0, 500));
            document.getElementById('requestDetailContent').innerHTML = '<div class="alert alert-danger">İçerik bulunamadı! (HTML parse edilemedi)</div>';
            return;
        }
        
        document.getElementById('requestDetailContent').innerHTML = containerElement.innerHTML;
        
        // Form ve event listener'ları başlat
        setTimeout(function() {
            initializeRequestDetailForm(requestId);
        }, 300);
    })
    .catch(error => {
        console.error('Hata:', error);
        document.getElementById('requestDetailContent').innerHTML = '<div class="alert alert-danger">Randevu talebi yüklenirken bir hata oluştu: ' + error.message + '</div>';
    });
}

// Form ve event listener'ları başlat
function initializeRequestDetailForm(requestId) {
    const hourSelect = document.getElementById('appointment_hour');
    const minuteSelect = document.getElementById('appointment_minute');
    const timeInput = document.getElementById('appointment_time');
    const dateInput = document.getElementById('appointment_date');
    const personelSelect = document.getElementById('personel_id');
    
    // Dolu saatleri yükle
    let busyHours = {};
    
    function loadBusyHours() {
        const personelId = personelSelect?.value;
        const date = dateInput?.value;
        
        if (!personelId || !date) {
            busyHours = {};
            updateTimeOptions();
            return;
        }
        
        fetch(`/admin/guzellik-salonu/personel/${personelId}/busy-hours?date=${date}`, {
            headers: {
                'X-Requested-With': 'XMLHttpRequest',
                'Accept': 'application/json'
            }
        })
        .then(async response => {
            if (!response.ok) {
                throw new Error('HTTP ' + response.status);
            }
            const contentType = response.headers.get('content-type');
            if (contentType && contentType.includes('application/json')) {
                return response.json();
            } else {
                const text = await response.text();
                throw new Error('Beklenmeyen response formatı: ' + contentType);
            }
        })
        .then(data => {
            busyHours = data.busy_hours || {};
            updateTimeOptions();
        })
        .catch(error => {
            console.error('Dolu saatler yüklenemedi:', error);
            busyHours = {};
            updateTimeOptions();
        });
    }
    
    function updateTimeOptions() {
        const busyHoursInfo = document.getElementById('busyHoursInfo');
        const busyHoursText = document.getElementById('busyHoursText');
        
        if (!hourSelect || !minuteSelect) return;
        
        // Tüm saat seçeneklerini kontrol et
        Array.from(hourSelect.options).forEach(option => {
            if (option.value) {
                const hour = option.value;
                let hasAvailableMinute = false;
                
                ['00', '10', '20', '30', '40', '50'].forEach(minute => {
                    const timeSlot = `${hour}:${minute}`;
                    if (!busyHours[timeSlot]) {
                        hasAvailableMinute = true;
                    }
                });
                
                if (!hasAvailableMinute) {
                    option.style.color = '#ef4444';
                    option.style.fontWeight = 'bold';
                    option.textContent = option.value + ' (Dolu)';
                } else {
                    option.style.color = '';
                    option.style.fontWeight = '';
                    option.textContent = option.value;
                }
            }
        });
        
        // Seçili saat ve dakikayı kontrol et
        const selectedHour = hourSelect.value;
        const selectedMinute = minuteSelect.value;
        
        if (selectedHour && selectedMinute) {
            const selectedTime = `${selectedHour}:${selectedMinute}`;
            if (busyHours[selectedTime]) {
                if (busyHoursInfo) {
                    busyHoursInfo.style.display = 'block';
                    busyHoursText.textContent = 'Bu saat dolu! Lütfen başka bir saat seçin.';
                    busyHoursInfo.className = 'text-danger';
                }
            } else {
                if (busyHoursInfo) busyHoursInfo.style.display = 'none';
            }
        } else {
            const busyCount = Object.keys(busyHours).length;
            if (busyCount > 0 && dateInput?.value && personelSelect?.value) {
                if (busyHoursInfo) {
                    busyHoursInfo.style.display = 'block';
                    busyHoursText.textContent = `${busyCount} saat dolu. Lütfen boş saatleri seçin.`;
                    busyHoursInfo.className = 'text-warning';
                }
            } else {
                if (busyHoursInfo) busyHoursInfo.style.display = 'none';
            }
        }
        
        // Dakika seçeneklerini güncelle
        Array.from(minuteSelect.options).forEach(option => {
            if (option.value && selectedHour) {
                const timeSlot = `${selectedHour}:${option.value}`;
                if (busyHours[timeSlot]) {
                    option.disabled = true;
                    option.style.color = '#ef4444';
                    option.textContent = option.value + ' (Dolu)';
                } else {
                    option.disabled = false;
                    option.style.color = '';
                    option.textContent = option.value;
                }
            }
        });
    }
    
    function updateTime() {
        if (hourSelect && minuteSelect && timeInput) {
            if (hourSelect.value && minuteSelect.value) {
                const timeString = hourSelect.value + ':' + minuteSelect.value;
                timeInput.value = timeString;
                
                // Dolu saat kontrolü
                if (busyHours[timeString]) {
                    alert('Bu saat dolu! Lütfen başka bir saat seçin.');
                    timeInput.value = '';
                    return;
                }
            }
        }
        updateTimeOptions();
    }
    
    if (hourSelect) hourSelect.addEventListener('change', updateTime);
    if (minuteSelect) minuteSelect.addEventListener('change', updateTime);
    if (dateInput) dateInput.addEventListener('change', loadBusyHours);
    if (personelSelect) personelSelect.addEventListener('change', loadBusyHours);
    
    // İlk yükleme
    if (dateInput?.value && personelSelect?.value) {
        loadBusyHours();
    }
    
    // Form submit event listener
    const form = document.querySelector('#requestDetailContent form#approveRequestForm');
    if (form) {
        form.addEventListener('submit', function(e) {
            e.preventDefault();
            e.stopPropagation();
            
            const hourSelect = document.getElementById('appointment_hour');
            const minuteSelect = document.getElementById('appointment_minute');
            const timeInput = document.getElementById('appointment_time');
            
            if (hourSelect && minuteSelect && timeInput) {
                if (!hourSelect.value || !minuteSelect.value) {
                    alert('Lütfen saat ve dakika seçiniz.');
                    return false;
                }
                timeInput.value = hourSelect.value + ':' + minuteSelect.value;
            }
            
            if (!form.checkValidity()) {
                form.reportValidity();
                return false;
            }
            
            approveRequest(requestId, form);
            return false;
        });
    }
    
}

function approveRequest(requestId, form, createInvoice = false) {
    const submitButton = form.querySelector('button[type="submit"]');
    const originalButtonText = submitButton ? submitButton.innerHTML : '';
    
    if (submitButton) {
        submitButton.disabled = true;
        submitButton.innerHTML = '<i class="bi bi-hourglass-split me-2"></i>Onaylanıyor...';
    }
    
    const formData = new FormData(form);
    const approveUrlBase = '{{ route("admin.beauty-salon.appointment-requests.approve", ["id" => ":id"]) }}';
    const approveUrl = approveUrlBase.replace(':id', requestId);
    
    fetch(approveUrl, {
        method: 'POST',
        body: formData,
        headers: {
            'X-Requested-With': 'XMLHttpRequest',
            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.content || '',
            'Accept': 'application/json'
        }
    })
    .then(async response => {
        const contentType = response.headers.get('content-type');
        
        if (contentType && contentType.includes('application/json')) {
            const data = await response.json();
            if (!response.ok) {
                throw new Error(data.message || 'HTTP ' + response.status);
            }
            return data;
        }
        
        if (!response.ok) {
            const text = await response.text();
            throw new Error('HTTP ' + response.status);
        }
        
        return { success: true, reload: true };
    })
    .then(data => {
        if (submitButton) {
            submitButton.disabled = false;
            submitButton.innerHTML = originalButtonText;
        }
        
        const detailModal = bootstrap.Modal.getInstance(document.getElementById('requestDetailModal'));
        const pendingModal = bootstrap.Modal.getInstance(document.getElementById('pendingRequestsModal'));
        if (detailModal) detailModal.hide();
        if (pendingModal) pendingModal.hide();
        
        const message = data.message || 'Randevu talebi başarıyla onaylandı!';
        if (data.whatsapp_sent) {
            alert(message + ' WhatsApp mesajı gönderildi.');
        } else if (data.sms_sent) {
            alert(message + ' SMS gönderildi.');
        } else {
            alert(message);
        }
        setTimeout(() => window.location.reload(), 1000);
    })
    .catch(error => {
        console.error('Onaylama hatası:', error);
        
        if (submitButton) {
            submitButton.disabled = false;
            submitButton.innerHTML = originalButtonText;
        }
        
        alert(error.message || 'Randevu onaylanırken bir hata oluştu.');
    });
}

async function rejectAppointmentRequest(requestId) {
    if (!confirm('Bu randevu talebini iptal etmek istediğinize emin misiniz?')) return;
    
    const rejectUrlBase = '{{ route("admin.beauty-salon.appointment-requests.reject", ["id" => ":id"]) }}';
    const rejectUrl = rejectUrlBase.replace(':id', requestId);
    
    fetch(rejectUrl, {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.content || '',
            'X-Requested-With': 'XMLHttpRequest',
            'Accept': 'application/json'
        }
    })
    .then(async response => {
        const contentType = response.headers.get('content-type');
        if (!response.ok) {
            if (contentType && contentType.includes('application/json')) {
                const err = await response.json();
                throw new Error(err.message || err.error || 'HTTP ' + response.status);
            } else {
                const text = await response.text();
                throw new Error('HTTP ' + response.status);
            }
        }
        if (contentType && contentType.includes('application/json')) {
            return response.json();
        } else {
            throw new Error('Beklenmeyen response formatı');
        }
    })
    .then(data => {
        if (data.success) {
            alert('Randevu talebi başarıyla iptal edildi!');
            
            const detailModal = bootstrap.Modal.getInstance(document.getElementById('requestDetailModal'));
            const pendingModal = bootstrap.Modal.getInstance(document.getElementById('pendingRequestsModal'));
            if (detailModal) detailModal.hide();
            if (pendingModal) pendingModal.hide();
            
            setTimeout(() => {
                loadPendingRequests();
                location.reload();
            }, 1000);
        } else {
            alert(data.message || 'Hata oluştu');
        }
    })
    .catch(error => {
        console.error('Randevu talebi iptal hatası:', error);
        alert(error.message || 'Randevu talebi iptal edilirken bir hata oluştu.');
    });
}

// ============================================
// HIZLI İŞLEMLER - YENİ ÖZELLİKLER
// ============================================

// Hızlı İşlemler Modal'ını Göster
function showQuickActionsModal(reservationId, totalPrice, depositPaid, status, customerPhone) {
    const modal = document.getElementById('quickActionsModal');
    const content = document.getElementById('quickActionsContent');
    
    let html = `
        <div class="d-grid gap-3">
            <button type="button" class="btn btn-quick-action btn-quick-payment" onclick="quickPayment(${reservationId}, ${totalPrice}, ${depositPaid}); bootstrap.Modal.getInstance(document.getElementById('quickActionsModal')).hide();">
                <i class="bi bi-cash-coin me-2"></i>Hızlı Ödeme Al
            </button>
            <button type="button" class="btn btn-quick-action btn-quick-status" onclick="quickStatusChange(${reservationId}, '${status}'); bootstrap.Modal.getInstance(document.getElementById('quickActionsModal')).hide();">
                <i class="bi bi-arrow-repeat me-2"></i>Durum Değiştir
            </button>
    `;
    
    if (customerPhone) {
        // Telefon numarasını temizle ve +90 ekle
        let cleanPhone = customerPhone.replace(/[\s\(\)\-]/g, '');
        
        // Eğer 0 ile başlıyorsa 0'ı kaldır
        if (cleanPhone.startsWith('0')) {
            cleanPhone = cleanPhone.substring(1);
        }
        
        // Eğer +90 ile başlamıyorsa +90 ekle
        if (!cleanPhone.startsWith('+90') && !cleanPhone.startsWith('90')) {
            cleanPhone = '90' + cleanPhone;
        } else if (cleanPhone.startsWith('90') && !cleanPhone.startsWith('+90')) {
            cleanPhone = '+' + cleanPhone;
        }
        
        html += `
            <hr class="my-3" style="border-color: #e5e7eb;">
            <button type="button" class="btn btn-quick-action btn-quick-call" onclick="window.location.href='tel:${customerPhone}';">
                <i class="bi bi-telephone me-2"></i>Ara
            </button>
            <button type="button" class="btn btn-quick-action btn-quick-whatsapp" onclick="window.open('https://wa.me/${cleanPhone}', '_blank');">
                <i class="bi bi-whatsapp me-2"></i>WhatsApp
            </button>
        `;
    }
    
    html += `</div>`;
    content.innerHTML = html;
    
    const bsModal = new bootstrap.Modal(modal);
    bsModal.show();
}

// Hızlı Randevu Oluştur
function quickCreateReservation(selectedDate = null) {
    const modal = document.getElementById('newReservationModal');
    const loading = document.getElementById('newReservationLoading');
    const formContainer = document.getElementById('newReservationFormContainer');
    
    // Loading göster, formu gizle
    loading.style.display = 'block';
    formContainer.style.display = 'none';
    
    // Modal'ı aç
    const bsModal = new bootstrap.Modal(modal);
    bsModal.show();
    
    // Seçili tarih varsa sakla (form oluşturulduğunda kullanılacak)
    window.selectedDateForQuickReservation = selectedDate;
    
    // Personel, oda ve müşteri verilerini yükle
    Promise.all([
        fetch('{{ route("admin.customers.search") }}?q=', {
            headers: {
                'X-Requested-With': 'XMLHttpRequest',
                'Accept': 'application/json'
            }
        }).then(r => {
            if (!r.ok) {
                console.error('Müşteri verileri yüklenemedi:', r.status);
                return [];
            }
            return r.json();
        }).catch(err => {
            console.error('Müşteri verileri yükleme hatası:', err);
            return [];
        }),
        fetch('{{ route("admin.beauty-salon.appointment-requests.data") }}', {
            headers: {
                'X-Requested-With': 'XMLHttpRequest',
                'Accept': 'application/json'
            }
        }).then(r => {
            if (!r.ok) {
                console.error('Randevu verileri yüklenemedi:', r.status);
                return {personels: [], rooms: [], services: [], categories: []};
            }
            return r.json();
        }).catch(err => {
            console.error('Randevu verileri yükleme hatası:', err);
            return {personels: [], rooms: [], services: [], categories: []};
        })
    ])
    .then(([customers, data]) => {
        console.log('Yüklenen veriler:', {customers, data});
        const personels = data.personels || [];
        const rooms = data.rooms || [];
        const services = data.services || [];
        const categories = data.categories || [];
        console.log('Personel sayısı:', personels.length, 'Oda sayısı:', rooms.length, 'Hizmet sayısı:', services.length);
        // Form HTML'ini oluştur
        const today = new Date().toISOString().split('T')[0];
        
        let customersHtml = '<option value="">Seçiniz</option>';
        if (customers && customers.length > 0) {
            customers.forEach(customer => {
                customersHtml += `<option value="${customer.id}">${customer.name}${customer.phone ? ' (' + customer.phone + ')' : ''}</option>`;
            });
        }
        
        let personelsHtml = '<option value="">Seçiniz</option>';
        if (personels && personels.length > 0) {
            personels.forEach(personel => {
                personelsHtml += `<option value="${personel.id}">${personel.name}</option>`;
            });
        }
        
        let roomsHtml = '<option value="">Seçiniz (Opsiyonel)</option>';
        if (rooms && rooms.length > 0) {
            rooms.forEach(room => {
                roomsHtml += `<option value="${room.id}">${room.name}</option>`;
            });
        }
        
        // Hizmetleri kategorilere göre grupla
        const servicesByCategory = {};
        services.forEach(service => {
            const categoryId = service.service_category_id || 'uncategorized';
            if (!servicesByCategory[categoryId]) {
                servicesByCategory[categoryId] = [];
            }
            servicesByCategory[categoryId].push(service);
        });
        
        // Kategori sekmeleri HTML'i
        let categoryTabsHtml = '';
        if (categories && categories.length > 0) {
            categoryTabsHtml = `
                <div class="service-category-tabs mb-3" style="border-bottom: 2px solid #e5e7eb; padding-bottom: 12px;">
                    <div class="d-flex flex-wrap gap-2" style="overflow-x: auto; -webkit-overflow-scrolling: touch;">
                        <button type="button" class="service-category-tab active" data-category-id="all" onclick="filterServicesByCategory('all')" style="padding: 8px 16px; border: none; background: linear-gradient(135deg, #ec4899 0%, #be185d 100%); color: white; border-radius: 8px; cursor: pointer; white-space: nowrap; transition: all 0.3s; font-weight: 500; font-size: 0.85rem; box-shadow: 0 2px 8px rgba(236, 72, 153, 0.3);">
                            <i class="bi bi-grid me-1"></i>Tümü
                        </button>
                        ${categories.map(category => {
                            const categoryServicesCount = servicesByCategory[category.id]?.length || 0;
                            return `
                                <button type="button" class="service-category-tab" data-category-id="${category.id}" onclick="filterServicesByCategory(${category.id})" style="padding: 8px 16px; border: none; background: #f3f4f6; color: #000000; border-radius: 8px; cursor: pointer; white-space: nowrap; transition: all 0.3s; font-weight: 500; font-size: 0.85rem; ${category.color ? 'border-left: 3px solid ' + category.color + ';' : ''}" onmouseover="this.style.background='#e5e7eb'; this.style.transform='translateY(-2px)';" onmouseout="if(!this.classList.contains('active')) this.style.background='#f3f4f6'; this.style.transform='translateY(0)';">
                                    ${category.name || 'Kategorisiz'}
                                    ${categoryServicesCount > 0 ? `<span class="badge bg-secondary ms-1" style="font-size: 0.7rem; background: rgba(0,0,0,0.1) !important; color: #000 !important;">${categoryServicesCount}</span>` : ''}
                                </button>
                            `;
                        }).join('')}
                    </div>
                </div>
            `;
        }
        
        // Her kategori için hizmet kartlarını oluştur
        function generateServicesHtml(categoryId) {
            let html = '';
            const servicesToShow = categoryId === 'all' ? services : (servicesByCategory[categoryId] || []);
            
            if (servicesToShow.length > 0) {
                servicesToShow.forEach(service => {
                    const price = parseFloat(service.price || 0).toLocaleString('tr-TR', {minimumFractionDigits: 2, maximumFractionDigits: 2});
                    const duration = service.duration || 30;
                    html += `
                        <div class="service-card-wrapper" data-service-id="${service.id}" data-service-category-id="${service.service_category_id || 'uncategorized'}">
                            <input class="service-checkbox" 
                                   type="checkbox" 
                                   name="service_id[]" 
                                   value="${service.id}" 
                                   id="service_${service.id}" 
                                   data-price="${service.price || 0}" 
                                   data-duration="${duration}"
                                   onchange="updateServicePrice()">
                            <label class="service-card" for="service_${service.id}">
                                <div class="service-card-header">
                                    <div class="service-card-icon">
                                        <i class="bi bi-scissors"></i>
                                    </div>
                                    <div class="service-card-content">
                                        <h6 class="service-card-title">${service.name}</h6>
                                        ${service.description ? `<p class="service-card-description">${service.description.substring(0, 60)}${service.description.length > 60 ? '...' : ''}</p>` : ''}
                                    </div>
                                    <div class="service-card-check">
                                        <i class="bi bi-check-circle-fill"></i>
                                    </div>
                                </div>
                                <div class="service-card-footer">
                                    <div class="service-card-info">
                                        ${duration ? `<span class="service-card-badge"><i class="bi bi-clock"></i> ${duration} dk</span>` : ''}
                                    </div>
                                    <div class="service-card-price">
                                        <span class="service-card-price-amount">${price} ₺</span>
                                    </div>
                                </div>
                            </label>
                        </div>
                    `;
                });
            } else {
                html = '<p class="text-muted text-center mb-0" style="font-size: 0.85rem; padding: 20px; grid-column: 1 / -1;">Bu kategoride henüz hizmet bulunmamaktadır.</p>';
            }
            return html;
        }
        
        let servicesHtml = generateServicesHtml('all');
        
        // Global olarak kategori ve hizmet verilerini sakla
        window.servicesData = services;
        window.servicesByCategoryData = servicesByCategory;
        window.generateServicesHtmlFunction = generateServicesHtml;
        
        // Takvimden seçilen tarih varsa kullan
        const dateToUse = window.selectedDateForQuickReservation || today;
        
        const formHtml = `
            <div class="row g-3" style="margin: 0;">
                <!-- Sol Taraf: Müşteri Bilgileri -->
                <div class="col-lg-6 col-md-12" style="padding: 0 0.75rem;">
                    <div class="card shadow-sm mb-3" style="border-radius: 12px; border: none;">
                        <div class="card-header" style="background: linear-gradient(135deg, #ec4899 0%, #be185d 100%); border-radius: 12px 12px 0 0; padding: 12px 16px;">
                            <h6 class="mb-0 text-white fw-bold" style="font-size: 0.95rem;">
                                <i class="bi bi-person-circle me-2"></i>Müşteri Seçimi
                            </h6>
                        </div>
                        <div class="card-body p-3">
                            <div class="mb-3">
                                <label class="form-label fw-bold" style="font-size: 0.85rem;">
                                    <i class="bi bi-person me-1"></i>Müşteri <span class="text-danger">*</span>
                                </label>
                                <div class="d-flex gap-2">
                                    <select name="customer_id" id="modal_customer_id" class="form-select" required style="border-radius: 8px; border: 1px solid #e5e7eb; padding: 8px; font-size: 0.9rem; flex: 1;">
                                        ${customersHtml}
                                    </select>
                                    <button type="button" class="btn btn-sm btn-outline-primary" onclick="openCreateCustomerModal()" style="border-radius: 8px; padding: 8px 12px; white-space: nowrap;" title="Yeni Müşteri Ekle">
                                        <i class="bi bi-plus-circle"></i>
                                    </button>
                                </div>
                            </div>
                            <div id="customerInfoContainer" style="display: none;">
                                <!-- Müşteri bilgileri buraya yüklenecek -->
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Sağ Taraf: Randevu Formu -->
                <div class="col-lg-6 col-md-12" style="padding: 0 0.75rem;">
                    <div class="card shadow-sm" style="border-radius: 12px; border: none;">
                        <div class="card-header" style="background: linear-gradient(135deg, #ec4899 0%, #be185d 100%); border-radius: 12px 12px 0 0; padding: 12px 16px;">
                            <h6 class="mb-0 text-white fw-bold" style="font-size: 0.95rem;">
                                <i class="bi bi-calendar-check me-2"></i>Randevu Bilgileri
                            </h6>
                        </div>
                        <div class="card-body p-3">
                            <form id="newReservationForm" method="POST" action="{{ route('admin.appointments.store') }}">
                                @csrf
                                <input type="hidden" name="customer_id" id="form_customer_id">
                                
                                <!-- Tarih Seçimi -->
                                <div class="mb-3">
                                    <label for="appointment_date" class="form-label fw-bold" style="font-size: 0.85rem;">
                                        <i class="bi bi-calendar3 me-1"></i>Randevu Tarihi <span class="text-danger">*</span>
                                    </label>
                                    <input type="date" 
                                           class="form-control" 
                                           id="appointment_date" 
                                           name="date" 
                                           min="${today}" 
                                           value="${dateToUse}"
                                           required
                                           style="border-radius: 8px; border: 1px solid #e5e7eb; padding: 8px; font-size: 0.9rem;">
                                </div>
                                
                                <!-- Hizmet Seçimi -->
                                <div class="mb-3">
                                    <div class="d-flex justify-content-between align-items-center mb-2">
                                        <label class="form-label fw-bold mb-0" style="font-size: 0.85rem;">
                                            <i class="bi bi-scissors me-1"></i>Hizmet(ler) <span class="text-danger">*</span>
                                        </label>
                                        <button type="button" class="btn btn-sm btn-outline-primary" onclick="openCreateServiceModal()" style="border-radius: 8px; padding: 6px 12px; white-space: nowrap; font-size: 0.8rem;" title="Yeni Hizmet Ekle">
                                            <i class="bi bi-plus-circle me-1"></i>Yeni Ekle
                                        </button>
                                    </div>
                                    ${categoryTabsHtml}
                                    <div id="servicesCheckboxContainer" style="max-height: 300px; overflow-y: auto; border: 1px solid #e5e7eb;">
                                        ${servicesHtml || '<p class="text-muted text-center mb-0" style="font-size: 0.85rem; padding: 20px; grid-column: 1 / -1;">Henüz hizmet bulunmamaktadır.</p>'}
                                    </div>
                                    <small class="text-muted" style="font-size: 0.75rem;">İstediğiniz hizmet(ler)i seçmek için kartları tıklayın</small>
                                </div>
                                
                                <!-- Fiyat Seçimi -->
                                <div class="mb-3">
                                    <label for="modal_total_price" class="form-label fw-bold" style="font-size: 0.85rem;">
                                        <i class="bi bi-currency-exchange me-1"></i>Fiyat <span class="text-danger">*</span>
                                    </label>
                                    <input type="number" 
                                           class="form-control" 
                                           id="modal_total_price" 
                                           name="total_price" 
                                           step="0.01" 
                                           min="0" 
                                           required
                                           placeholder="Hizmet fiyatı otomatik doldurulacak"
                                           style="border-radius: 8px; border: 1px solid #e5e7eb; padding: 8px; font-size: 0.9rem;">
                                    <small class="text-muted" style="font-size: 0.75rem;">Hizmet seçildiğinde otomatik doldurulur, isterseniz değiştirebilirsiniz</small>
                                </div>
                                
                                <!-- Personel Seçimi -->
                                <div class="mb-3">
                                    <label for="modal_personel_id" class="form-label fw-bold" style="font-size: 0.85rem;">
                                        <i class="bi bi-person-badge me-1"></i>Personel <span class="text-danger">*</span>
                                    </label>
                                    <div class="d-flex gap-2">
                                        <select class="form-select" 
                                                id="modal_personel_id" 
                                                name="personel_id" 
                                                required
                                                style="border-radius: 8px; border: 1px solid #e5e7eb; padding: 8px; font-size: 0.9rem; flex: 1;">
                                            ${personelsHtml}
                                        </select>
                                        <button type="button" class="btn btn-sm btn-outline-primary" onclick="openCreatePersonelModal()" style="border-radius: 8px; padding: 8px 12px; white-space: nowrap;" title="Yeni Personel Ekle">
                                            <i class="bi bi-plus-circle"></i>
                                        </button>
                                    </div>
                                </div>
                                
                                <!-- Saat Seçimi -->
                                <div class="mb-3">
                                    <label class="form-label fw-bold" style="font-size: 0.85rem;">
                                        <i class="bi bi-clock me-1"></i>Randevu Saati <span class="text-danger">*</span>
                                    </label>
                                    <div class="row g-2">
                                        <div class="col-6">
                                            <select class="form-select" 
                                                    id="appointment_hour" 
                                                    required
                                                    style="border-radius: 8px; border: 1px solid #e5e7eb; padding: 8px; font-size: 0.9rem;">
                                                <option value="">Saat</option>
                                                ${Array.from({length: 15}, (_, i) => {
                                                    const hour = i + 8;
                                                    return `<option value="${String(hour).padStart(2, '0')}">${String(hour).padStart(2, '0')}</option>`;
                                                }).join('')}
                                            </select>
                                        </div>
                                        <div class="col-6">
                                            <select class="form-select" 
                                                    id="appointment_minute" 
                                                    required
                                                    style="border-radius: 8px; border: 1px solid #e5e7eb; padding: 8px; font-size: 0.9rem;">
                                                <option value="">Dakika</option>
                                                <option value="00">00</option>
                                                <option value="10">10</option>
                                                <option value="20">20</option>
                                                <option value="30">30</option>
                                                <option value="40">40</option>
                                                <option value="50">50</option>
                                            </select>
                                        </div>
                                    </div>
                                    <input type="hidden" id="appointment_time" name="start_time" required>
                                    <input type="hidden" id="appointment_end_time" name="end_time">
                                </div>
                                
                                <!-- Oda Seçimi -->
                                <div class="mb-3">
                                    <label for="modal_room_id" class="form-label fw-bold" style="font-size: 0.85rem;">
                                        <i class="bi bi-door-open me-1"></i>Oda <span class="text-muted">(Opsiyonel)</span>
                                    </label>
                                    <div class="d-flex gap-2">
                                        <select class="form-select" 
                                                id="modal_room_id" 
                                                name="room_id"
                                                style="border-radius: 8px; border: 1px solid #e5e7eb; padding: 8px; font-size: 0.9rem; flex: 1;">
                                            ${roomsHtml}
                                        </select>
                                        <button type="button" class="btn btn-sm btn-outline-primary" onclick="openCreateRoomModal()" style="border-radius: 8px; padding: 8px 12px; white-space: nowrap;" title="Yeni Oda Ekle">
                                            <i class="bi bi-plus-circle"></i>
                                        </button>
                                    </div>
                                </div>
                                
                                <!-- Not Kısmı -->
                                <div class="mb-3">
                                    <label for="notes" class="form-label fw-bold" style="font-size: 0.85rem;">
                                        <i class="bi bi-sticky me-1"></i>Not <span class="text-muted">(Opsiyonel)</span>
                                    </label>
                                    <textarea class="form-control" 
                                              id="notes" 
                                              name="notes" 
                                              rows="3"
                                              placeholder="Randevu ile ilgili notlar..."
                                              style="border-radius: 8px; border: 1px solid #e5e7eb; padding: 8px; font-size: 0.9rem;"></textarea>
                                </div>
                                
                                <!-- Butonlar -->
                                <div class="d-grid gap-2">
                                    <button type="submit" class="btn fw-bold" 
                                            style="background: linear-gradient(135deg, #ec4899 0%, #be185d 100%); color: white; border: none; border-radius: 8px; padding: 10px; font-size: 0.9rem;">
                                        <i class="bi bi-check-circle me-2"></i>Randevuyu Oluştur
                                    </button>
                                    <button type="button" class="btn btn-light fw-bold" 
                                            data-bs-dismiss="modal"
                                            style="border-radius: 8px; padding: 10px; font-size: 0.9rem;">
                                        <i class="bi bi-arrow-left me-2"></i>İptal
                                    </button>
                                </div>
                            </form>
                        </div>
                    </div>
                </div>
            </div>
        `;
        
        formContainer.innerHTML = formHtml;
        loading.style.display = 'none';
        formContainer.style.display = 'block';
        
        // Select elementlerinin doldurulduğunu kontrol et
        setTimeout(() => {
            const personelSelect = document.getElementById('modal_personel_id');
            const roomSelect = document.getElementById('modal_room_id');
            const serviceSelect = document.getElementById('modal_service_id');
            
            console.log('Personel select options:', personelSelect ? personelSelect.options.length : 'bulunamadı');
            console.log('Oda select options:', roomSelect ? roomSelect.options.length : 'bulunamadı');
            console.log('Hizmet select options:', serviceSelect ? serviceSelect.options.length : 'bulunamadı');
            
            if (personelSelect && personelSelect.options.length <= 1) {
                console.warn('Personel verileri yüklenemedi!');
            }
            if (roomSelect && roomSelect.options.length <= 1) {
                console.warn('Oda verileri yüklenemedi!');
            }
            if (serviceSelect && serviceSelect.options.length <= 1) {
                console.warn('Hizmet verileri yüklenemedi!');
            }
        }, 100);
        
        // Müşteri seçimi değiştiğinde form'a aktar ve bilgilerini yükle
        const customerSelect = document.getElementById('modal_customer_id');
        const formCustomerId = document.getElementById('form_customer_id');
        const customerInfoContainer = document.getElementById('customerInfoContainer');
        
        if (customerSelect && formCustomerId) {
            customerSelect.addEventListener('change', function() {
                const customerId = this.value;
                formCustomerId.value = customerId;
                
                if (customerId) {
                    // Müşteri bilgilerini yükle
                    fetch('{{ route("admin.beauty-salon.appointment-requests.customer-info", ["customerId" => ":id"]) }}'.replace(':id', customerId), {
                        headers: {
                            'X-Requested-With': 'XMLHttpRequest',
                            'Accept': 'application/json'
                        }
                    })
                    .then(r => r.json())
                    .then(data => {
                        if (data.customer) {
                            const customer = data.customer;
                            const stats = data.statistics || {};
                            const reservations = data.reservations || [];
                            
                            let customerInfoHtml = `
                                <hr class="my-3" style="border-color: #e5e7eb;">
                                <div class="mb-2">
                                    <small class="text-muted d-block mb-1" style="font-size: 0.75rem; font-weight: 600;">Ad Soyad</small>
                                    <p class="mb-2 fw-bold" style="font-size: 0.9rem; color: #1f2937;">${customer.name} ${customer.surname || ''}</p>
                                </div>
                                ${customer.phone ? `
                                <div class="mb-2">
                                    <small class="text-muted d-block mb-1" style="font-size: 0.75rem; font-weight: 600;">Telefon</small>
                                    <p class="mb-2" style="font-size: 0.9rem;">
                                        <a href="tel:${customer.phone}" class="text-decoration-none" style="color: #ec4899;">${customer.phone}</a>
                                    </p>
                                </div>
                                ` : ''}
                                ${customer.age ? `
                                <div class="mb-2">
                                    <small class="text-muted d-block mb-1" style="font-size: 0.75rem; font-weight: 600;">Yaş</small>
                                    <p class="mb-2" style="font-size: 0.9rem; color: #1f2937;">${customer.age}</p>
                                </div>
                                ` : ''}
                                <hr class="my-2" style="border-color: #e5e7eb;">
                                <div class="row text-center mb-3">
                                    <div class="col-4">
                                        <small class="text-muted d-block" style="font-size: 0.7rem;">Toplam</small>
                                        <strong style="font-size: 1rem; color: #8b5cf6;">${stats.total || 0}</strong>
                                    </div>
                                    <div class="col-4">
                                        <small class="text-muted d-block" style="font-size: 0.7rem;">Tamamlanan</small>
                                        <strong style="font-size: 1rem; color: #10b981;">${stats.completed || 0}</strong>
                                    </div>
                                    <div class="col-4">
                                        <small class="text-muted d-block" style="font-size: 0.7rem;">Toplam Harcama</small>
                                        <strong style="font-size: 0.9rem; color: #ec4899;">${parseFloat(stats.total_spent || 0).toLocaleString('tr-TR', {minimumFractionDigits: 2, maximumFractionDigits: 2})} ₺</strong>
                                    </div>
                                </div>
                            `;
                            
                            if (reservations.length > 0) {
                                customerInfoHtml += `
                                    <div class="mb-2">
                                        <small class="text-muted d-block mb-2" style="font-size: 0.75rem; font-weight: 600;">
                                            <i class="bi bi-clock-history me-1"></i>Son Randevular
                                        </small>
                                        <div style="max-height: 150px; overflow-y: auto;">
                                            ${reservations.map(res => {
                                                const date = new Date(res.date);
                                                const formattedDate = date.toLocaleDateString('tr-TR', { day: '2-digit', month: '2-digit', year: 'numeric' });
                                                let statusBadge = '';
                                                if (res.status === 'completed') {
                                                    statusBadge = '<span class="badge bg-success" style="font-size: 0.65rem;">Tamamlandı</span>';
                                                } else if (res.status === 'confirmed') {
                                                    statusBadge = '<span class="badge bg-primary" style="font-size: 0.65rem;">Onaylandı</span>';
                                                } else if (res.status === 'cancelled') {
                                                    statusBadge = '<span class="badge bg-danger" style="font-size: 0.65rem;">İptal</span>';
                                                } else {
                                                    statusBadge = '<span class="badge bg-warning" style="font-size: 0.65rem;">Beklemede</span>';
                                                }
                                                return `
                                                    <div class="reservation-item mb-2 p-2 rounded" style="background: #f8fafc; border-left: 3px solid #ec4899; font-size: 0.8rem;">
                                                        <div class="d-flex justify-content-between align-items-start">
                                                            <div>
                                                                <strong class="d-block">${formattedDate} ${res.start_time}</strong>
                                                                <small class="text-muted">
                                                                    ${res.personel_name ? '<i class="bi bi-person me-1"></i>' + res.personel_name : ''}
                                                                    ${res.service_name ? '<span class="ms-2"><i class="bi bi-scissors me-1"></i>' + res.service_name + '</span>' : ''}
                                                                </small>
                                                            </div>
                                                            <div class="text-end">
                                                                ${statusBadge}
                                                                <div class="mt-1">
                                                                    <small class="text-success fw-bold">${parseFloat(res.total_price || 0).toLocaleString('tr-TR', {minimumFractionDigits: 2, maximumFractionDigits: 2})} ₺</small>
                                                                </div>
                                                            </div>
                                                        </div>
                                                    </div>
                                                `;
                                            }).join('')}
                                        </div>
                                    </div>
                                `;
                            }
                            
                            customerInfoContainer.innerHTML = customerInfoHtml;
                            customerInfoContainer.style.display = 'block';
                        }
                    })
                    .catch(error => {
                        console.error('Müşteri bilgileri yüklenemedi:', error);
                        customerInfoContainer.style.display = 'none';
                    });
                } else {
                    customerInfoContainer.style.display = 'none';
                    customerInfoContainer.innerHTML = '';
                }
            });
        }
        
        // Saat ve dakika birleştirme
        const hourSelect = document.getElementById('appointment_hour');
        const minuteSelect = document.getElementById('appointment_minute');
        const timeInput = document.getElementById('appointment_time');
        const endTimeInput = document.getElementById('appointment_end_time');
        
        function updateTime() {
            if (hourSelect && minuteSelect && timeInput && endTimeInput) {
                const hour = hourSelect.value;
                const minute = minuteSelect.value;
                if (hour && minute) {
                    timeInput.value = hour + ':' + minute + ':00';
                    
                    // Seçilen hizmetlerin toplam süresine göre bitiş saatini hesapla
                    let totalDuration = 60; // Varsayılan 60 dakika
                    const checkedServices = document.querySelectorAll('.service-checkbox:checked');
                    if (checkedServices.length > 0) {
                        totalDuration = 0;
                        checkedServices.forEach(checkbox => {
                            const duration = parseInt(checkbox.getAttribute('data-duration') || 30);
                            totalDuration += duration;
                        });
                    }
                    
                    // Başlangıç saatini dakikaya çevir
                    const startMinutes = parseInt(hour) * 60 + parseInt(minute);
                    const endMinutes = startMinutes + totalDuration;
                    
                    // Bitiş saatini hesapla
                    const endHour = Math.floor(endMinutes / 60);
                    const endMin = endMinutes % 60;
                    
                    endTimeInput.value = String(endHour).padStart(2, '0') + ':' + String(endMin).padStart(2, '0') + ':00';
                }
            }
        }
        
        // Hizmet seçildiğinde fiyat alanını doldur (birden fazla hizmet için toplam fiyat)
        function updateServicePrice() {
            let totalPrice = 0;
            const checkedServices = document.querySelectorAll('.service-checkbox:checked');
            checkedServices.forEach(checkbox => {
                const price = parseFloat(checkbox.getAttribute('data-price') || 0);
                totalPrice += price;
                
                // CSS :checked selector ile otomatik yönetiliyor
            });
            
            // CSS :checked selector ile otomatik yönetiliyor
            
            const priceInput = document.getElementById('modal_total_price');
            if (priceInput) {
                priceInput.value = totalPrice > 0 ? totalPrice.toFixed(2) : '';
            }
            updateTime();
        }
        
        // Tüm checkbox'lara event listener ekle
        setTimeout(() => {
            const serviceCheckboxes = document.querySelectorAll('.service-checkbox');
            serviceCheckboxes.forEach(checkbox => {
                checkbox.addEventListener('change', updateServicePrice);
                // CSS :checked selector ile otomatik yönetiliyor
            });
        }, 100);
        
        if (hourSelect) hourSelect.addEventListener('change', updateTime);
        if (minuteSelect) minuteSelect.addEventListener('change', updateTime);
        
        // Form submit
        const form = document.getElementById('newReservationForm');
        if (form) {
            form.addEventListener('submit', function(e) {
                const customerId = document.getElementById('modal_customer_id')?.value;
                if (!customerId) {
                    e.preventDefault();
                    alert('Lütfen bir müşteri seçiniz.');
                    return false;
                }
                
                // Checkbox'lardan seçilen hizmetleri kontrol et
                const checkedServices = document.querySelectorAll('.service-checkbox:checked');
                const selectedServices = Array.from(checkedServices).map(checkbox => checkbox.value);
                if (!selectedServices || selectedServices.length === 0) {
                    e.preventDefault();
                    alert('Lütfen en az bir hizmet seçiniz.');
                    return false;
                }
                
                const personelId = document.getElementById('modal_personel_id')?.value;
                if (!personelId) {
                    e.preventDefault();
                    alert('Lütfen bir personel seçiniz.');
                    return false;
                }
                
                const hour = document.getElementById('appointment_hour')?.value;
                const minute = document.getElementById('appointment_minute')?.value;
                if (!hour || !minute) {
                    e.preventDefault();
                    alert('Lütfen saat ve dakika seçiniz.');
                    return false;
                }
                
                // Zamanı güncelle
                updateTime();
                
                // End time kontrolü
                if (!endTimeInput.value) {
                    e.preventDefault();
                    alert('Bitiş saati hesaplanamadı. Lütfen tekrar deneyin.');
                    return false;
                }
            });
        }
    })
    .catch(error => {
        console.error('Veriler yüklenemedi:', error);
        loading.innerHTML = '<div class="alert alert-danger">Veriler yüklenirken bir hata oluştu. Lütfen sayfayı yenileyin.</div>';
    });
    
    // Modal kapatıldığında formu sıfırla
    modal.addEventListener('hidden.bs.modal', function() {
        loading.style.display = 'block';
        formContainer.style.display = 'none';
        formContainer.innerHTML = '';
    }, { once: true });
}

// Kategoriye göre hizmetleri filtrele
function filterServicesByCategory(categoryId) {
    const container = document.getElementById('servicesCheckboxContainer');
    const tabs = document.querySelectorAll('.service-category-tab');
    
    if (!container || !tabs.length) return;
    
    // Tüm sekmelerden active class'ını kaldır
    tabs.forEach(tab => {
        tab.classList.remove('active');
        tab.style.background = '#f3f4f6';
        tab.style.color = '#000000';
        tab.style.boxShadow = 'none';
        const tabCategoryId = tab.getAttribute('data-category-id');
        if (tabCategoryId === String(categoryId)) {
            // Aktif sekme için stilleri uygula
            tab.classList.add('active');
            tab.style.background = 'linear-gradient(135deg, #ec4899 0%, #be185d 100%)';
            tab.style.color = 'white';
            tab.style.boxShadow = '0 2px 8px rgba(236, 72, 153, 0.3)';
        }
    });
    
    // Hizmetleri filtrele
    if (window.generateServicesHtmlFunction) {
        const filteredHtml = window.generateServicesHtmlFunction(categoryId);
        container.innerHTML = filteredHtml;
        
        // Seçili checkbox'ları koru
        const checkedIds = Array.from(document.querySelectorAll('.service-checkbox:checked')).map(cb => cb.value);
        checkedIds.forEach(serviceId => {
            const card = container.querySelector(`[data-service-id="${serviceId}"]`);
            if (card) {
                const cardCheckbox = card.querySelector('.service-checkbox');
                if (cardCheckbox) {
                    cardCheckbox.checked = true;
                }
            }
        });
        
        // UpdateServicePrice fonksiyonunu çağır
        if (typeof updateServicePrice === 'function') {
            updateServicePrice();
        }
    } else {
        // Fallback: DOM ile filtreleme
        const allCards = container.querySelectorAll('.service-card-wrapper');
        allCards.forEach(card => {
            const serviceCategoryId = card.getAttribute('data-service-category-id');
            if (categoryId === 'all' || serviceCategoryId == categoryId) {
                card.style.display = 'block';
            } else {
                card.style.display = 'none';
            }
        });
    }
}

// Yeni Müşteri Oluşturma Modal
function openCreateCustomerModal() {
    const modalHtml = `
        <div class="modal fade" id="createCustomerModal" tabindex="-1">
            <div class="modal-dialog modal-dialog-centered modal-lg">
                <div class="modal-content">
                    <div class="modal-header" style="background: linear-gradient(135deg, #ec4899 0%, #be185d 100%);">
                        <h5 class="modal-title text-white fw-bold">
                            <i class="bi bi-person-plus me-2"></i>Yeni Müşteri Ekle
                        </h5>
                        <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                    </div>
                    <form id="createCustomerForm">
                        <div class="modal-body p-4">
                            <div class="row">
                                <div class="col-md-6 mb-3">
                                    <label class="form-label fw-bold">
                                        <i class="bi bi-person me-1"></i>Ad Soyad <span class="text-danger">*</span>
                                    </label>
                                    <input type="text" name="name" class="form-control form-control-lg" required placeholder="Müşteri adı soyadı">
                                </div>
                                <div class="col-md-6 mb-3">
                                    <label class="form-label fw-bold">
                                        <i class="bi bi-telephone me-1"></i>Telefon <span class="text-danger">*</span>
                                    </label>
                                    <input type="tel" name="phone" class="form-control form-control-lg" required placeholder="05XX XXX XX XX">
                                </div>
                            </div>
                            <div class="row">
                                <div class="col-md-6 mb-3">
                                    <label class="form-label fw-bold">
                                        <i class="bi bi-envelope me-1"></i>E-posta <small class="text-muted">(Opsiyonel)</small>
                                    </label>
                                    <input type="email" name="email" class="form-control form-control-lg" placeholder="ornek@email.com">
                                </div>
                                <div class="col-md-6 mb-3">
                                    <label class="form-label fw-bold">
                                        <i class="bi bi-geo-alt me-1"></i>Adres
                                    </label>
                                    <textarea name="address" class="form-control" rows="2" placeholder="Müşteri adresi"></textarea>
                                </div>
                            </div>
                            <div class="mb-3">
                                <label class="form-label fw-bold">
                                    <i class="bi bi-sticky me-1"></i>Notlar
                                </label>
                                <textarea name="notes" class="form-control" rows="3" placeholder="Müşteri hakkında notlar..."></textarea>
                            </div>
                            <div id="createCustomerErrors" class="alert alert-danger" style="display: none;"></div>
                        </div>
                        <div class="modal-footer">
                            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
                                <i class="bi bi-x me-2"></i>İptal
                            </button>
                            <button type="submit" class="btn btn-primary">
                                <i class="bi bi-save me-2"></i>Kaydet
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    `;
    
    // Eski modal varsa kaldır
    const oldModal = document.getElementById('createCustomerModal');
    if (oldModal) oldModal.remove();
    
    // Yeni modal ekle
    document.body.insertAdjacentHTML('beforeend', modalHtml);
    const modal = new bootstrap.Modal(document.getElementById('createCustomerModal'));
    modal.show();
    
    // Form submit
    document.getElementById('createCustomerForm').addEventListener('submit', async function(e) {
        e.preventDefault();
        const form = this;
        const submitBtn = form.querySelector('button[type="submit"]');
        const errorsDiv = document.getElementById('createCustomerErrors');
        
        submitBtn.disabled = true;
        submitBtn.textContent = 'Kaydediliyor...';
        errorsDiv.style.display = 'none';
        
        const formData = new FormData(form);
        formData.append('_token', '{{ csrf_token() }}');
        
        try {
            const response = await fetch('{{ route("admin.customers.store") }}', {
                method: 'POST',
                headers: {
                    'X-Requested-With': 'XMLHttpRequest',
                    'Accept': 'application/json'
                },
                body: formData
            });
            
            const data = await response.json();
            
            if (data.success && data.customer) {
                // Müşteri seçimine ekle
                const customerSelect = document.getElementById('modal_customer_id');
                if (customerSelect) {
                    const option = document.createElement('option');
                    option.value = data.customer.id;
                    option.textContent = data.customer.name + (data.customer.phone ? ' (' + data.customer.phone + ')' : '');
                    option.selected = true;
                    customerSelect.appendChild(option);
                    
                    // Form'a da aktar
                    const formCustomerId = document.getElementById('form_customer_id');
                    if (formCustomerId) formCustomerId.value = data.customer.id;
                    
                    // Müşteri bilgilerini yükle
                    customerSelect.dispatchEvent(new Event('change'));
                }
                
                modal.hide();
                setTimeout(() => document.getElementById('createCustomerModal')?.remove(), 300);
            } else {
                let errorHtml = '<ul class="mb-0">';
                if (data.errors) {
                    Object.values(data.errors).forEach(error => {
                        if (Array.isArray(error)) {
                            error.forEach(msg => errorHtml += '<li>' + msg + '</li>');
                        } else {
                            errorHtml += '<li>' + error + '</li>';
                        }
                    });
                } else if (data.message) {
                    errorHtml += '<li>' + data.message + '</li>';
                }
                errorHtml += '</ul>';
                errorsDiv.innerHTML = errorHtml;
                errorsDiv.style.display = 'block';
                submitBtn.disabled = false;
                submitBtn.textContent = 'Kaydet';
            }
        } catch (error) {
            console.error('Error:', error);
            errorsDiv.innerHTML = '<ul class="mb-0"><li>Bir hata oluştu. Lütfen tekrar deneyin.</li></ul>';
            errorsDiv.style.display = 'block';
            submitBtn.disabled = false;
            submitBtn.textContent = 'Kaydet';
        }
    });
}

// Yeni Hizmet Oluşturma Modal
function openCreateServiceModal() {
    const modalHtml = `
        <div class="modal fade" id="createServiceModal" tabindex="-1">
            <div class="modal-dialog modal-dialog-centered modal-lg">
                <div class="modal-content">
                    <div class="modal-header" style="background: linear-gradient(135deg, #ec4899 0%, #be185d 100%);">
                        <h5 class="modal-title text-white fw-bold">
                            <i class="bi bi-scissors me-2"></i>Yeni Hizmet Ekle
                        </h5>
                        <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                    </div>
                    <form id="createServiceForm">
                        <div class="modal-body p-4">
                            <div class="mb-3">
                                <label class="form-label fw-bold">
                                    <i class="bi bi-folder me-1"></i>Kategori <span class="text-danger">*</span>
                                </label>
                                <select name="service_category_id" class="form-select form-select-lg" required>
                                    <option value="">Kategori Seçin</option>
                                </select>
                            </div>
                            <div class="mb-3">
                                <label class="form-label fw-bold">
                                    <i class="bi bi-tag me-1"></i>Hizmet Adı <span class="text-danger">*</span>
                                </label>
                                <input type="text" name="name" class="form-control form-control-lg" required placeholder="Örn: Saç Yıkama Paketi">
                            </div>
                            <div class="mb-3">
                                <label class="form-label fw-bold">
                                    <i class="bi bi-file-text me-1"></i>Açıklama
                                </label>
                                <textarea name="description" class="form-control" rows="3" placeholder="Hizmet hakkında detaylı bilgi..."></textarea>
                            </div>
                            <div class="mb-3">
                                <label class="form-label fw-bold">
                                    <i class="bi bi-currency-exchange me-1"></i>Fiyat (₺) <span class="text-danger">*</span>
                                </label>
                                <input type="number" name="price" class="form-control form-control-lg" step="0.01" min="0" required placeholder="0.00">
                            </div>
                            <div id="createServiceErrors" class="alert alert-danger" style="display: none;"></div>
                        </div>
                        <div class="modal-footer">
                            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
                                <i class="bi bi-x me-2"></i>İptal
                            </button>
                            <button type="submit" class="btn btn-primary">
                                <i class="bi bi-save me-2"></i>Kaydet
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    `;
    
    const oldModal = document.getElementById('createServiceModal');
    if (oldModal) oldModal.remove();
    
    document.body.insertAdjacentHTML('beforeend', modalHtml);
    const modalElement = document.getElementById('createServiceModal');
    const modal = new bootstrap.Modal(modalElement);
    
    // Kategorileri yükle
    fetch('{{ route("admin.beauty-salon.appointment-requests.data") }}', {
        headers: {
            'X-Requested-With': 'XMLHttpRequest',
            'Accept': 'application/json'
        }
    })
    .then(r => r.json())
    .then(data => {
        const categories = data.categories || [];
        const categorySelect = modalElement.querySelector('select[name="service_category_id"]');
        if (categorySelect && categories.length > 0) {
            categories.forEach(cat => {
                const option = document.createElement('option');
                option.value = cat.id;
                option.textContent = cat.name;
                categorySelect.appendChild(option);
            });
        }
    })
    .catch(err => console.error('Kategoriler yüklenemedi:', err));
    
    modal.show();
    
    document.getElementById('createServiceForm').addEventListener('submit', async function(e) {
        e.preventDefault();
        const form = this;
        const submitBtn = form.querySelector('button[type="submit"]');
        const errorsDiv = document.getElementById('createServiceErrors');
        
        submitBtn.disabled = true;
        submitBtn.textContent = 'Kaydediliyor...';
        errorsDiv.style.display = 'none';
        
        const formData = new FormData(form);
        formData.append('_token', '{{ csrf_token() }}');
        
        try {
            const response = await fetch('{{ route("admin.beauty-salon.services.store") }}', {
                method: 'POST',
                headers: {
                    'X-Requested-With': 'XMLHttpRequest',
                    'Accept': 'application/json'
                },
                body: formData
            });
            
            const data = await response.json();
            
            if (data.success && data.service) {
                const servicesContainer = document.getElementById('servicesCheckboxContainer');
                if (servicesContainer) {
                    // Yeni checkbox oluştur
                    const price = parseFloat(data.service.price || 0).toLocaleString('tr-TR', {minimumFractionDigits: 2, maximumFractionDigits: 2});
                    const duration = data.service.duration || 30;
                    const serviceId = data.service.id;
                    const serviceName = data.service.name;
                    const serviceDesc = data.service.description || '';
                    
                    const checkboxHtml = `
                        <div class="service-card-wrapper" data-service-id="${serviceId}">
                            <input class="service-checkbox" 
                                   type="checkbox" 
                                   name="service_id[]" 
                                   value="${serviceId}" 
                                   id="service_${serviceId}" 
                                   data-price="${data.service.price || 0}" 
                                   data-duration="${duration}"
                                   checked
                                   onchange="updateServicePrice()">
                            <label class="service-card" for="service_${serviceId}">
                                <div class="service-card-header">
                                    <div class="service-card-icon">
                                        <i class="bi bi-scissors"></i>
                                    </div>
                                    <div class="service-card-content">
                                        <h6 class="service-card-title">${serviceName}</h6>
                                        ${serviceDesc ? `<p class="service-card-description">${serviceDesc.substring(0, 60)}${serviceDesc.length > 60 ? '...' : ''}</p>` : ''}
                                    </div>
                                    <div class="service-card-check">
                                        <i class="bi bi-check-circle-fill"></i>
                                    </div>
                                </div>
                                <div class="service-card-footer">
                                    <div class="service-card-info">
                                        ${duration ? `<span class="service-card-badge"><i class="bi bi-clock"></i> ${duration} dk</span>` : ''}
                                    </div>
                                    <div class="service-card-price">
                                        <span class="service-card-price-amount">${price} ₺</span>
                                    </div>
                                </div>
                            </label>
                        </div>
                    `;
                    
                    // Container'ın başına ekle
                    if (servicesContainer.innerHTML.includes('Henüz hizmet bulunmamaktadır')) {
                        servicesContainer.innerHTML = checkboxHtml;
                    } else {
                        servicesContainer.insertAdjacentHTML('afterbegin', checkboxHtml);
                    }
                    
                    // Fiyat alanını güncelle
                    if (typeof updateServicePrice === 'function') {
                        updateServicePrice();
                    }
                    
                    // updateTime fonksiyonunu manuel olarak da çağır (eğer tanımlıysa)
                    if (typeof updateTime === 'function') {
                        updateTime();
                    }
                }
                
                modal.hide();
                setTimeout(() => document.getElementById('createServiceModal')?.remove(), 300);
            } else {
                let errorHtml = '<ul class="mb-0">';
                if (data.errors) {
                    Object.values(data.errors).forEach(error => {
                        if (Array.isArray(error)) {
                            error.forEach(msg => errorHtml += '<li>' + msg + '</li>');
                        } else {
                            errorHtml += '<li>' + error + '</li>';
                        }
                    });
                } else if (data.message) {
                    errorHtml += '<li>' + data.message + '</li>';
                }
                errorHtml += '</ul>';
                errorsDiv.innerHTML = errorHtml;
                errorsDiv.style.display = 'block';
                submitBtn.disabled = false;
                submitBtn.textContent = 'Kaydet';
            }
        } catch (error) {
            console.error('Error:', error);
            errorsDiv.innerHTML = '<ul class="mb-0"><li>Bir hata oluştu. Lütfen tekrar deneyin.</li></ul>';
            errorsDiv.style.display = 'block';
            submitBtn.disabled = false;
            submitBtn.textContent = 'Kaydet';
        }
    });
}

// Yeni Personel Oluşturma Modal
function openCreatePersonelModal() {
    const modalHtml = `
        <div class="modal fade" id="createPersonelModal" tabindex="-1">
            <div class="modal-dialog modal-dialog-centered modal-lg">
                <div class="modal-content">
                    <div class="modal-header" style="background: linear-gradient(135deg, #ec4899 0%, #be185d 100%);">
                        <h5 class="modal-title text-white fw-bold">
                            <i class="bi bi-person-badge me-2"></i>Yeni Personel Ekle
                        </h5>
                        <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                    </div>
                    <form id="createPersonelForm">
                        <div class="modal-body p-4">
                            <div class="row">
                                <div class="col-md-6 mb-3">
                                    <label class="form-label fw-bold">
                                        <i class="bi bi-person me-1"></i>Ad Soyad <span class="text-danger">*</span>
                                    </label>
                                    <input type="text" name="name" class="form-control form-control-lg" required placeholder="Personel adı soyadı">
                                </div>
                                <div class="col-md-6 mb-3">
                                    <label class="form-label fw-bold">
                                        <i class="bi bi-envelope me-1"></i>E-posta <span class="text-danger">*</span>
                                    </label>
                                    <input type="email" name="email" class="form-control form-control-lg" required placeholder="ornek@email.com">
                                    <small class="text-muted">Personel girişi için kullanılacak</small>
                                </div>
                            </div>
                            <div class="row">
                                <div class="col-md-6 mb-3">
                                    <label class="form-label fw-bold">
                                        <i class="bi bi-telephone me-1"></i>Telefon
                                    </label>
                                    <input type="tel" name="phone" class="form-control form-control-lg" placeholder="05XX XXX XX XX">
                                </div>
                                <div class="col-md-6 mb-3">
                                    <label class="form-label fw-bold">
                                        <i class="bi bi-lock me-1"></i>Şifre
                                    </label>
                                    <input type="password" name="password" class="form-control form-control-lg" placeholder="Boş bırakılırsa otomatik oluşturulur">
                                    <small class="text-muted">Boş bırakılırsa varsayılan şifre: personel123</small>
                                </div>
                            </div>
                            <div id="createPersonelErrors" class="alert alert-danger" style="display: none;"></div>
                        </div>
                        <div class="modal-footer">
                            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
                                <i class="bi bi-x me-2"></i>İptal
                            </button>
                            <button type="submit" class="btn btn-primary">
                                <i class="bi bi-save me-2"></i>Kaydet
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    `;
    
    const oldModal = document.getElementById('createPersonelModal');
    if (oldModal) oldModal.remove();
    
    document.body.insertAdjacentHTML('beforeend', modalHtml);
    const modal = new bootstrap.Modal(document.getElementById('createPersonelModal'));
    modal.show();
    
    document.getElementById('createPersonelForm').addEventListener('submit', async function(e) {
        e.preventDefault();
        const form = this;
        const submitBtn = form.querySelector('button[type="submit"]');
        const errorsDiv = document.getElementById('createPersonelErrors');
        
        submitBtn.disabled = true;
        submitBtn.textContent = 'Kaydediliyor...';
        errorsDiv.style.display = 'none';
        
        const formData = new FormData(form);
        formData.append('_token', '{{ csrf_token() }}');
        
        try {
            const response = await fetch('{{ route("admin.beauty-salon.personel.store") }}', {
                method: 'POST',
                headers: {
                    'X-Requested-With': 'XMLHttpRequest',
                    'Accept': 'application/json'
                },
                body: formData
            });
            
            const data = await response.json();
            
            if (data.success && data.personel) {
                const personelSelect = document.getElementById('modal_personel_id');
                if (personelSelect) {
                    const option = document.createElement('option');
                    option.value = data.personel.id;
                    option.textContent = data.personel.name;
                    option.selected = true;
                    personelSelect.appendChild(option);
                }
                
                modal.hide();
                setTimeout(() => document.getElementById('createPersonelModal')?.remove(), 300);
            } else {
                let errorHtml = '<ul class="mb-0">';
                if (data.errors) {
                    Object.values(data.errors).forEach(error => {
                        if (Array.isArray(error)) {
                            error.forEach(msg => errorHtml += '<li>' + msg + '</li>');
                        } else {
                            errorHtml += '<li>' + error + '</li>';
                        }
                    });
                } else if (data.message) {
                    errorHtml += '<li>' + data.message + '</li>';
                }
                errorHtml += '</ul>';
                errorsDiv.innerHTML = errorHtml;
                errorsDiv.style.display = 'block';
                submitBtn.disabled = false;
                submitBtn.textContent = 'Kaydet';
            }
        } catch (error) {
            console.error('Error:', error);
            errorsDiv.innerHTML = '<ul class="mb-0"><li>Bir hata oluştu. Lütfen tekrar deneyin.</li></ul>';
            errorsDiv.style.display = 'block';
            submitBtn.disabled = false;
            submitBtn.textContent = 'Kaydet';
        }
    });
}

// Yeni Oda Oluşturma Modal
function openCreateRoomModal() {
    const modalHtml = `
        <div class="modal fade" id="createRoomModal" tabindex="-1">
            <div class="modal-dialog modal-dialog-centered modal-lg">
                <div class="modal-content">
                    <div class="modal-header" style="background: linear-gradient(135deg, #ec4899 0%, #be185d 100%);">
                        <h5 class="modal-title text-white fw-bold">
                            <i class="bi bi-door-open me-2"></i>Yeni Oda Ekle
                        </h5>
                        <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                    </div>
                    <form id="createRoomForm">
                        <div class="modal-body p-4">
                            <div class="row">
                                <div class="col-md-6 mb-3">
                                    <label class="form-label fw-bold">
                                        <i class="bi bi-tag me-1"></i>Oda Adı <span class="text-danger">*</span>
                                    </label>
                                    <input type="text" name="name" class="form-control form-control-lg" required placeholder="Örn: Oda 1, VIP Salon">
                                </div>
                                <div class="col-md-6 mb-3">
                                    <label class="form-label fw-bold">
                                        <i class="bi bi-people me-1"></i>Kapasite
                                    </label>
                                    <input type="number" name="capacity" class="form-control form-control-lg" min="1" placeholder="Kişi sayısı">
                                </div>
                            </div>
                            <div class="mb-3">
                                <label class="form-label fw-bold">
                                    <i class="bi bi-file-text me-1"></i>Açıklama
                                </label>
                                <textarea name="description" class="form-control" rows="3" placeholder="Oda hakkında detaylı bilgi..."></textarea>
                            </div>
                            <div class="mb-3">
                                <label class="form-label fw-bold">
                                    <i class="bi bi-palette me-1"></i>Renk (Takvimde görünecek)
                                </label>
                                <input type="color" name="color" class="form-control form-control-color" value="#10b981" title="Renk seçin" style="width: 100%; height: 50px;">
                            </div>
                            <div id="createRoomErrors" class="alert alert-danger" style="display: none;"></div>
                        </div>
                        <div class="modal-footer">
                            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
                                <i class="bi bi-x me-2"></i>İptal
                            </button>
                            <button type="submit" class="btn btn-primary">
                                <i class="bi bi-save me-2"></i>Kaydet
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    `;
    
    const oldModal = document.getElementById('createRoomModal');
    if (oldModal) oldModal.remove();
    
    document.body.insertAdjacentHTML('beforeend', modalHtml);
    const modal = new bootstrap.Modal(document.getElementById('createRoomModal'));
    modal.show();
    
    document.getElementById('createRoomForm').addEventListener('submit', async function(e) {
        e.preventDefault();
        const form = this;
        const submitBtn = form.querySelector('button[type="submit"]');
        const errorsDiv = document.getElementById('createRoomErrors');
        
        submitBtn.disabled = true;
        submitBtn.textContent = 'Kaydediliyor...';
        errorsDiv.style.display = 'none';
        
        const formData = new FormData(form);
        formData.append('_token', '{{ csrf_token() }}');
        
        try {
            const response = await fetch('{{ route("admin.beauty-salon.rooms.store") }}', {
                method: 'POST',
                headers: {
                    'X-Requested-With': 'XMLHttpRequest',
                    'Accept': 'application/json'
                },
                body: formData
            });
            
            const data = await response.json();
            
            if (data.success && data.room) {
                const roomSelect = document.getElementById('modal_room_id');
                if (roomSelect) {
                    const option = document.createElement('option');
                    option.value = data.room.id;
                    option.textContent = data.room.name;
                    option.selected = true;
                    roomSelect.appendChild(option);
                }
                
                modal.hide();
                setTimeout(() => document.getElementById('createRoomModal')?.remove(), 300);
            } else {
                let errorHtml = '<ul class="mb-0">';
                if (data.errors) {
                    Object.values(data.errors).forEach(error => {
                        if (Array.isArray(error)) {
                            error.forEach(msg => errorHtml += '<li>' + msg + '</li>');
                        } else {
                            errorHtml += '<li>' + error + '</li>';
                        }
                    });
                } else if (data.message) {
                    errorHtml += '<li>' + data.message + '</li>';
                }
                errorHtml += '</ul>';
                errorsDiv.innerHTML = errorHtml;
                errorsDiv.style.display = 'block';
                submitBtn.disabled = false;
                submitBtn.textContent = 'Kaydet';
            }
        } catch (error) {
            console.error('Error:', error);
            errorsDiv.innerHTML = '<ul class="mb-0"><li>Bir hata oluştu. Lütfen tekrar deneyin.</li></ul>';
            errorsDiv.style.display = 'block';
            submitBtn.disabled = false;
            submitBtn.textContent = 'Kaydet';
        }
    });
}

// Format Price Helper
function formatPrice(price) {
    return parseFloat(price || 0).toLocaleString('tr-TR', { minimumFractionDigits: 2, maximumFractionDigits: 2 });
}

// Hızlı Ödeme Alma
async function quickPayment(reservationId, totalPrice, depositPaid) {
    const remaining = parseFloat(totalPrice) - parseFloat(depositPaid || 0);
    
    if (remaining <= 0) {
        if (typeof showSuccessMessage === 'function') {
            showSuccessMessage('Bu randevu için ödenecek tutar kalmamış.');
        } else {
            alert('Bu randevu için ödenecek tutar kalmamış.');
        }
        return;
    }
    
    const paymentAmount = await showPaymentModal(remaining, totalPrice, depositPaid);
    if (!paymentAmount || paymentAmount <= 0) return;
    
    // Ödeme işlemini yap
    const url = '{{ route("admin.appointments.update-status", ["id" => ":id"]) }}'.replace(':id', reservationId);
    
    fetch(url, {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.content || '',
            'X-Requested-With': 'XMLHttpRequest',
            'Accept': 'application/json'
        },
        body: JSON.stringify({
            status: 'confirmed',
            mark_paid: true,
            payment_amount: paymentAmount
        })
    })
    .then(async response => {
        if (!response.ok) {
            const err = await response.json().catch(() => ({}));
            throw new Error(err.message || 'Ödeme alınamadı');
        }
        return response.json();
    })
    .then(data => {
        if (data.success) {
            if (typeof showSuccessMessage === 'function') {
                showSuccessMessage('Ödeme başarıyla kaydedildi!');
            } else {
                alert('Ödeme başarıyla kaydedildi!');
            }
            setTimeout(() => location.reload(), 1000);
        } else {
            if (typeof showErrorMessage === 'function') {
                showErrorMessage(data.message || 'Ödeme kaydedilemedi');
            } else {
                alert(data.message || 'Ödeme kaydedilemedi');
            }
        }
    })
    .catch(error => {
        console.error('Ödeme hatası:', error);
        if (typeof showErrorMessage === 'function') {
            showErrorMessage(error.message || 'Ödeme işlemi sırasında bir hata oluştu');
        } else {
            alert(error.message || 'Ödeme işlemi sırasında bir hata oluştu');
        }
    });
}

// Ödeme Modal
function showPaymentModal(remaining, totalPrice, depositPaid) {
    return new Promise((resolve) => {
        const modal = document.createElement('div');
        modal.className = 'modal fade';
        modal.id = 'quickPaymentModal';
        modal.innerHTML = `
            <div class="modal-dialog modal-dialog-centered">
                <div class="modal-content">
                    <div class="modal-header" style="background: linear-gradient(135deg, #ec4899 0%, #be185d 100%);">
                        <h5 class="modal-title text-white fw-bold">
                            <i class="bi bi-cash-coin me-2"></i>Hızlı Ödeme Al
                        </h5>
                        <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                    </div>
                    <div class="modal-body p-4">
                        <div class="mb-3">
                            <label class="form-label fw-bold">Toplam Tutar</label>
                            <div class="form-control bg-light" style="font-size: 1.2rem; font-weight: bold;">
                                ${formatPrice(totalPrice)} ₺
                            </div>
                        </div>
                        <div class="mb-3">
                            <label class="form-label fw-bold">Alınan Kapora</label>
                            <div class="form-control bg-light">
                                ${formatPrice(depositPaid || 0)} ₺
                            </div>
                        </div>
                        <div class="mb-3">
                            <label class="form-label fw-bold">Kalan Tutar</label>
                            <div class="form-control bg-light text-danger" style="font-size: 1.1rem; font-weight: bold;">
                                ${formatPrice(remaining)} ₺
                            </div>
                        </div>
                        <div class="mb-3">
                            <label class="form-label fw-bold">Ödeme Tutarı <span class="text-danger">*</span></label>
                            <input type="number" id="paymentAmount" class="form-control form-control-lg" 
                                   step="0.01" min="0" max="${remaining}" value="${remaining}" 
                                   style="font-size: 1.2rem; font-weight: bold;">
                            <div class="d-flex gap-2 mt-2">
                                <button type="button" class="btn btn-sm btn-outline-primary" onclick="document.getElementById('paymentAmount').value = '${remaining}'">
                                    Tamamını Al
                                </button>
                                <button type="button" class="btn btn-sm btn-outline-secondary" onclick="document.getElementById('paymentAmount').value = (${remaining} / 2).toFixed(2)">
                                    Yarısını Al
                                </button>
                            </div>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">İptal</button>
                        <button type="button" class="btn btn-success" onclick="confirmQuickPayment()">
                            <i class="bi bi-check-circle me-2"></i>Ödemeyi Al
                        </button>
                    </div>
                </div>
            </div>
        `;
        
        document.body.appendChild(modal);
        const bsModal = new bootstrap.Modal(modal);
        
        window.confirmQuickPayment = function() {
            const amount = parseFloat(document.getElementById('paymentAmount').value);
            if (amount <= 0 || amount > remaining) {
                alert('Geçerli bir tutar giriniz.');
                return;
            }
            bsModal.hide();
            setTimeout(() => modal.remove(), 300);
            resolve(amount);
        };
        
        modal.addEventListener('hidden.bs.modal', function() {
            if (!window.confirmQuickPayment) {
                resolve(null);
            }
            setTimeout(() => modal.remove(), 300);
        });
        
        bsModal.show();
    });
}

// Hızlı Durum Değiştirme
async function quickStatusChange(reservationId, currentStatus) {
    const statuses = [
        { value: 'pending', label: 'Bekleyen', icon: 'bi-clock-history', color: 'warning' },
        { value: 'confirmed', label: 'Onaylandı', icon: 'bi-check-circle', color: 'info' },
        { value: 'open', label: 'Açık', icon: 'bi-clock', color: 'primary' },
        { value: 'completed', label: 'Tamamlandı', icon: 'bi-check-circle-fill', color: 'success' },
        { value: 'cancelled', label: 'İptal', icon: 'bi-x-circle', color: 'danger' }
    ];
    
    const availableStatuses = statuses.filter(s => s.value !== currentStatus);
    
    if (availableStatuses.length === 0) {
        if (typeof showSuccessMessage === 'function') {
            showSuccessMessage('Değiştirilebilecek durum yok.');
        } else {
            alert('Değiştirilebilecek durum yok.');
        }
        return;
    }
    
    // Durum seçim modalı
    const modal = document.createElement('div');
    modal.className = 'modal fade';
    modal.id = 'quickStatusModal';
    modal.innerHTML = `
        <div class="modal-dialog modal-dialog-centered">
            <div class="modal-content">
                <div class="modal-header" style="background: linear-gradient(135deg, #ec4899 0%, #be185d 100%);">
                    <h5 class="modal-title text-white fw-bold">
                        <i class="bi bi-arrow-repeat me-2"></i>Durum Değiştir
                    </h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body p-4">
                    <div class="mb-3">
                        <label class="form-label fw-bold">Mevcut Durum</label>
                        <div class="badge bg-${statuses.find(s => s.value === currentStatus)?.color || 'secondary'} p-2" style="font-size: 0.9rem;">
                            ${statuses.find(s => s.value === currentStatus)?.label || currentStatus}
                        </div>
                    </div>
                    <div class="mb-3">
                        <label class="form-label fw-bold">Yeni Durum Seçin</label>
                        <div class="d-grid gap-2">
                            ${availableStatuses.map(status => `
                                <button type="button" class="btn btn-outline-${status.color} text-start" onclick="selectStatus('${status.value}')">
                                    <i class="bi ${status.icon} me-2"></i>${status.label}
                                </button>
                            `).join('')}
                        </div>
                    </div>
                </div>
            </div>
        </div>
    `;
    
    document.body.appendChild(modal);
    const bsModal = new bootstrap.Modal(modal);
    
    window.selectStatus = async function(newStatus) {
        bsModal.hide();
        setTimeout(() => modal.remove(), 300);
        
        const confirmed = await customConfirm(`Randevu durumunu "${statuses.find(s => s.value === newStatus)?.label}" olarak değiştirmek istediğinize emin misiniz?`);
        if (!confirmed) return;
        
        const url = '{{ route("admin.appointments.update-status", ["id" => ":id"]) }}'.replace(':id', reservationId);
        
        fetch(url, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.content || '',
                'X-Requested-With': 'XMLHttpRequest',
                'Accept': 'application/json'
            },
            body: JSON.stringify({ status: newStatus })
        })
        .then(async response => {
            if (!response.ok) {
                const err = await response.json().catch(() => ({}));
                throw new Error(err.message || 'Durum değiştirilemedi');
            }
            return response.json();
        })
        .then(data => {
            if (data.success) {
                if (typeof showSuccessMessage === 'function') {
                    showSuccessMessage('Durum başarıyla değiştirildi!');
                } else {
                    alert('Durum başarıyla değiştirildi!');
                }
                setTimeout(() => location.reload(), 1000);
            } else {
                if (typeof showErrorMessage === 'function') {
                    showErrorMessage(data.message || 'Durum değiştirilemedi');
                } else {
                    alert(data.message || 'Durum değiştirilemedi');
                }
            }
        })
        .catch(error => {
            console.error('Durum değiştirme hatası:', error);
            if (typeof showErrorMessage === 'function') {
                showErrorMessage(error.message || 'Durum değiştirilirken bir hata oluştu');
            } else {
                alert(error.message || 'Durum değiştirilirken bir hata oluştu');
            }
        });
    };
    
    modal.addEventListener('hidden.bs.modal', function() {
        setTimeout(() => modal.remove(), 300);
    });
    
    bsModal.show();
}


// Klavye Kısayolları
document.addEventListener('keydown', function(e) {
    // Modal açıkken kısayolları devre dışı bırak
    const activeModal = document.querySelector('.modal.show');
    if (activeModal) {
        // Modal içindeyken sadece ESC'ye izin ver
        if (e.key === 'Escape') return;
        // Diğer kısayolları engelle
        return;
    }
    
    // Ctrl+N - Yeni Randevu
    if (e.ctrlKey && e.key === 'n') {
        e.preventDefault();
        quickCreateReservation();
    }
    
    // Ctrl+R - Yenile (sadece modal yokken)
    if (e.ctrlKey && e.key === 'r') {
        e.preventDefault();
        location.reload();
    }
    
    // Ctrl+Shift+C - Yeni Müşteri
    if (e.ctrlKey && e.shiftKey && e.key === 'C') {
        e.preventDefault();
        openCreateCustomerModal();
    }
    
    // Ctrl+Shift+S - Yeni Hizmet
    if (e.ctrlKey && e.shiftKey && e.key === 'S') {
        e.preventDefault();
        openCreateServiceModal();
    }
    
    // Ctrl+Shift+P - Yeni Personel
    if (e.ctrlKey && e.shiftKey && e.key === 'P') {
        e.preventDefault();
        openCreatePersonelModal();
    }
    
    // Ctrl+Shift+R - Yeni Oda
    if (e.ctrlKey && e.shiftKey && e.key === 'R') {
        e.preventDefault();
        openCreateRoomModal();
    }
});
</script>

<style>
    /* Şimşek düğmesi dropdown menüsü için yüksek z-index */
    .btn-group .dropdown-menu {
        z-index: 10001 !important;
    }
    
    @media (max-width: 768px) {
        /* Pending Requests Modal - Ortada ve düzgün görünsün */
        #pendingRequestsModal {
            z-index: 10060 !important;
        }
        
        #pendingRequestsModal .modal-dialog {
            margin: 0.5rem auto !important;
            max-width: calc(100% - 20px) !important;
            max-height: calc(100vh - 20px) !important;
            display: flex !important;
            align-items: center !important;
            justify-content: center !important;
            min-height: auto !important;
            height: auto !important;
            position: relative !important;
        }
        
        #pendingRequestsModal .modal-content {
            max-height: calc(100vh - 20px) !important;
            margin: auto !important;
            width: 100% !important;
            border-radius: 16px !important;
            border: none !important;
            box-shadow: 0 20px 60px rgba(0,0,0,0.3) !important;
        }
        
        /* Modal Header - Gradient Background ve Text Renkleri */
        #pendingRequestsModal .modal-header {
            background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%) !important;
            border-radius: 16px 16px 0 0 !important;
            padding: 16px 20px !important;
            flex-shrink: 0 !important;
            border-bottom: none !important;
        }
        
        #pendingRequestsModal .modal-header .modal-title,
        #pendingRequestsModal .modal-header .text-white,
        #pendingRequestsModal .modal-header h5,
        #pendingRequestsModal .modal-header * {
            color: #ffffff !important;
        }
        
        #pendingRequestsModal .modal-header .btn-close-white {
            filter: brightness(0) invert(1) !important;
            opacity: 1 !important;
        }
        
        #pendingRequestsModal .modal-body {
            padding: 1rem !important;
            background: #ffffff !important;
            color: #111827 !important;
        }
        
        /* Request Detail Modal - Ortada ve düzgün görünsün */
        #requestDetailModal {
            z-index: 10060 !important;
        }
        
        #requestDetailModal .modal-dialog {
            margin: 0.5rem auto !important;
            max-width: calc(100% - 20px) !important;
            max-height: calc(100vh - 20px) !important;
            display: flex !important;
            align-items: center !important;
            justify-content: center !important;
            min-height: auto !important;
            height: auto !important;
            position: relative !important;
        }
        
        #requestDetailModal .modal-content {
            max-height: calc(100vh - 20px) !important;
            margin: auto !important;
            width: 100% !important;
            border-radius: 16px !important;
            border: none !important;
            box-shadow: 0 20px 60px rgba(0,0,0,0.3) !important;
        }
        
        /* Modal Header - Gradient Background ve Text Renkleri */
        #requestDetailModal .modal-header {
            background: linear-gradient(135deg, #ec4899 0%, #be185d 100%) !important;
            border-radius: 16px 16px 0 0 !important;
            padding: 16px 20px !important;
            flex-shrink: 0 !important;
            border-bottom: none !important;
        }
        
        #requestDetailModal .modal-header .modal-title,
        #requestDetailModal .modal-header .text-white,
        #requestDetailModal .modal-header h5,
        #requestDetailModal .modal-header .btn-close,
        #requestDetailModal .modal-header * {
            color: #ffffff !important;
        }
        
        #requestDetailModal .modal-header .btn-close-white {
            filter: brightness(0) invert(1) !important;
            opacity: 1 !important;
        }
        
        /* Modal Body - Text Renkleri */
        #requestDetailModal .modal-body {
            padding: 1rem !important;
            background: #ffffff !important;
            color: #111827 !important;
            overflow-y: auto !important;
        }
        
        /* Modal body içindeki text renklerini düzelt */
        #requestDetailModal .modal-body .text-white,
        #requestDetailModal .modal-body .text-white *,
        #requestDetailModal .modal-body p:not(.text-success):not(.text-danger):not(.text-primary):not(.text-info):not(.text-warning),
        #requestDetailModal .modal-body span:not(.badge):not(.text-success):not(.text-danger):not(.text-primary):not(.text-info):not(.text-warning),
        #requestDetailModal .modal-body div:not(.text-white):not(.text-success):not(.text-danger):not(.text-primary):not(.text-info):not(.text-warning),
        #requestDetailModal .modal-body small:not(.text-muted):not(.text-white) {
            color: #111827 !important;
        }
        
        #requestDetailModal .modal-body .text-muted {
            color: #6b7280 !important;
        }
        
        /* Card header renklerini koru */
        #requestDetailModal .modal-body .card-header[style*="background: linear-gradient"] {
            background: inherit !important;
        }
        
        #requestDetailModal .modal-body .card-header .text-white {
            color: #ffffff !important;
        }
        
        /* Card body text renkleri */
        #requestDetailModal .modal-body .card-body {
            color: #111827 !important;
            background: #ffffff !important;
        }
        
        #requestDetailModal .modal-body .card-body .text-white {
            color: #111827 !important;
        }
        
        #requestDetailModal .row.g-3 > div {
            padding: 0 0.5rem !important;
            margin-bottom: 1rem;
        }
        
        #requestDetailModal .card {
            margin-bottom: 1rem;
        }
        
        #requestDetailModal .btn {
            width: 100%;
            margin-bottom: 0.5rem;
        }
        
        #requestDetailModal .btn-group {
            flex-direction: column;
            width: 100%;
        }
        
        #requestDetailModal .btn-group .btn {
            width: 100%;
            border-radius: 8px !important;
        }
    }
    
    @media (max-width: 480px) {
        #pendingRequestsModal .modal-dialog {
            margin: 0.25rem auto !important;
            max-width: calc(100% - 10px) !important;
        }
        
        #pendingRequestsModal .modal-header {
            padding: 14px 16px !important;
            background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%) !important;
        }
        
        #pendingRequestsModal .modal-header .modal-title,
        #pendingRequestsModal .modal-header .text-white {
            color: #ffffff !important;
        }
        
        #pendingRequestsModal .modal-body {
            padding: 0.75rem !important;
            background: #ffffff !important;
            color: #111827 !important;
        }
        
        #requestDetailModal .modal-dialog {
            margin: 0.25rem auto !important;
            max-width: calc(100% - 10px) !important;
        }
        
        #requestDetailModal .modal-header {
            padding: 14px 16px !important;
            background: linear-gradient(135deg, #ec4899 0%, #be185d 100%) !important;
        }
        
        #requestDetailModal .modal-header .modal-title,
        #requestDetailModal .modal-header .text-white {
            color: #ffffff !important;
        }
        
        #requestDetailModal .modal-body {
            padding: 0.75rem !important;
            background: #ffffff !important;
            color: #111827 !important;
        }
        
        #requestDetailModal .card-header {
            padding: 10px 12px !important;
        }
        
        #requestDetailModal .card-header .text-white {
            color: #ffffff !important;
        }
        
        #requestDetailModal .card-body {
            padding: 0.75rem !important;
            background: #ffffff !important;
            color: #111827 !important;
        }
        
        #requestDetailModal .card-body .text-white {
            color: #111827 !important;
        }
    }
</style>
@endpush

