@extends('admin.mobile.layouts.app')

@section('title', 'Kasa')

@push('styles')
<style>
    .accounting-page {
        padding-bottom: 20px;
    }
    
    .page-header {
        display: flex;
        justify-content: space-between;
        align-items: center;
        margin-bottom: 20px;
        position: sticky;
        top: 0;
        background: #f9fafb;
        padding: 16px 0;
        z-index: 100;
        border-bottom: 1px solid #e5e7eb;
    }
    
    .page-title {
        font-size: 24px;
        font-weight: 700;
        color: #111827;
        margin: 0;
    }
    
    .stats-grid {
        display: grid;
        grid-template-columns: repeat(2, 1fr);
        gap: 12px;
        margin-bottom: 20px;
    }
    
    .stat-card {
        background: white;
        border-radius: 12px;
        padding: 16px;
        box-shadow: 0 2px 8px rgba(0,0,0,0.08);
        border: 1px solid #e5e7eb;
        border-left: 4px solid;
    }
    
    .stat-card.income {
        border-left-color: #ec4899;
    }
    
    .stat-card.expense {
        border-left-color: #ef4444;
    }
    
    .stat-card.balance {
        border-left-color: #ec4899;
    }
    
    .stat-card.net {
        border-left-color: #f59e0b;
    }
    
    .stat-label {
        font-size: 12px;
        color: #6b7280;
        font-weight: 600;
        text-transform: uppercase;
        letter-spacing: 0.5px;
        margin-bottom: 8px;
    }
    
    .stat-value {
        font-size: 20px;
        font-weight: 700;
        color: #111827;
    }
    
    .stat-value.income {
        color: #ec4899;
    }
    
    .stat-value.expense {
        color: #ef4444;
    }
    
    .stat-value.balance {
        color: #ec4899;
    }
    
    .filters-card {
        background: white;
        border-radius: 12px;
        padding: 16px;
        margin-bottom: 16px;
        box-shadow: 0 2px 8px rgba(0,0,0,0.08);
        border: 1px solid #e5e7eb;
    }
    
    .filter-row {
        display: grid;
        grid-template-columns: 1fr 1fr;
        gap: 12px;
        margin-bottom: 12px;
    }
    
    .filter-group {
        display: flex;
        flex-direction: column;
    }
    
    .filter-label {
        font-size: 12px;
        font-weight: 600;
        color: #6b7280;
        margin-bottom: 6px;
        text-transform: uppercase;
        letter-spacing: 0.5px;
    }
    
    .filter-input {
        width: 100%;
        padding: 12px 16px;
        border: 1px solid #e5e7eb;
        border-radius: 8px;
        font-size: 14px;
        background: white;
        -webkit-appearance: none;
    }
    
    .filter-input:focus {
        outline: none;
        border-color: #ec4899;
        box-shadow: 0 0 0 3px rgba(236, 72, 153, 0.1);
    }
    
    .transactions-list {
        display: flex;
        flex-direction: column;
        gap: 12px;
    }
    
    .transaction-item {
        background: white;
        border-radius: 12px;
        padding: 16px;
        box-shadow: 0 2px 8px rgba(0,0,0,0.08);
        border: 1px solid #e5e7eb;
        border-left: 4px solid;
        transition: all 0.2s;
    }
    
    .transaction-item:active {
        transform: translateX(4px);
        box-shadow: 0 4px 12px rgba(0,0,0,0.12);
    }
    
    .transaction-item.income {
        border-left-color: #ec4899;
    }
    
    .transaction-item.expense {
        border-left-color: #ef4444;
    }
    
    .transaction-header {
        display: flex;
        justify-content: space-between;
        align-items: flex-start;
        margin-bottom: 12px;
    }
    
    .transaction-type {
        padding: 6px 12px;
        border-radius: 8px;
        font-size: 12px;
        font-weight: 600;
        white-space: nowrap;
    }
    
    .transaction-type.income {
        background: #fce7f3;
        color: #ec4899;
    }
    
    .transaction-type.expense {
        background: #fee2e2;
        color: #991b1b;
    }
    
    .transaction-amount {
        font-size: 20px;
        font-weight: 700;
    }
    
    .transaction-amount.income {
        color: #ec4899;
    }
    
    .transaction-amount.expense {
        color: #ef4444;
    }
    
    .transaction-details {
        display: flex;
        flex-direction: column;
        gap: 4px;
    }
    
    .transaction-date {
        font-size: 13px;
        color: #6b7280;
        display: flex;
        align-items: center;
        gap: 6px;
    }
    
    .transaction-description {
        font-size: 14px;
        color: #111827;
        font-weight: 500;
    }
    
    .empty-state {
        text-align: center;
        padding: 60px 20px;
        color: #9ca3af;
    }
    
    .empty-state-icon {
        font-size: 64px;
        margin-bottom: 16px;
        opacity: 0.5;
    }
    
    .empty-state-text {
        font-size: 16px;
        font-weight: 600;
        margin-bottom: 8px;
    }
    
    .empty-state-subtext {
        font-size: 14px;
        color: #6b7280;
    }
    
    .modal-overlay {
        display: none;
        position: fixed;
        top: 0;
        left: 0;
        right: 0;
        bottom: 0;
        background: rgba(0, 0, 0, 0.5);
        z-index: 1000;
        align-items: center;
        justify-content: center;
        padding: 20px;
    }
    
    .modal-overlay.active {
        display: flex;
    }
    
    .modal-content {
        background: white;
        border-radius: 16px;
        width: 100%;
        max-width: 500px;
        max-height: 90vh;
        display: flex;
        flex-direction: column;
        box-shadow: 0 20px 25px -5px rgba(0, 0, 0, 0.1);
        overflow: hidden;
    }
    
    .modal-header {
        padding: 20px;
        border-bottom: 1px solid #e5e7eb;
        display: flex;
        justify-content: space-between;
        align-items: center;
        flex-shrink: 0;
    }
    
    .modal-title {
        font-size: 20px;
        font-weight: 700;
        color: #111827;
    }
    
    .modal-close {
        background: none;
        border: none;
        font-size: 24px;
        color: #6b7280;
        cursor: pointer;
        padding: 0;
        width: 32px;
        height: 32px;
        display: flex;
        align-items: center;
        justify-content: center;
        border-radius: 8px;
        transition: all 0.2s;
    }
    
    .modal-close:active {
        background: #f3f4f6;
    }
    
    .modal-body {
        padding: 20px;
        overflow-y: auto;
        flex: 1;
        -webkit-overflow-scrolling: touch;
    }
    
    .modal-footer {
        padding: 20px;
        border-top: 1px solid #e5e7eb;
        display: flex;
        gap: 12px;
        flex-shrink: 0;
        background: white;
    }
    
    .btn-modal {
        flex: 1;
        padding: 12px 24px;
        border-radius: 8px;
        font-size: 14px;
        font-weight: 600;
        border: none;
        cursor: pointer;
        transition: all 0.2s;
        min-height: 44px;
    }
    
    .btn-modal-primary {
        background: #ec4899;
        color: white;
    }
    
    .btn-modal-primary:active {
        background: #be185d;
        transform: scale(0.98);
    }
    
    .btn-modal-secondary {
        background: #f3f4f6;
        color: #111827;
    }
    
    .btn-modal-secondary:active {
        background: #e5e7eb;
        transform: scale(0.98);
    }
</style>
@endpush

@section('content')
<div class="accounting-page">
    <div class="page-header">
        <h1 class="page-title">Kasa</h1>
        <div style="display: flex; gap: 8px;">
            <button type="button" onclick="openIncomeModal()" style="background: #ec4899; color: white; border: none; padding: 10px 16px; border-radius: 8px; font-size: 14px; font-weight: 600; cursor: pointer; display: flex; align-items: center; gap: 6px;">
                <i class="bi bi-plus-circle"></i> Gelir
            </button>
            <button type="button" onclick="openExpenseModal()" style="background: #ef4444; color: white; border: none; padding: 10px 16px; border-radius: 8px; font-size: 14px; font-weight: 600; cursor: pointer; display: flex; align-items: center; gap: 6px;">
                <i class="bi bi-dash-circle"></i> Gider
            </button>
        </div>
    </div>
    
    <!-- İstatistikler -->
    <div class="stats-grid">
        <div class="stat-card balance">
            <div class="stat-label">Kasa Bakiyesi</div>
            <div class="stat-value balance">{{ number_format($totalBalance ?? 0, 2, ',', '.') }} ₺</div>
        </div>
        
        <div class="stat-card net">
            <div class="stat-label">Net Gelir (Bugün)</div>
            <div class="stat-value" style="color: {{ (($todayIncome ?? 0) - ($todayExpense ?? 0)) >= 0 ? '#ec4899' : '#ef4444' }};">
                {{ number_format(($todayIncome ?? 0) - ($todayExpense ?? 0), 2, ',', '.') }} ₺
            </div>
        </div>
        
        <div class="stat-card income">
            <div class="stat-label">Toplam Gelir</div>
            <div class="stat-value income">{{ number_format($totalIncome ?? 0, 2, ',', '.') }} ₺</div>
        </div>
        
        <div class="stat-card expense">
            <div class="stat-label">Toplam Gider</div>
            <div class="stat-value expense">{{ number_format($totalExpense ?? 0, 2, ',', '.') }} ₺</div>
        </div>
        
        <div class="stat-card income">
            <div class="stat-label">Bugünkü Gelir</div>
            <div class="stat-value income">{{ number_format($todayIncome ?? 0, 2, ',', '.') }} ₺</div>
        </div>
        
        <div class="stat-card expense">
            <div class="stat-label">Bugünkü Gider</div>
            <div class="stat-value expense">{{ number_format($todayExpense ?? 0, 2, ',', '.') }} ₺</div>
        </div>
        
        <div class="stat-card income">
            <div class="stat-label">Aylık Gelir</div>
            <div class="stat-value income">{{ number_format($monthlyIncome ?? 0, 2, ',', '.') }} ₺</div>
        </div>
        
        <div class="stat-card expense">
            <div class="stat-label">Aylık Gider</div>
            <div class="stat-value expense">{{ number_format($monthlyExpense ?? 0, 2, ',', '.') }} ₺</div>
        </div>
    </div>
    
    <!-- Filtreler -->
    <div class="filters-card">
        <form method="GET" action="{{ route('mobile.accounting.index') }}" id="filterForm">
            <div class="filter-row">
                <div class="filter-group">
                    <label class="filter-label">Başlangıç Tarihi</label>
                    <input type="date" name="date_from" class="filter-input" value="{{ request('date_from') }}" onchange="document.getElementById('filterForm').submit();">
                </div>
                
                <div class="filter-group">
                    <label class="filter-label">Bitiş Tarihi</label>
                    <input type="date" name="date_to" class="filter-input" value="{{ request('date_to') }}" onchange="document.getElementById('filterForm').submit();">
                </div>
            </div>
            
            <div class="filter-group">
                <label class="filter-label">Tip</label>
                <select name="type" class="filter-input" onchange="document.getElementById('filterForm').submit();">
                    <option value="">Tümü</option>
                    <option value="income" {{ request('type') == 'income' ? 'selected' : '' }}>Gelir</option>
                    <option value="expense" {{ request('type') == 'expense' ? 'selected' : '' }}>Gider</option>
                </select>
            </div>
        </form>
    </div>
    
    <!-- İşlemler Listesi -->
    @if(isset($transactions) && $transactions->count() > 0)
        <div class="transactions-list">
            @foreach($transactions as $transaction)
                <div class="transaction-item {{ $transaction->type }}">
                    <div class="transaction-header">
                        <span class="transaction-type {{ $transaction->type }}">
                            {{ $transaction->type === 'income' ? 'Gelir' : 'Gider' }}
                        </span>
                        <div class="transaction-amount {{ $transaction->type }}">
                            {{ $transaction->type === 'income' ? '+' : '-' }}{{ number_format($transaction->amount, 2, ',', '.') }} ₺
                        </div>
                    </div>
                    <div class="transaction-details">
                        <div class="transaction-date">
                            <i class="bi bi-calendar3"></i>
                            {{ \Carbon\Carbon::parse($transaction->date ?? $transaction->created_at)->format('d.m.Y') }}
                            @if($transaction->time)
                                <i class="bi bi-clock" style="margin-left: 8px;"></i>
                                {{ $transaction->time }}
                            @endif
                        </div>
                        @if($transaction->description)
                            <div class="transaction-description">{{ $transaction->description }}</div>
                        @endif
                        @if($transaction->payer_name)
                            <div class="transaction-description" style="font-size: 13px; color: #6b7280;">
                                <i class="bi bi-person"></i> {{ $transaction->payer_name }}
                            </div>
                        @endif
                        @if($transaction->customer)
                            <div class="transaction-description" style="font-size: 13px; color: #6b7280;">
                                <i class="bi bi-person-circle"></i> {{ $transaction->customer->name }}
                            </div>
                        @endif
                        @if($transaction->reservation)
                            <div class="transaction-description" style="font-size: 13px; color: #6b7280;">
                                <i class="bi bi-calendar-check"></i> Randevu #{{ $transaction->reservation->code ?? $transaction->reservation->id }}
                            </div>
                        @endif
                    </div>
                    <div style="margin-top: 12px; display: flex; gap: 8px; justify-content: flex-end;">
                        <button type="button" onclick="deleteTransaction({{ $transaction->id }}, this)" style="background: #ef4444; color: white; border: none; padding: 8px 16px; border-radius: 8px; font-size: 13px; font-weight: 600; cursor: pointer; display: flex; align-items: center; gap: 6px;">
                            <i class="bi bi-trash"></i> Sil
                        </button>
                    </div>
                </div>
            @endforeach
        </div>
        
        <div style="margin-top: 24px; display: flex; justify-content: center;">
            {{ $transactions->links() }}
        </div>
    @else
        <div class="empty-state">
            <div class="empty-state-icon">💰</div>
            <div class="empty-state-text">Henüz işlem yok</div>
            <div class="empty-state-subtext">
                @if(request()->hasAny(['date_from', 'date_to', 'type']))
                    Filtrelere uygun işlem bulunamadı
                @else
                    Kasa işlemleriniz burada görünecek
                @endif
            </div>
        </div>
    @endif
</div>

<!-- Gelir Ekleme Modal -->
<div class="modal-overlay" id="incomeModal">
    <div class="modal-content">
        <div class="modal-header">
            <h2 class="modal-title"><i class="bi bi-plus-circle"></i> Gelir Ekle</h2>
            <button type="button" class="modal-close" onclick="closeIncomeModal()">&times;</button>
        </div>
        <form id="incomeForm" onsubmit="submitIncomeForm(event)">
            <div class="modal-body">
                <div id="incomeFormErrors" style="display: none;"></div>
                
                <div class="form-group">
                    <label class="form-label">Tutar <span style="color: #dc3545;">*</span></label>
                    <input type="number" step="0.01" name="amount" class="form-control" required min="0" placeholder="0.00">
                </div>
                
                <div class="form-group">
                    <label class="form-label">Tarih <span style="color: #dc3545;">*</span></label>
                    <input type="date" name="date" class="form-control" value="{{ date('Y-m-d') }}" required>
                </div>
                
                <div class="form-group">
                    <label class="form-label">Saat</label>
                    <input type="time" name="time" class="form-control">
                </div>
                
                <div class="form-group">
                    <label class="form-label">Müşteri</label>
                    <select name="customer_id" class="form-control" id="incomeCustomerSelect">
                        <option value="">Seçiniz (Opsiyonel)</option>
                        @foreach($customers ?? [] as $customer)
                            <option value="{{ $customer->id }}">{{ $customer->name }}</option>
                        @endforeach
                    </select>
                </div>
                
                <div class="form-group">
                    <label class="form-label">Randevu</label>
                    <select name="reservation_id" class="form-control" id="incomeReservationSelect">
                        <option value="">Önce müşteri seçin</option>
                    </select>
                </div>
                
                <div class="form-group">
                    <label class="form-label">Ödeme Yapan</label>
                    <input type="text" name="payer_name" class="form-control" placeholder="Ödeme yapan kişinin adı">
                </div>
                
                <div class="form-group">
                    <label class="form-label">Açıklama</label>
                    <textarea name="description" class="form-control" rows="3" placeholder="İşlem açıklaması"></textarea>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn-modal btn-modal-secondary" onclick="closeIncomeModal()">İptal</button>
                <button type="submit" class="btn-modal btn-modal-primary" id="incomeSubmitBtn">
                    <i class="bi bi-check-circle"></i> Kaydet
                </button>
            </div>
        </form>
    </div>
</div>

<!-- Gider Ekleme Modal -->
<div class="modal-overlay" id="expenseModal">
    <div class="modal-content">
        <div class="modal-header">
            <h2 class="modal-title"><i class="bi bi-dash-circle"></i> Gider Ekle</h2>
            <button type="button" class="modal-close" onclick="closeExpenseModal()">&times;</button>
        </div>
        <form id="expenseForm" onsubmit="submitExpenseForm(event)">
            <div class="modal-body">
                <div id="expenseFormErrors" style="display: none;"></div>
                
                <div class="form-group">
                    <label class="form-label">Tutar <span style="color: #dc3545;">*</span></label>
                    <input type="number" step="0.01" name="amount" class="form-control" required min="0" placeholder="0.00">
                </div>
                
                <div class="form-group">
                    <label class="form-label">Tarih <span style="color: #dc3545;">*</span></label>
                    <input type="date" name="date" class="form-control" value="{{ date('Y-m-d') }}" required>
                </div>
                
                <div class="form-group">
                    <label class="form-label">Saat</label>
                    <input type="time" name="time" class="form-control">
                </div>
                
                <div class="form-group">
                    <label class="form-label">Müşteri</label>
                    <select name="customer_id" class="form-control" id="expenseCustomerSelect">
                        <option value="">Seçiniz (Opsiyonel)</option>
                        @foreach($customers ?? [] as $customer)
                            <option value="{{ $customer->id }}">{{ $customer->name }}</option>
                        @endforeach
                    </select>
                </div>
                
                <div class="form-group">
                    <label class="form-label">Randevu</label>
                    <select name="reservation_id" class="form-control" id="expenseReservationSelect">
                        <option value="">Seçiniz (Opsiyonel)</option>
                    </select>
                </div>
                
                <div class="form-group">
                    <label class="form-label">Açıklama</label>
                    <textarea name="description" class="form-control" rows="3" placeholder="İşlem açıklaması"></textarea>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn-modal btn-modal-secondary" onclick="closeExpenseModal()">İptal</button>
                <button type="submit" class="btn-modal btn-modal-primary" id="expenseSubmitBtn">
                    <i class="bi bi-check-circle"></i> Kaydet
                </button>
            </div>
        </form>
    </div>
</div>

@push('scripts')
<script>
@php
    $appointmentsJson = ($appointments ?? collect())->map(function($r) {
        return [
            'id' => $r->id,
            'customer_id' => $r->customer_id,
            'customer' => $r->customer ? ['name' => $r->customer->name] : null,
            'date' => $r->date ? \Carbon\Carbon::parse($r->date)->format('d.m.Y') : ($r->created_at ? $r->created_at->format('d.m.Y') : '')
        ];
    })->values()->all();
@endphp
const allAppointments = @json($appointmentsJson ?? []);

function openIncomeModal() {
    document.getElementById('incomeModal').classList.add('active');
    document.getElementById('incomeForm').reset();
    document.getElementById('incomeForm').querySelector('input[name="date"]').value = '{{ date("Y-m-d") }}';
    document.getElementById('incomeFormErrors').style.display = 'none';
    
    // Randevu listesini sıfırla ve tüm randevuları göster
    const reservationSelect = document.getElementById('incomeReservationSelect');
    if (reservationSelect) {
        reservationSelect.innerHTML = '<option value="">Seçiniz (Opsiyonel)</option>';
        allAppointments.forEach(reservation => {
            const option = document.createElement('option');
            option.value = reservation.id;
            option.dataset.customerId = reservation.customer_id;
            const customerName = reservation.customer?.name || 'Müşteri Yok';
            option.textContent = `#${reservation.id} - ${customerName} (${reservation.date})`;
            reservationSelect.appendChild(option);
        });
    }
}

function closeIncomeModal() {
    document.getElementById('incomeModal').classList.remove('active');
}

function openExpenseModal() {
    document.getElementById('expenseModal').classList.add('active');
    document.getElementById('expenseForm').reset();
    document.getElementById('expenseForm').querySelector('input[name="date"]').value = '{{ date("Y-m-d") }}';
    document.getElementById('expenseFormErrors').style.display = 'none';
    
    // Randevu listesini sıfırla ve tüm randevuları göster
    const reservationSelect = document.getElementById('expenseReservationSelect');
    if (reservationSelect) {
        reservationSelect.innerHTML = '<option value="">Seçiniz (Opsiyonel)</option>';
        allAppointments.forEach(reservation => {
            const option = document.createElement('option');
            option.value = reservation.id;
            option.dataset.customerId = reservation.customer_id;
            const customerName = reservation.customer?.name || 'Müşteri Yok';
            option.textContent = `#${reservation.id} - ${customerName} (${reservation.date})`;
            reservationSelect.appendChild(option);
        });
    }
}

function closeExpenseModal() {
    document.getElementById('expenseModal').classList.remove('active');
}

// Sonsuz döngüyü önlemek için flag
let isUpdatingReservation = false;

// Müşteri seçildiğinde randevuları filtrele (Gelir)
document.getElementById('incomeCustomerSelect')?.addEventListener('change', function() {
    if (isUpdatingReservation) return;
    
    const customerId = this.value;
    const reservationSelect = document.getElementById('incomeReservationSelect');
    
    if (!reservationSelect) return;
    
    // Mevcut seçili randevuu koru
    const currentReservationId = reservationSelect.value;
    
    if (customerId) {
        const filteredAppointments = allAppointments.filter(r => r.customer_id == customerId);
        reservationSelect.innerHTML = '<option value="">Seçiniz (Opsiyonel)</option>';
        filteredAppointments.forEach(reservation => {
            const option = document.createElement('option');
            option.value = reservation.id;
            option.dataset.customerId = reservation.customer_id;
            const customerName = reservation.customer?.name || 'Müşteri Yok';
            option.textContent = `#${reservation.id} - ${customerName} (${reservation.date})`;
            reservationSelect.appendChild(option);
        });
        
        // Eğer seçili randevu bu müşteriye aitse, seçimini koru
        if (currentReservationId && filteredAppointments.find(r => r.id == currentReservationId)) {
            reservationSelect.value = currentReservationId;
        }
    } else {
        reservationSelect.innerHTML = '<option value="">Seçiniz (Opsiyonel)</option>';
        allAppointments.forEach(reservation => {
            const option = document.createElement('option');
            option.value = reservation.id;
            option.dataset.customerId = reservation.customer_id;
            const customerName = reservation.customer?.name || 'Müşteri Yok';
            option.textContent = `#${reservation.id} - ${customerName} (${reservation.date})`;
            reservationSelect.appendChild(option);
        });
        
        // Eğer seçili randevu varsa, seçimini koru
        if (currentReservationId) {
            reservationSelect.value = currentReservationId;
        }
    }
});

// Randevu seçildiğinde müşteriyi otomatik doldur (Gelir)
document.getElementById('incomeReservationSelect')?.addEventListener('change', function() {
    const selectedOption = this.options[this.selectedIndex];
    if (!selectedOption) return;
    
    const customerId = selectedOption.dataset.customerId;
    const reservationId = this.value;
    
    if (customerId && reservationId) {
        isUpdatingReservation = true;
        const customerSelect = document.getElementById('incomeCustomerSelect');
        const currentCustomerId = customerSelect.value;
        
        // Müşteri değişecekse güncelle
        if (currentCustomerId != customerId) {
            customerSelect.value = customerId;
            // Müşteri değiştiği için randevu listesini güncelle ama seçimi koru
            customerSelect.dispatchEvent(new Event('change'));
        }
        
        // Randevu seçimini koru
        setTimeout(() => {
            const reservationSelect = document.getElementById('incomeReservationSelect');
            if (reservationSelect) {
                reservationSelect.value = reservationId;
            }
            isUpdatingReservation = false;
        }, 50);
    }
});

// Müşteri seçildiğinde randevuları filtrele (Gider)
document.getElementById('expenseCustomerSelect')?.addEventListener('change', function() {
    if (isUpdatingReservation) return;
    
    const customerId = this.value;
    const reservationSelect = document.getElementById('expenseReservationSelect');
    
    if (!reservationSelect) return;
    
    // Mevcut seçili randevuu koru
    const currentReservationId = reservationSelect.value;
    
    if (customerId) {
        const filteredAppointments = allAppointments.filter(r => r.customer_id == customerId);
        reservationSelect.innerHTML = '<option value="">Seçiniz (Opsiyonel)</option>';
        filteredAppointments.forEach(reservation => {
            const option = document.createElement('option');
            option.value = reservation.id;
            option.dataset.customerId = reservation.customer_id;
            const customerName = reservation.customer?.name || 'Müşteri Yok';
            option.textContent = `#${reservation.id} - ${customerName} (${reservation.date})`;
            reservationSelect.appendChild(option);
        });
        
        // Eğer seçili randevu bu müşteriye aitse, seçimini koru
        if (currentReservationId && filteredAppointments.find(r => r.id == currentReservationId)) {
            reservationSelect.value = currentReservationId;
        }
    } else {
        reservationSelect.innerHTML = '<option value="">Seçiniz (Opsiyonel)</option>';
        allAppointments.forEach(reservation => {
            const option = document.createElement('option');
            option.value = reservation.id;
            option.dataset.customerId = reservation.customer_id;
            const customerName = reservation.customer?.name || 'Müşteri Yok';
            option.textContent = `#${reservation.id} - ${customerName} (${reservation.date})`;
            reservationSelect.appendChild(option);
        });
        
        // Eğer seçili randevu varsa, seçimini koru
        if (currentReservationId) {
            reservationSelect.value = currentReservationId;
        }
    }
});

// Randevu seçildiğinde müşteriyi otomatik doldur (Gider)
document.getElementById('expenseReservationSelect')?.addEventListener('change', function() {
    const selectedOption = this.options[this.selectedIndex];
    if (!selectedOption) return;
    
    const customerId = selectedOption.dataset.customerId;
    const reservationId = this.value;
    
    if (customerId && reservationId) {
        isUpdatingReservation = true;
        const customerSelect = document.getElementById('expenseCustomerSelect');
        const currentCustomerId = customerSelect.value;
        
        // Müşteri değişecekse güncelle
        if (currentCustomerId != customerId) {
            customerSelect.value = customerId;
            // Müşteri değiştiği için randevu listesini güncelle ama seçimi koru
            customerSelect.dispatchEvent(new Event('change'));
        }
        
        // Randevu seçimini koru
        setTimeout(() => {
            const reservationSelect = document.getElementById('expenseReservationSelect');
            if (reservationSelect) {
                reservationSelect.value = reservationId;
            }
            isUpdatingReservation = false;
        }, 50);
    }
});

function submitIncomeForm(e) {
    e.preventDefault();
    const form = document.getElementById('incomeForm');
    const submitBtn = document.getElementById('incomeSubmitBtn');
    const errorsDiv = document.getElementById('incomeFormErrors');
    
    submitBtn.disabled = true;
    submitBtn.innerHTML = '<i class="bi bi-hourglass-split"></i> Kaydediliyor...';
    errorsDiv.style.display = 'none';
    
    const formData = new FormData(form);
    
    fetch('{{ route("mobile.accounting.store-income") }}', {
        method: 'POST',
        headers: {
            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content'),
            'Accept': 'application/json',
            'X-Requested-With': 'XMLHttpRequest'
        },
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            closeIncomeModal();
            location.reload();
        } else {
            let errorHtml = '<div class="alert alert-danger"><ul style="margin: 0; padding-left: 20px;">';
            if (data.errors) {
                Object.values(data.errors).forEach(error => {
                    if (Array.isArray(error)) {
                        error.forEach(msg => errorHtml += '<li>' + msg + '</li>');
                    } else {
                        errorHtml += '<li>' + error + '</li>';
                    }
                });
            } else if (data.message) {
                errorHtml += '<li>' + data.message + '</li>';
            }
            errorHtml += '</ul></div>';
            errorsDiv.innerHTML = errorHtml;
            errorsDiv.style.display = 'block';
            submitBtn.disabled = false;
            submitBtn.innerHTML = '<i class="bi bi-check-circle"></i> Kaydet';
        }
    })
    .catch(error => {
        console.error('Error:', error);
        errorsDiv.innerHTML = '<div class="alert alert-danger">Bir hata oluştu. Lütfen tekrar deneyin.</div>';
        errorsDiv.style.display = 'block';
        submitBtn.disabled = false;
        submitBtn.innerHTML = '<i class="bi bi-check-circle"></i> Kaydet';
    });
}

function submitExpenseForm(e) {
    e.preventDefault();
    const form = document.getElementById('expenseForm');
    const submitBtn = document.getElementById('expenseSubmitBtn');
    const errorsDiv = document.getElementById('expenseFormErrors');
    
    submitBtn.disabled = true;
    submitBtn.innerHTML = '<i class="bi bi-hourglass-split"></i> Kaydediliyor...';
    errorsDiv.style.display = 'none';
    
    const formData = new FormData(form);
    
    fetch('{{ route("mobile.accounting.store-expense") }}', {
        method: 'POST',
        headers: {
            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content'),
            'Accept': 'application/json',
            'X-Requested-With': 'XMLHttpRequest'
        },
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            closeExpenseModal();
            location.reload();
        } else {
            let errorHtml = '<div class="alert alert-danger"><ul style="margin: 0; padding-left: 20px;">';
            if (data.errors) {
                Object.values(data.errors).forEach(error => {
                    if (Array.isArray(error)) {
                        error.forEach(msg => errorHtml += '<li>' + msg + '</li>');
                    } else {
                        errorHtml += '<li>' + error + '</li>';
                    }
                });
            } else if (data.message) {
                errorHtml += '<li>' + data.message + '</li>';
            }
            errorHtml += '</ul></div>';
            errorsDiv.innerHTML = errorHtml;
            errorsDiv.style.display = 'block';
            submitBtn.disabled = false;
            submitBtn.innerHTML = '<i class="bi bi-check-circle"></i> Kaydet';
        }
    })
    .catch(error => {
        console.error('Error:', error);
        errorsDiv.innerHTML = '<div class="alert alert-danger">Bir hata oluştu. Lütfen tekrar deneyin.</div>';
        errorsDiv.style.display = 'block';
        submitBtn.disabled = false;
        submitBtn.innerHTML = '<i class="bi bi-check-circle"></i> Kaydet';
    });
}

// Modal dışına tıklanınca kapat
document.getElementById('incomeModal')?.addEventListener('click', function(e) {
    if (e.target === this) {
        closeIncomeModal();
    }
});

document.getElementById('expenseModal')?.addEventListener('click', function(e) {
    if (e.target === this) {
        closeExpenseModal();
    }
});

// İşlem silme
async function deleteTransaction(transactionId, eventElement) {
    showConfirmModal(
        'İşlemi Sil',
        'Bu işlemi silmek istediğinize emin misiniz? Bu işlem geri alınamaz ve bakiyeler güncellenecektir.',
        async function() {
            const deleteBtn = eventElement ? eventElement.closest('button') : document.querySelector(`button[onclick*="deleteTransaction(${transactionId})"]`);
            if (!deleteBtn) {
                console.error('Delete button not found');
                showAlertModal('Hata', 'Silme butonu bulunamadı.');
                return;
            }
            
            const originalText = deleteBtn.innerHTML;
            deleteBtn.disabled = true;
            deleteBtn.innerHTML = '<img src="{{ asset('splashlogo.gif') }}" alt="Yükleniyor" style="width: 20px; height: 20px; display: inline-block; margin-right: 8px;"> Siliniyor...';
            
            try {
                const response = await fetch(`{{ route("mobile.accounting.destroy", ":id") }}`.replace(':id', transactionId), {
                    method: 'DELETE',
                    headers: {
                        'X-Requested-With': 'XMLHttpRequest',
                        'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content'),
                        'Accept': 'application/json'
                    }
                });
                
                if (!response.ok) {
                    const errorData = await response.json().catch(() => ({ message: 'Bir hata oluştu.' }));
                    throw new Error(errorData.message || 'Bir hata oluştu.');
                }
                
                const data = await response.json();
                
                if (data.success) {
                    showAlertModal('Başarılı', 'İşlem başarıyla silindi.', function() {
                        window.location.reload();
                    });
                } else {
                    showAlertModal('Hata', data.message || 'Bir hata oluştu.');
                    deleteBtn.disabled = false;
                    deleteBtn.innerHTML = originalText;
                }
            } catch (error) {
                console.error('Error:', error);
                showAlertModal('Hata', error.message || 'İşlem silinirken bir hata oluştu.');
                deleteBtn.disabled = false;
                deleteBtn.innerHTML = originalText;
            }
        }
    );
}
</script>
@endpush
@endsection
