@extends('admin.mobile.layouts.app')

@section('title', 'Yeni Randevu')

@push('styles')
<style>
    .reservation-create {
        max-width: 100%;
    }
    
    .page-header {
        display: flex;
        justify-content: space-between;
        align-items: center;
        margin-bottom: 24px;
        padding-bottom: 16px;
        border-bottom: 1px solid #e5e7eb;
    }
    
    .page-title {
        font-size: 24px;
        font-weight: 700;
        color: #111827;
    }
    
    .back-link {
        background: #e5e7eb;
        color: #4b5563;
        padding: 8px 12px;
        border-radius: 8px;
        text-decoration: none;
        font-size: 14px;
        font-weight: 600;
        display: inline-flex;
        align-items: center;
        gap: 6px;
        transition: all 0.2s;
    }
    
    .back-link:active {
        background: #d1d5db;
        transform: scale(0.98);
    }
    
    .section-card {
        background: white;
        border-radius: 12px;
        padding: 16px;
        margin-bottom: 16px;
        box-shadow: 0 2px 8px rgba(0,0,0,0.08);
        border: 1px solid #e5e7eb;
    }
    
    .section-header {
        font-size: 16px;
        font-weight: 700;
        color: #111827;
        margin-bottom: 16px;
        display: flex;
        align-items: center;
        gap: 8px;
        padding-bottom: 12px;
        border-bottom: 2px solid #f3f4f6;
    }
    
    .section-header i {
        color: #10b981;
        font-size: 18px;
    }
    
    .form-group {
        margin-bottom: 16px;
    }
    
    .form-label {
        display: block;
        font-size: 14px;
        font-weight: 600;
        color: #374151;
        margin-bottom: 8px;
    }
    
    .form-label .required {
        color: #ef4444;
    }
    
    .form-control, .form-select {
        width: 100%;
        padding: 12px 16px;
        border: 1px solid #d1d5db;
        border-radius: 8px;
        font-size: 16px;
        background: white;
        transition: all 0.2s;
    }
    
    .form-control:focus, .form-select:focus {
        outline: none;
        border-color: #10b981;
        box-shadow: 0 0 0 3px rgba(16, 185, 129, 0.1);
    }
    
    .row {
        display: flex;
        gap: 12px;
        margin-bottom: 16px;
    }
    
    .col-6 {
        flex: 1;
    }
    
    .item-row {
        background: #f9fafb;
        border: 1px solid #e5e7eb;
        border-radius: 8px;
        padding: 12px;
        margin-bottom: 12px;
    }
    
    .item-row-header {
        display: flex;
        justify-content: space-between;
        align-items: center;
        margin-bottom: 12px;
    }
    
    .item-row-title {
        font-size: 14px;
        font-weight: 600;
        color: #111827;
    }
    
    .remove-btn {
        background: #ef4444;
        color: white;
        border: none;
        border-radius: 6px;
        padding: 6px 12px;
        font-size: 14px;
        cursor: pointer;
        display: flex;
        align-items: center;
        gap: 4px;
    }
    
    .add-btn {
        background: #10b981;
        color: white;
        border: none;
        border-radius: 8px;
        padding: 10px 16px;
        font-size: 14px;
        font-weight: 600;
        cursor: pointer;
        display: inline-flex;
        align-items: center;
        gap: 6px;
        margin-top: 8px;
    }
    
    .add-btn:active {
        background: #059669;
        transform: scale(0.98);
    }
    
    .price-summary {
        background: linear-gradient(135deg, #f0f9ff 0%, #e0f2fe 100%);
        border: 2px solid #10b981;
        border-radius: 12px;
        padding: 20px;
        margin-bottom: 16px;
    }
    
    .price-summary-title {
        font-size: 18px;
        font-weight: 700;
        color: #111827;
        margin-bottom: 16px;
        display: flex;
        align-items: center;
        gap: 8px;
    }
    
    .price-row {
        display: flex;
        justify-content: space-between;
        align-items: center;
        padding: 8px 0;
        border-bottom: 1px solid #e5e7eb;
    }
    
    .price-row:last-child {
        border-bottom: none;
    }
    
    .price-label {
        font-size: 14px;
        color: #6b7280;
    }
    
    .price-value {
        font-size: 16px;
        font-weight: 600;
        color: #111827;
    }
    
    .price-total {
        display: flex;
        justify-content: space-between;
        align-items: center;
        padding: 16px 0;
        margin-top: 12px;
        border-top: 2px solid #10b981;
    }
    
    .price-total-label {
        font-size: 20px;
        font-weight: 700;
        color: #111827;
    }
    
    .price-total-value {
        font-size: 24px;
        font-weight: 700;
        color: #10b981;
    }
    
    .submit-btn {
        width: 100%;
        padding: 16px;
        background: #10b981;
        color: white;
        border: none;
        border-radius: 12px;
        font-size: 18px;
        font-weight: 700;
        cursor: pointer;
        margin-top: 24px;
        transition: all 0.2s;
    }
    
    .submit-btn:active {
        background: #047857;
        transform: scale(0.98);
    }
    
    .checkbox-item {
        background: white;
        border: 1px solid #e5e7eb;
        border-radius: 8px;
        padding: 12px;
        margin-bottom: 8px;
        display: flex;
        align-items: center;
        gap: 12px;
    }
    
    .checkbox-item input[type="checkbox"] {
        width: 20px;
        height: 20px;
        cursor: pointer;
    }
    
    .checkbox-item-label {
        flex: 1;
    }
    
    .checkbox-item-name {
        font-weight: 600;
        color: #111827;
        margin-bottom: 4px;
    }
    
    .checkbox-item-desc {
        font-size: 12px;
        color: #6b7280;
    }
    
    .checkbox-item-price {
        font-size: 16px;
        font-weight: 700;
        color: #10b981;
    }
    
    .info-text {
        font-size: 12px;
        color: #6b7280;
        margin-top: 4px;
        display: flex;
        align-items: center;
        gap: 4px;
    }
    
    .btn {
        background: #10b981;
        color: white;
        border: none;
        padding: 12px 16px;
        border-radius: 8px;
        font-size: 14px;
        font-weight: 600;
        cursor: pointer;
        transition: all 0.2s;
        display: inline-flex;
        align-items: center;
        gap: 6px;
    }
    
    .btn:active {
        background: #047857;
        transform: scale(0.98);
    }
    
    .btn-primary {
        background: #10b981;
    }
    
    .btn-primary:active {
        background: #047857;
    }
    
    /* New Customer Modal */
    .modal-overlay {
        display: none;
        position: fixed;
        top: 0;
        left: 0;
        right: 0;
        bottom: 0;
        background: rgba(0, 0, 0, 0.5);
        z-index: 10000;
        align-items: center;
        justify-content: center;
        padding: 20px;
    }
    
    .modal-overlay.active {
        display: flex;
    }
    
    .modal-content {
        background: white;
        border-radius: 16px;
        width: 100%;
        max-width: 400px;
        max-height: 90vh;
        overflow-y: auto;
        box-shadow: 0 20px 25px -5px rgba(0, 0, 0, 0.1);
        animation: modalSlideIn 0.3s ease-out;
    }
    
    @keyframes modalSlideIn {
        from {
            opacity: 0;
            transform: translateY(-20px) scale(0.95);
        }
        to {
            opacity: 1;
            transform: translateY(0) scale(1);
        }
    }
    
    .modal-header {
        padding: 20px;
        border-bottom: 1px solid #e5e7eb;
        display: flex;
        justify-content: space-between;
        align-items: center;
    }
    
    .modal-title {
        font-size: 20px;
        font-weight: 700;
        color: #111827;
        margin: 0;
    }
    
    .modal-close {
        background: none;
        border: none;
        font-size: 24px;
        color: #6b7280;
        cursor: pointer;
        padding: 0;
        width: 32px;
        height: 32px;
        display: flex;
        align-items: center;
        justify-content: center;
        border-radius: 8px;
        transition: all 0.2s;
    }
    
    .modal-close:active {
        background: #f3f4f6;
    }
    
    .modal-body {
        padding: 20px;
    }
    
    .modal-footer {
        padding: 20px;
        border-top: 1px solid #e5e7eb;
        display: flex;
        gap: 12px;
        justify-content: flex-end;
    }
    
    .spinner-border-sm {
        width: 1rem;
        height: 1rem;
        border-width: 0.15em;
        border: 0.15em solid currentColor;
        border-right-color: transparent;
        border-radius: 50%;
        display: inline-block;
        animation: spinner-border 0.75s linear infinite;
    }
    
    @keyframes spinner-border {
        to { transform: rotate(360deg); }
    }
</style>
@endpush

@section('content')
<div class="reservation-create">
    <div class="page-header">
        <h1 class="page-title">Yeni Randevu</h1>
        <a href="{{ route('mobile.appointments.index') }}" class="back-link">
            <i class="bi bi-x"></i> İptal
        </a>
    </div>
    
    <form method="POST" action="{{ route('mobile.appointments.store') }}" id="reservationForm">
        @csrf
        <input type="hidden" name="total_price" id="hiddenTotalPrice" value="0">
        <input type="hidden" name="items_data" id="hiddenItemsData" value="">
        
        <!-- Müşteri ve Salon -->
        <div class="section-card">
            <div class="section-header">
                <i class="bi bi-person"></i> Müşteri ve Salon Bilgileri
            </div>
            
            <div class="form-group">
                <label class="form-label">
                    Müşteri <span class="required">*</span>
                </label>
                <div style="display: flex; gap: 8px;">
                    <select name="customer_id" id="customer_id" class="form-select" required style="flex: 1;">
                        <option value="">Müşteri Seçiniz</option>
                        @foreach($customers as $customer)
                            <option value="{{ $customer->id }}">
                                {{ $customer->name }}@if($customer->phone) ({{ $customer->phone }})@endif
                            </option>
                        @endforeach
                    </select>
                    <button type="button" class="btn btn-primary" onclick="openNewCustomerModal()" style="padding: 12px 16px; border-radius: 8px; white-space: nowrap;">
                        <i class="bi bi-plus-circle"></i> Yeni
                    </button>
                </div>
            </div>
            
            <div class="form-group">
                <label class="form-label">
                    Salon <span class="required">*</span>
                </label>
                <select name="salon_id" id="salon_id" class="form-select" required>
                    <option value="">Salon Seçiniz</option>
                    @foreach($salons as $salon)
                        <option value="{{ $salon->id }}">{{ $salon->name }}</option>
                    @endforeach
                </select>
            </div>
            
            <div class="form-group">
                <label class="form-label">
                    Gelin Adı <span class="required">*</span>
                </label>
                <input type="text" name="bride_name" id="bride_name" class="form-control" placeholder="Gelin adı" required>
            </div>
            
            <div class="form-group">
                <label class="form-label">
                    Damat Adı <span class="required">*</span>
                </label>
                <input type="text" name="groom_name" id="groom_name" class="form-control" placeholder="Damat adı" required>
            </div>
            
            <div class="form-group">
                <label class="form-label">
                    <i class="bi bi-telephone-fill"></i> Gelin & Damat Telefon <span class="required">*</span>
                </label>
                <input type="tel" name="bride_groom_phone" id="bride_groom_phone" 
                       class="form-control" placeholder="05XX XXX XX XX" 
                       pattern="[0-9]{10,11}" required>
                <div class="info-text">
                    <i class="bi bi-info-circle"></i>
                    Gelin ve damatın iletişim numarası - Bu numaraya SMS gönderilecektir
                </div>
            </div>
        </div>
        
        <!-- Tarih ve Saat -->
        <div class="section-card">
            <div class="section-header">
                <i class="bi bi-calendar-event"></i> Tarih ve Saat
            </div>
            
            <div class="form-group">
                <label class="form-label">
                    Tarih <span class="required">*</span>
                </label>
                <input type="date" name="date" id="reservation_date" class="form-control" 
                       value="{{ request('date', date('Y-m-d')) }}" min="{{ date('Y-m-d') }}" required>
            </div>
            
            <div class="row">
                <div class="col-6">
                    <div class="form-group">
                        <label class="form-label">
                            Başlangıç Saati <span class="required">*</span>
                        </label>
                        <select name="start_time" id="start_time" class="form-select" required>
                            <option value="">Saat Seçiniz</option>
                        </select>
                    </div>
                </div>
                <div class="col-6">
                    <div class="form-group">
                        <label class="form-label">
                            Bitiş Saati <span class="required">*</span>
                        </label>
                        <select name="end_time" id="end_time" class="form-select" required>
                            <option value="">Saat Seçiniz</option>
                        </select>
                    </div>
                </div>
            </div>
            <div id="timeCheckResult" style="display: none; margin-top: 12px; padding: 12px; border-radius: 8px; font-size: 14px;">
                <span id="timeCheckMessage"></span>
            </div>
        </div>
        
        <!-- Yemekler -->
        <div class="section-card">
            <div class="section-header">
                <i class="bi bi-cup-hot"></i> Yemekler
            </div>
            
            <div id="mealsContainer">
                <div class="item-row" data-meal-index="0">
                    <div class="item-row-header">
                        <span class="item-row-title">Yemek 1</span>
                        <button type="button" class="remove-btn remove-meal-btn">
                            <i class="bi bi-trash"></i>
                        </button>
                    </div>
                    <div class="form-group">
                        <label class="form-label">Yemek Adı</label>
                        <input type="text" class="form-control meal-name" data-meal-index="0" placeholder="Yemek adı">
                    </div>
                    <div class="row">
                        <div class="col-6">
                            <div class="form-group">
                                <label class="form-label">Miktar</label>
                                <input type="number" class="form-control meal-quantity" value="1" min="1" data-meal-index="0">
                            </div>
                        </div>
                        <div class="col-6">
                            <div class="form-group">
                                <label class="form-label">Birim Fiyat (₺)</label>
                                <input type="number" step="0.01" class="form-control meal-price" value="0" min="0" data-meal-index="0">
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            
            <button type="button" class="add-btn" id="addMealBtn">
                <i class="bi bi-plus-circle"></i> Yemek Ekle
            </button>
        </div>
        
        <!-- Organizasyon -->
        <div class="section-card">
            <div class="section-header">
                <i class="bi bi-gift"></i> Organizasyon
            </div>
            
            <div id="organizationsContainer">
                <div class="item-row" data-org-index="0">
                    <div class="item-row-header">
                        <span class="item-row-title">Organizasyon 1</span>
                        <button type="button" class="remove-btn remove-org-btn">
                            <i class="bi bi-trash"></i>
                        </button>
                    </div>
                    <div class="form-group">
                        <label class="form-label">Organizasyon Adı</label>
                        <input type="text" class="form-control org-name" data-org-index="0" placeholder="Örn: Düğün Organizasyonu">
                    </div>
                    <div class="row">
                        <div class="col-6">
                            <div class="form-group">
                                <label class="form-label">Adet</label>
                                <input type="number" class="form-control org-quantity" data-org-index="0" value="1" min="1">
                            </div>
                        </div>
                        <div class="col-6">
                            <div class="form-group">
                                <label class="form-label">Birim Fiyat (₺)</label>
                                <input type="number" step="0.01" class="form-control org-price" data-org-index="0" value="0" min="0">
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            
            <button type="button" class="add-btn" id="addOrgBtn">
                <i class="bi bi-plus-circle"></i> Organizasyon Ekle
            </button>
        </div>
        
        <!-- Ekstra Hizmetler -->
        <div class="section-card">
            <div class="section-header">
                <i class="bi bi-star"></i> Ekstra Hizmetler
            </div>
            
            @if($extras->count() > 0)
            <div id="extrasCheckboxContainer" style="margin-bottom: 16px;">
                @foreach($extras as $extra)
                <div class="checkbox-item">
                    <input type="checkbox" class="extra-checkbox" 
                           value="{{ $extra->id }}" 
                           id="extra_{{ $extra->id }}"
                           data-price="{{ $extra->price }}"
                           data-name="{{ $extra->name }}">
                    <label for="extra_{{ $extra->id }}" class="checkbox-item-label">
                        <div class="checkbox-item-name">{{ $extra->name }}</div>
                        @if($extra->description)
                            <div class="checkbox-item-desc">{{ $extra->description }}</div>
                        @endif
                    </label>
                    <div class="checkbox-item-price">{{ number_format($extra->price, 2, ',', '.') }} ₺</div>
                </div>
                @endforeach
            </div>
            @endif
            
            <div id="extrasContainer">
                <div class="item-row" data-extra-index="0">
                    <div class="item-row-header">
                        <span class="item-row-title">Ekstra 1</span>
                        <button type="button" class="remove-btn remove-extra-btn">
                            <i class="bi bi-trash"></i>
                        </button>
                    </div>
                    <div class="row">
                        <div class="col-6">
                            <div class="form-group">
                                <label class="form-label">Ekstra Adı</label>
                                <input type="text" class="form-control extra-name" data-extra-index="0" placeholder="Örn: Fotoğraf Çekimi">
                            </div>
                        </div>
                        <div class="col-6">
                            <div class="form-group">
                                <label class="form-label">Fiyat (₺)</label>
                                <input type="number" step="0.01" class="form-control extra-price" data-extra-index="0" value="0" min="0">
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            
            <button type="button" class="add-btn" id="addExtraBtn">
                <i class="bi bi-plus-circle"></i> Ekstra Ekle
            </button>
        </div>
        
        <!-- Fiyatlandırma -->
        <div class="section-card">
            <div class="price-summary">
                <div class="price-summary-title">
                    <i class="bi bi-calculator"></i> Fiyatlandırma
                </div>
                
                <div class="form-group">
                    <label class="form-label">
                        Salon Ücreti (₺) <span class="required">*</span>
                    </label>
                    <input type="number" step="0.01" id="salon_price" name="salon_price" class="form-control" value="0" min="0" required>
                </div>
                
                <div class="price-row">
                    <span class="price-label">Salon Ücreti:</span>
                    <span class="price-value" id="summarySalon">0.00 ₺</span>
                </div>
                <div class="price-row">
                    <span class="price-label">Yemek Toplamı:</span>
                    <span class="price-value" id="summaryMeals">0.00 ₺</span>
                </div>
                <div class="price-row">
                    <span class="price-label">Organizasyon:</span>
                    <span class="price-value" id="summaryOrg">0.00 ₺</span>
                </div>
                <div class="price-row">
                    <span class="price-label">Ekstra Hizmetler:</span>
                    <span class="price-value" id="summaryExtras">0.00 ₺</span>
                </div>
                
                <div class="price-total">
                    <span class="price-total-label">TOPLAM:</span>
                    <span class="price-total-value" id="summaryTotal">0.00 ₺</span>
                </div>
                
                <div class="form-group" style="margin-top: 16px;">
                    <label class="form-label">Ödenen Kapora (₺)</label>
                    <input type="number" step="0.01" name="deposit_paid" id="deposit_paid" class="form-control" value="0" min="0">
                </div>
                
                <div class="form-group">
                    <label class="form-label">Kalan Tutar</label>
                    <input type="text" id="remaining_amount" class="form-control" value="0.00 ₺" readonly style="font-weight: 600; color: #10b981;">
                </div>
            </div>
        </div>
        
        <!-- Notlar -->
        <div class="section-card">
            <div class="section-header">
                <i class="bi bi-file-text"></i> Notlar
            </div>
            <div class="form-group">
                <textarea name="notes" id="notes" class="form-control" rows="4" placeholder="Randevu ile ilgili notlar..."></textarea>
            </div>
        </div>
        
        <button type="submit" class="submit-btn">
            <i class="bi bi-check-circle"></i> Randevuu Kaydet
        </button>
    </form>
</div>

<!-- Yeni Müşteri Oluşturma Modalı -->
<div class="modal-overlay" id="newCustomerModalOverlay">
    <div class="modal-content">
        <div class="modal-header">
            <h3 class="modal-title">Yeni Müşteri Ekle</h3>
            <button type="button" class="modal-close" onclick="closeNewCustomerModal()">
                <i class="bi bi-x-lg"></i>
            </button>
        </div>
        <form id="newCustomerForm">
            @csrf
            <div class="modal-body">
                <div id="newCustomerErrors" style="display: none; margin-bottom: 16px;"></div>
                
                <div class="form-group">
                    <label class="form-label">
                        Müşteri Adı <span class="required">*</span>
                    </label>
                    <input type="text" name="name" class="form-control" placeholder="Müşteri adı" required>
                </div>
                
                <div class="form-group">
                    <label class="form-label">
                        Telefon <span class="required">*</span>
                    </label>
                    <input type="tel" name="phone" class="form-control" placeholder="05XX XXX XX XX" pattern="[0-9]{10,11}" required>
                </div>
                
                <div class="form-group">
                    <label class="form-label">E-posta</label>
                    <input type="email" name="email" class="form-control" placeholder="ornek@email.com">
                </div>
                
                <div class="form-group">
                    <label class="form-label">Adres</label>
                    <textarea name="address" class="form-control" rows="3" placeholder="Adres bilgisi"></textarea>
                </div>
                
                <div class="form-group">
                    <label class="form-label">Notlar</label>
                    <textarea name="notes" class="form-control" rows="2" placeholder="Müşteri hakkında notlar"></textarea>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="custom-modal-btn custom-modal-btn-secondary" onclick="closeNewCustomerModal()">İptal</button>
                <button type="submit" class="custom-modal-btn custom-modal-btn-primary">Oluştur</button>
            </div>
        </form>
    </div>
</div>

<!-- SMS Bakiyesi Uyarı Modalı -->
@if(session('sms_warning'))
<div class="modal fade" id="smsBalanceWarningModal" tabindex="-1" aria-labelledby="smsBalanceWarningModalLabel" aria-hidden="true" data-bs-backdrop="static">
    <div class="modal-dialog modal-dialog-centered">
        <div class="modal-content">
            <div class="modal-header bg-warning text-dark">
                <h5 class="modal-title" id="smsBalanceWarningModalLabel">
                    <i class="bi bi-exclamation-triangle-fill me-2"></i>SMS Bakiyesi Uyarısı
                </h5>
            </div>
            <div class="modal-body">
                <p class="mb-3">{{ session('sms_warning') }}</p>
                <p class="mb-0">SMS göndermeye devam etmek için lütfen bakiye yükleyin.</p>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Daha Sonra</button>
                <a href="{{ route('sms-packages.index') }}" class="btn btn-primary">
                    <i class="bi bi-currency-dollar me-2"></i>SMS Al
                </a>
            </div>
        </div>
    </div>
</div>
@endif

@push('scripts')
@if(session('sms_warning'))
<script>
    document.addEventListener('DOMContentLoaded', function() {
        var smsWarningModal = new bootstrap.Modal(document.getElementById('smsBalanceWarningModal'));
        smsWarningModal.show();
    });
</script>
@endif
<script>
let mealCount = 1;
let orgCount = 1;
let extraCount = 1;

// Fiyat hesaplama
function calculateTotal() {
    let salonPrice = parseFloat(document.getElementById('salon_price').value) || 0;
    let mealsTotal = 0;
    let orgTotal = 0;
    let extrasTotal = 0;
    
    // Yemek toplamı
    document.querySelectorAll('.meal-row, .item-row[data-meal-index]').forEach(row => {
        const quantity = parseFloat(row.querySelector('.meal-quantity')?.value) || 0;
        const price = parseFloat(row.querySelector('.meal-price')?.value) || 0;
        mealsTotal += quantity * price;
    });
    
    // Organizasyon toplamı
    document.querySelectorAll('.item-row[data-org-index]').forEach(row => {
        const quantity = parseFloat(row.querySelector('.org-quantity')?.value) || 0;
        const price = parseFloat(row.querySelector('.org-price')?.value) || 0;
        orgTotal += quantity * price;
    });
    
    // Ekstra toplamı (checkbox)
    document.querySelectorAll('.extra-checkbox:checked').forEach(checkbox => {
        extrasTotal += parseFloat(checkbox.dataset.price) || 0;
    });
    
    // Ekstra toplamı (manuel)
    document.querySelectorAll('.item-row[data-extra-index]').forEach(row => {
        const price = parseFloat(row.querySelector('.extra-price')?.value) || 0;
        extrasTotal += price;
    });
    
    const total = salonPrice + mealsTotal + orgTotal + extrasTotal;
    const depositPaid = parseFloat(document.getElementById('deposit_paid').value) || 0;
    const remaining = total - depositPaid;
    
    // Fiyat formatla (binlik ayırıcı nokta, ondalık virgül)
    function formatPrice(amount) {
        const parts = amount.toFixed(2).split('.');
        const integerPart = parts[0].replace(/\B(?=(\d{3})+(?!\d))/g, '.');
        return integerPart + ',' + parts[1];
    }
    
    // Güncelle
    document.getElementById('summarySalon').textContent = formatPrice(salonPrice) + ' ₺';
    document.getElementById('summaryMeals').textContent = formatPrice(mealsTotal) + ' ₺';
    document.getElementById('summaryOrg').textContent = formatPrice(orgTotal) + ' ₺';
    document.getElementById('summaryExtras').textContent = formatPrice(extrasTotal) + ' ₺';
    document.getElementById('summaryTotal').textContent = formatPrice(total) + ' ₺';
    document.getElementById('remaining_amount').value = formatPrice(remaining) + ' ₺';
    document.getElementById('hiddenTotalPrice').value = total;
    
    // Items data
    const itemsData = {
        meals: [],
        organizations: [],
        extras: []
    };
    
    document.querySelectorAll('.item-row[data-meal-index]').forEach(row => {
        const name = row.querySelector('.meal-name')?.value;
        const quantity = row.querySelector('.meal-quantity')?.value;
        const price = row.querySelector('.meal-price')?.value;
        if (name) {
            itemsData.meals.push({ name, quantity, price });
        }
    });
    
    document.querySelectorAll('.item-row[data-org-index]').forEach(row => {
        const name = row.querySelector('.org-name')?.value;
        const quantity = row.querySelector('.org-quantity')?.value;
        const price = row.querySelector('.org-price')?.value;
        if (name) {
            itemsData.organizations.push({ name, quantity, price });
        }
    });
    
    document.querySelectorAll('.extra-checkbox:checked').forEach(checkbox => {
        itemsData.extras.push({
            id: checkbox.value,
            name: checkbox.dataset.name,
            price: checkbox.dataset.price
        });
    });
    
    document.querySelectorAll('.item-row[data-extra-index]').forEach(row => {
        const name = row.querySelector('.extra-name')?.value;
        const price = row.querySelector('.extra-price')?.value;
        if (name) {
            itemsData.extras.push({ name, price });
        }
    });
    
    document.getElementById('hiddenItemsData').value = JSON.stringify(itemsData);
}

// Event listeners
document.getElementById('salon_price').addEventListener('input', calculateTotal);
document.getElementById('deposit_paid').addEventListener('input', calculateTotal);

// Yemek ekle
document.getElementById('addMealBtn').addEventListener('click', function() {
    const container = document.getElementById('mealsContainer');
    const newRow = document.createElement('div');
    newRow.className = 'item-row';
    newRow.setAttribute('data-meal-index', mealCount);
    newRow.innerHTML = `
        <div class="item-row-header">
            <span class="item-row-title">Yemek ${mealCount + 1}</span>
            <button type="button" class="remove-btn remove-meal-btn">
                <i class="bi bi-trash"></i>
            </button>
        </div>
        <div class="form-group">
            <label class="form-label">Yemek Adı</label>
            <input type="text" class="form-control meal-name" data-meal-index="${mealCount}" placeholder="Yemek adı">
        </div>
        <div class="row">
            <div class="col-6">
                <div class="form-group">
                    <label class="form-label">Miktar</label>
                    <input type="number" class="form-control meal-quantity" value="1" min="1" data-meal-index="${mealCount}">
                </div>
            </div>
            <div class="col-6">
                <div class="form-group">
                    <label class="form-label">Birim Fiyat (₺)</label>
                    <input type="number" step="0.01" class="form-control meal-price" value="0" min="0" data-meal-index="${mealCount}">
                </div>
            </div>
        </div>
    `;
    container.appendChild(newRow);
    mealCount++;
    
    // Event listeners ekle
    newRow.querySelector('.meal-quantity').addEventListener('input', calculateTotal);
    newRow.querySelector('.meal-price').addEventListener('input', calculateTotal);
    newRow.querySelector('.remove-meal-btn').addEventListener('click', function() {
        newRow.remove();
        calculateTotal();
    });
});

// Organizasyon ekle
document.getElementById('addOrgBtn').addEventListener('click', function() {
    const container = document.getElementById('organizationsContainer');
    const newRow = document.createElement('div');
    newRow.className = 'item-row';
    newRow.setAttribute('data-org-index', orgCount);
    newRow.innerHTML = `
        <div class="item-row-header">
            <span class="item-row-title">Organizasyon ${orgCount + 1}</span>
            <button type="button" class="remove-btn remove-org-btn">
                <i class="bi bi-trash"></i>
            </button>
        </div>
        <div class="form-group">
            <label class="form-label">Organizasyon Adı</label>
            <input type="text" class="form-control org-name" data-org-index="${orgCount}" placeholder="Örn: Düğün Organizasyonu">
        </div>
        <div class="row">
            <div class="col-6">
                <div class="form-group">
                    <label class="form-label">Adet</label>
                    <input type="number" class="form-control org-quantity" data-org-index="${orgCount}" value="1" min="1">
                </div>
            </div>
            <div class="col-6">
                <div class="form-group">
                    <label class="form-label">Birim Fiyat (₺)</label>
                    <input type="number" step="0.01" class="form-control org-price" data-org-index="${orgCount}" value="0" min="0">
                </div>
            </div>
        </div>
    `;
    container.appendChild(newRow);
    orgCount++;
    
    // Event listeners ekle
    newRow.querySelector('.org-quantity').addEventListener('input', calculateTotal);
    newRow.querySelector('.org-price').addEventListener('input', calculateTotal);
    newRow.querySelector('.remove-org-btn').addEventListener('click', function() {
        newRow.remove();
        calculateTotal();
    });
});

// Ekstra ekle
document.getElementById('addExtraBtn').addEventListener('click', function() {
    const container = document.getElementById('extrasContainer');
    const newRow = document.createElement('div');
    newRow.className = 'item-row';
    newRow.setAttribute('data-extra-index', extraCount);
    newRow.innerHTML = `
        <div class="item-row-header">
            <span class="item-row-title">Ekstra ${extraCount + 1}</span>
            <button type="button" class="remove-btn remove-extra-btn">
                <i class="bi bi-trash"></i>
            </button>
        </div>
        <div class="row">
            <div class="col-6">
                <div class="form-group">
                    <label class="form-label">Ekstra Adı</label>
                    <input type="text" class="form-control extra-name" data-extra-index="${extraCount}" placeholder="Örn: Fotoğraf Çekimi">
                </div>
            </div>
            <div class="col-6">
                <div class="form-group">
                    <label class="form-label">Fiyat (₺)</label>
                    <input type="number" step="0.01" class="form-control extra-price" data-extra-index="${extraCount}" value="0" min="0">
                </div>
            </div>
        </div>
    `;
    container.appendChild(newRow);
    extraCount++;
    
    // Event listeners ekle
    newRow.querySelector('.extra-price').addEventListener('input', calculateTotal);
    newRow.querySelector('.remove-extra-btn').addEventListener('click', function() {
        newRow.remove();
        calculateTotal();
    });
});

// Mevcut item'lar için event listeners
document.querySelectorAll('.meal-quantity, .meal-price').forEach(el => {
    el.addEventListener('input', calculateTotal);
});

document.querySelectorAll('.org-quantity, .org-price').forEach(el => {
    el.addEventListener('input', calculateTotal);
});

document.querySelectorAll('.extra-price').forEach(el => {
    el.addEventListener('input', calculateTotal);
});

document.querySelectorAll('.extra-checkbox').forEach(checkbox => {
    checkbox.addEventListener('change', calculateTotal);
});

document.querySelectorAll('.remove-meal-btn, .remove-org-btn, .remove-extra-btn').forEach(btn => {
    btn.addEventListener('click', function() {
        this.closest('.item-row').remove();
        calculateTotal();
    });
});

// İlk hesaplama
calculateTotal();

// Yeni Müşteri Modal
function openNewCustomerModal() {
    document.getElementById('newCustomerModalOverlay').classList.add('active');
    document.body.style.overflow = 'hidden';
    // Form'u temizle
    document.getElementById('newCustomerForm').reset();
    document.getElementById('newCustomerErrors').style.display = 'none';
    document.getElementById('newCustomerErrors').innerHTML = '';
}

function closeNewCustomerModal() {
    document.getElementById('newCustomerModalOverlay').classList.remove('active');
    document.body.style.overflow = '';
}

// Yeni müşteri oluştur
document.getElementById('newCustomerForm').addEventListener('submit', async function(e) {
    e.preventDefault();
    
    const form = this;
    const formData = new FormData(form);
    const submitBtn = form.querySelector('button[type="submit"]');
    const originalText = submitBtn.innerHTML;
    const errorDiv = document.getElementById('newCustomerErrors');
    
    submitBtn.disabled = true;
    submitBtn.innerHTML = '<img src="{{ asset('splashlogo.gif') }}" alt="Yükleniyor" style="width: 20px; height: 20px; display: inline-block; margin-right: 8px;"> Oluşturuluyor...';
    errorDiv.style.display = 'none';
    errorDiv.innerHTML = '';
    
    try {
        const response = await fetch('{{ route("mobile.customers.store") }}', {
            method: 'POST',
            body: formData,
            headers: {
                'X-Requested-With': 'XMLHttpRequest',
                'Accept': 'application/json',
                'X-CSRF-TOKEN': '{{ csrf_token() }}'
            }
        });
        
        const data = await response.json();
        
        if (response.ok && data.success) {
            // Yeni müşteriyi select'e ekle ve seç
            const customerSelect = document.getElementById('customer_id');
            const newOption = document.createElement('option');
            newOption.value = data.customer.id;
            newOption.textContent = data.customer.name + (data.customer.phone ? ' (' + data.customer.phone + ')' : '');
            newOption.selected = true;
            customerSelect.appendChild(newOption);
            
            // Modal'ı kapat
            closeNewCustomerModal();
            
            // Başarı mesajı
            showAlertModal('Başarılı', 'Yeni müşteri oluşturuldu ve seçildi.');
        } else {
            // Hata var
            let errorHtml = '<div class="alert alert-danger"><ul class="mb-0">';
            if (data.errors) {
                Object.keys(data.errors).forEach(key => {
                    data.errors[key].forEach(error => {
                        errorHtml += '<li>' + error + '</li>';
                    });
                });
            } else if (data.message) {
                errorHtml += '<li>' + data.message + '</li>';
            } else {
                errorHtml += '<li>Bir hata oluştu. Lütfen tekrar deneyin.</li>';
            }
            errorHtml += '</ul></div>';
            errorDiv.innerHTML = errorHtml;
            errorDiv.style.display = 'block';
        }
    } catch (error) {
        console.error('Customer Create Error:', error);
        errorDiv.innerHTML = '<div class="alert alert-danger"><ul class="mb-0"><li>Bir hata oluştu. Lütfen tekrar deneyin.</li></ul></div>';
        errorDiv.style.display = 'block';
    } finally {
        submitBtn.disabled = false;
        submitBtn.innerHTML = originalText;
    }
});

// Form submit edildiğinde hesaplamayı yap ve verileri güncelle
document.getElementById('reservationForm').addEventListener('submit', function(e) {
    e.preventDefault(); // Normal submit'i engelle, AJAX ile gönder
    
    // Önce hesaplamayı yap
    calculateTotal();
    
    // Total price kontrolü
    const totalPrice = parseFloat(document.getElementById('hiddenTotalPrice').value) || 0;
    if (totalPrice <= 0) {
        showAlertModal('Uyarı', 'Toplam tutar 0\'dan büyük olmalıdır. Lütfen fiyat bilgilerini kontrol edin.');
        return false;
    }
    
    // Items data kontrolü
    const itemsData = document.getElementById('hiddenItemsData').value;
    if (!itemsData || itemsData === '{}') {
        // Items data boşsa bile gönder (salon fiyatı varsa yeterli)
        document.getElementById('hiddenItemsData').value = JSON.stringify({ meals: [], organizations: [], extras: [] });
    }
    
    // Submit butonunu disable et ve loading göster
    const submitBtn = document.querySelector('.submit-btn');
    const originalHtml = submitBtn.innerHTML;
    submitBtn.disabled = true;
    submitBtn.innerHTML = '<i class="bi bi-hourglass-split"></i> Kaydediliyor...';
    
    // Form verilerini al
    const formData = new FormData(this);
    
    // AJAX ile gönder
    fetch('{{ route("mobile.appointments.store") }}', {
        method: 'POST',
        body: formData,
        headers: {
            'X-Requested-With': 'XMLHttpRequest',
            'Accept': 'application/json',
            'X-CSRF-TOKEN': '{{ csrf_token() }}'
        },
        credentials: 'same-origin'
    })
    .then(response => {
        // Content-Type kontrolü
        const contentType = response.headers.get('content-type');
        if (contentType && contentType.includes('application/json')) {
            return response.json().then(data => ({ status: response.status, data }));
        } else {
            // JSON değilse, muhtemelen redirect oldu
            // Response text'i al ve kontrol et
            return response.text().then(text => {
                // Eğer HTML döndüyse, muhtemelen redirect oldu
                if (response.status >= 300 && response.status < 400) {
                    // Redirect header'ını kontrol et
                    const location = response.headers.get('Location');
                    if (location) {
                        window.location.href = location;
                        return null;
                    }
                }
                // HTML response ise, sayfayı yenile
                if (text.includes('<!DOCTYPE') || text.includes('<html')) {
                    window.location.reload();
                    return null;
                }
                return { status: response.status, data: { message: 'Beklenmeyen yanıt formatı' } };
            });
        }
    })
    .then(result => {
        if (!result) return; // Redirect olduysa çık
        
        if (result.status === 200 && result.data?.success) {
            // Başarılı - redirect edilecek
            if (result.data?.redirect) {
                window.location.href = result.data.redirect;
            } else {
                // Randevu detay sayfasına git
                window.location.href = '{{ route("mobile.appointments.index") }}';
            }
        } else {
            // Hata var
            let errorMessage = 'Randevu oluşturulurken bir hata oluştu.';
            
            if (result.data?.errors) {
                // Validation hataları
                const errors = result.data.errors;
                const errorList = Object.keys(errors).map(key => {
                    return errors[key].join(', ');
                }).join('\n');
                errorMessage = 'Lütfen formu kontrol edin:\n\n' + errorList;
            } else if (result.data?.message) {
                errorMessage = result.data.message;
            }
            
            showAlertModal('Hata', errorMessage);
            submitBtn.disabled = false;
            submitBtn.innerHTML = originalHtml;
        }
    })
    .catch(error => {
        console.error('Form submit error:', error);
        showAlertModal('Hata', 'Bir hata oluştu. Lütfen tekrar deneyin.\n\nHata: ' + error.message);
        submitBtn.disabled = false;
        submitBtn.innerHTML = originalHtml;
    });
    
    return false;
});

// Saat seçeneklerini oluştur (30 dakika aralıklarla)
function generateTimeSlots() {
    const slots = [];
    for (let hour = 8; hour < 24; hour++) {
        for (let minute = 0; minute < 60; minute += 30) {
            const timeStr = String(hour).padStart(2, '0') + ':' + String(minute).padStart(2, '0');
            slots.push(timeStr);
        }
    }
    return slots;
}

// Dolu saatleri kontrol et ve select'leri güncelle
function updateTimeSelects() {
    const date = document.getElementById('reservation_date').value;
    const salonId = document.getElementById('salon_id').value;
    const startTimeSelect = document.getElementById('start_time');
    const endTimeSelect = document.getElementById('end_time');
    const resultDiv = document.getElementById('timeCheckResult');
    const messageSpan = document.getElementById('timeCheckMessage');
    
    // Tarih ve salon seçilmemişse sadece saatleri oluştur
    if (!date || !salonId) {
        const slots = generateTimeSlots();
        startTimeSelect.innerHTML = '<option value="">Saat Seçiniz</option>';
        endTimeSelect.innerHTML = '<option value="">Saat Seçiniz</option>';
        
        slots.forEach(slot => {
            const option1 = document.createElement('option');
            option1.value = slot;
            option1.textContent = slot;
            startTimeSelect.appendChild(option1);
            
            const option2 = document.createElement('option');
            option2.value = slot;
            option2.textContent = slot;
            endTimeSelect.appendChild(option2);
        });
        
        if (resultDiv) resultDiv.style.display = 'none';
        return;
    }
    
    // API'den dolu saatleri çek
    fetch('{{ route("admin.reservations.check-times") }}', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'X-CSRF-TOKEN': '{{ csrf_token() }}',
            'Accept': 'application/json'
        },
        body: JSON.stringify({ date, salon_id: salonId })
    })
    .then(response => response.json())
    .then(data => {
        const bookedSlots = data.booked_slots || [];
        const slots = generateTimeSlots();
        
        // Başlangıç saati select'ini güncelle
        startTimeSelect.innerHTML = '<option value="">Saat Seçiniz</option>';
        slots.forEach(slot => {
            const option = document.createElement('option');
            option.value = slot;
            const isBooked = bookedSlots.includes(slot);
            option.textContent = isBooked ? slot + ' (Dolu)' : slot;
            option.disabled = isBooked;
            if (isBooked) {
                option.style.color = '#dc3545';
                option.style.fontWeight = '600';
            }
            startTimeSelect.appendChild(option);
        });
        
        // Bitiş saati select'ini güncelle
        endTimeSelect.innerHTML = '<option value="">Saat Seçiniz</option>';
        slots.forEach(slot => {
            const option = document.createElement('option');
            option.value = slot;
            const isBooked = bookedSlots.includes(slot);
            option.textContent = isBooked ? slot + ' (Dolu)' : slot;
            option.disabled = isBooked;
            if (isBooked) {
                option.style.color = '#dc3545';
                option.style.fontWeight = '600';
            }
            endTimeSelect.appendChild(option);
        });
        
        // Seçilen saatlerde çakışma kontrolü
        checkTimeConflict();
    })
    .catch(error => {
        console.error('Saat kontrolü hatası:', error);
        // Hata durumunda da saatleri oluştur
        const slots = generateTimeSlots();
        startTimeSelect.innerHTML = '<option value="">Saat Seçiniz</option>';
        endTimeSelect.innerHTML = '<option value="">Saat Seçiniz</option>';
        slots.forEach(slot => {
            const option1 = document.createElement('option');
            option1.value = slot;
            option1.textContent = slot;
            startTimeSelect.appendChild(option1);
            
            const option2 = document.createElement('option');
            option2.value = slot;
            option2.textContent = slot;
            endTimeSelect.appendChild(option2);
        });
    });
}

// Seçilen saatlerde çakışma kontrolü
function checkTimeConflict() {
    const date = document.getElementById('reservation_date').value;
    const salonId = document.getElementById('salon_id').value;
    const startTime = document.getElementById('start_time').value;
    const endTime = document.getElementById('end_time').value;
    const resultDiv = document.getElementById('timeCheckResult');
    const messageSpan = document.getElementById('timeCheckMessage');
    
    if (!date || !salonId || !startTime || !endTime || !resultDiv || !messageSpan) {
        if (resultDiv) resultDiv.style.display = 'none';
        return;
    }
    
    if (startTime >= endTime) {
        resultDiv.style.display = 'block';
        resultDiv.style.background = '#fee2e2';
        resultDiv.style.color = '#991b1b';
        resultDiv.style.border = '1px solid #fca5a5';
        messageSpan.textContent = '⚠️ Bitiş saati başlangıç saatinden sonra olmalıdır!';
        return;
    }
    
    // API'den dolu saatleri tekrar çek ve kontrol et
    fetch('{{ route("admin.reservations.check-times") }}', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'X-CSRF-TOKEN': '{{ csrf_token() }}',
            'Accept': 'application/json'
        },
        body: JSON.stringify({ date, salon_id: salonId })
    })
    .then(response => response.json())
    .then(data => {
        const bookedSlots = data.booked_slots || [];
        let hasConflict = false;
        
        // Seçilen saat aralığında çakışma var mı kontrol et
        const slots = generateTimeSlots();
        const startIndex = slots.indexOf(startTime);
        const endIndex = slots.indexOf(endTime);
        
        for (let i = startIndex; i < endIndex; i++) {
            if (bookedSlots.includes(slots[i])) {
                hasConflict = true;
                break;
            }
        }
        
        if (hasConflict) {
            resultDiv.style.display = 'block';
            resultDiv.style.background = '#fef3c7';
            resultDiv.style.color = '#92400e';
            resultDiv.style.border = '1px solid #fcd34d';
            messageSpan.textContent = '⚠️ Seçilen saat aralığında başka bir randevu var!';
        } else {
            resultDiv.style.display = 'block';
            resultDiv.style.background = '#d1fae5';
            resultDiv.style.color = '#065f46';
            resultDiv.style.border = '1px solid #6ee7b7';
            messageSpan.textContent = '✅ Seçilen saat aralığı müsait!';
        }
    })
    .catch(error => {
        console.error('Saat kontrolü hatası:', error);
        if (resultDiv) resultDiv.style.display = 'none';
    });
}

// Event listeners
document.getElementById('reservation_date').addEventListener('change', updateTimeSelects);
document.getElementById('salon_id').addEventListener('change', updateTimeSelects);
document.getElementById('start_time').addEventListener('change', checkTimeConflict);
document.getElementById('end_time').addEventListener('change', checkTimeConflict);

// URL'den tarih parametresini al ve set et
document.addEventListener('DOMContentLoaded', function() {
    const urlParams = new URLSearchParams(window.location.search);
    const dateParam = urlParams.get('date');
    if (dateParam) {
        const dateInput = document.getElementById('reservation_date');
        if (dateInput) {
            dateInput.value = dateParam;
            // Tarih set edildikten sonra saatleri güncelle
            setTimeout(() => {
                updateTimeSelects();
            }, 200);
        }
    } else {
        // URL'de tarih yoksa normal saatleri oluştur
        updateTimeSelects();
    }
});
</script>
@endpush
@endsection
