// Appointment wizard v2: rebuilt, isolated and deterministic.
(function () {
    const modalEl = document.getElementById('appointmentModal');
    if (!modalEl || window.__appointmentWizardV2Ready) return;
    window.__appointmentWizardV2Ready = true;

    const STEPS = [1, 2, 3, 4, 5, 6];
    const state = {
        salonId: {{ $salon->id }},
        currentStep: 1,
        isRegistered: false,
        customerId: null,
        selectedServiceIds: [],
        selectedDate: '',
        selectedTime: '',
        resendCooldown: 0,
        resendTimer: null
    };

    function byId(id) {
        return document.getElementById(id);
    }

    function notify(message, type) {
        if (typeof window.showCustomAlert === 'function') {
            window.showCustomAlert(message, type || 'info');
            return;
        }
        window.alert(message);
    }

    function setStep(step) {
        state.currentStep = step;
        STEPS.forEach((s) => {
            const el = byId(`step${s}`);
            if (el) el.style.display = s === step ? 'block' : 'none';
        });

        const prevBtn = byId('prevStepBtn');
        const nextBtn = byId('nextStepBtn');
        const submitBtn = byId('submitBtn');
        if (prevBtn) prevBtn.style.display = step === 1 ? 'none' : 'inline-block';
        if (nextBtn) nextBtn.style.display = step >= 6 || step === 1 ? 'none' : 'inline-block';
        if (submitBtn) submitBtn.style.display = step === 6 ? 'inline-block' : 'none';

        const progressBar = byId('appointmentProgressBar');
        if (progressBar) {
            const percent = Math.max(1, Math.min(step, 6)) * (100 / 6);
            progressBar.style.width = `${percent}%`;
        }
        document.querySelectorAll('[data-step-indicator]').forEach((item) => {
            const itemStep = Number(item.getAttribute('data-step-indicator'));
            item.classList.toggle('is-active', itemStep <= step);
        });
    }

    function updateSelectedServicesCount() {
        const badge = byId('selectedServicesCount');
        if (!badge) return;
        const count = state.selectedServiceIds.length;
        if (count > 0) {
            badge.style.display = 'inline-block';
            badge.textContent = `${count} hizmet secildi`;
        } else {
            badge.style.display = 'none';
        }
        const serviceIdInput = byId('selectedServiceId');
        const serviceIdsInput = byId('selectedServiceIds');
        if (serviceIdInput) serviceIdInput.value = count > 0 ? String(state.selectedServiceIds[0]) : '';
        if (serviceIdsInput) serviceIdsInput.value = JSON.stringify(state.selectedServiceIds);
    }

    function applyServiceCardState(serviceId, selected) {
        modalEl.querySelectorAll(`.service-option-card[data-service-id="${serviceId}"]`).forEach((card) => {
            card.classList.toggle('border-primary', selected);
            const checkbox = card.querySelector('.service-checkbox');
            const icon = checkbox ? checkbox.querySelector('i') : null;
            if (checkbox) {
                checkbox.style.background = selected ? '#ec4899' : 'white';
                checkbox.style.borderColor = selected ? '#ec4899' : '#e5e7eb';
            }
            if (icon) icon.style.display = selected ? 'flex' : 'none';
        });
    }

    function selectService(serviceId) {
        const id = Number(serviceId);
        if (!id) return;
        const exists = state.selectedServiceIds.includes(id);
        if (exists) {
            state.selectedServiceIds = state.selectedServiceIds.filter((x) => x !== id);
        } else {
            state.selectedServiceIds.push(id);
        }
        applyServiceCardState(id, !exists);
        updateSelectedServicesCount();
    }

    function switchStepServiceCategory(categoryId) {
        const key = String(categoryId);
        modalEl.querySelectorAll('.step-category-tab-btn').forEach((btn) => {
            btn.classList.toggle('active', String(btn.getAttribute('data-step-category')) === key);
        });
        modalEl.querySelectorAll('.step-services-category-content').forEach((content) => {
            content.style.display = 'none';
        });
        const target = byId(`stepServicesContent-${key}`);
        if (target) target.style.display = 'block';
    }

    function formatLocalIsoDate(dateObj) {
        const y = dateObj.getFullYear();
        const m = String(dateObj.getMonth() + 1).padStart(2, '0');
        const d = String(dateObj.getDate()).padStart(2, '0');
        return `${y}-${m}-${d}`;
    }

    function renderDates() {
        const dateList = byId('dateList');
        if (!dateList) return;
        const names = ['Paz', 'Pzt', 'Sal', 'Çar', 'Per', 'Cum', 'Cmt'];
        const today = new Date();
        const rows = [];
        for (let i = 0; i < 14; i += 1) {
            const d = new Date(today);
            d.setDate(today.getDate() + i);
            const iso = formatLocalIsoDate(d);
            rows.push(
                `<div class="col-md-3 col-6 mb-3">
                    <button type="button" class="date-option-card btn w-100 ${state.selectedDate === iso ? 'border-primary' : ''}" data-date="${iso}" style="border: 1px solid #e5e7eb; border-radius: 10px; background: white;">
                        <div style="font-weight: 600;">${String(d.getDate()).padStart(2, '0')}.${String(d.getMonth() + 1).padStart(2, '0')}</div>
                        <small class="text-muted">${names[d.getDay()]}</small>
                    </button>
                </div>`
            );
        }
        dateList.innerHTML = rows.join('');
    }

    function selectDate(dateStr) {
        state.selectedDate = dateStr;
        const selectedDate = byId('selectedDate');
        if (selectedDate) selectedDate.value = dateStr;
        modalEl.querySelectorAll('.date-option-card').forEach((card) => {
            card.classList.toggle('border-primary', card.getAttribute('data-date') === dateStr);
        });
        loadAvailableSlots(dateStr);
    }

    function selectTime(time) {
        state.selectedTime = time;
        const selectedTime = byId('selectedTime');
        if (selectedTime) selectedTime.value = time;
        modalEl.querySelectorAll('.time-slot-btn').forEach((btn) => {
            const isActive = btn.getAttribute('data-time') === time;
            btn.classList.toggle('btn-primary', isActive);
            btn.classList.toggle('btn-outline-secondary', !isActive && !btn.disabled);
        });
    }

    async function loadAvailableSlots(dateStr) {
        const timeList = byId('timeList');
        if (!timeList) return;
        if (!dateStr || state.selectedServiceIds.length === 0) {
            timeList.innerHTML = '<div class="col-12"><div class="alert alert-warning">Lutfen once hizmet ve tarih seciniz.</div></div>';
            return;
        }
        timeList.innerHTML = '<div class="col-12"><div class="alert alert-info">Saatler yukleniyor...</div></div>';

        try {
            const query = new URLSearchParams({
                service_ids: state.selectedServiceIds.join(','),
                salon_id: String(state.salonId),
                date: dateStr
            });
            const response = await fetch(`{{ route("public.appointment-request.available-slots-ascii") }}?${query.toString()}`, {
                headers: {
                    'Accept': 'application/json',
                    'X-CSRF-TOKEN': '{{ csrf_token() }}'
                }
            });
            const data = await response.json().catch(() => ({}));
            if (!response.ok) {
                throw new Error(data.message || 'Musait saatler alinamadi.');
            }

            const slotsRaw = Array.isArray(data.available_slots) ? data.available_slots : [];
            const normalized = slotsRaw.map((slot) => {
                if (typeof slot === 'string') return { time: slot, available: true };
                return { time: slot.time, available: slot.available !== false };
            }).filter((slot) => !!slot.time);

            if (!normalized.length) {
                timeList.innerHTML = '<div class="col-12"><div class="alert alert-warning">Bu tarih icin uygun saat bulunamadi.</div></div>';
                return;
            }

            normalized.sort((a, b) => String(a.time).localeCompare(String(b.time)));
            const html = normalized.map((slot) => {
                const cls = slot.available ? 'btn-outline-secondary' : 'btn-outline-secondary disabled';
                return `<div class="col-md-3 col-6 mb-2">
                    <button type="button" class="btn w-100 time-slot-btn ${cls}" data-time="${slot.time}" ${slot.available ? '' : 'disabled'}>${slot.time}</button>
                </div>`;
            }).join('');
            timeList.innerHTML = html;
        } catch (error) {
            timeList.innerHTML = `<div class="col-12"><div class="alert alert-danger">${error.message || 'Saatler alinamadi.'}</div></div>`;
        }
    }

    function getRecurringData() {
        const enabled = byId('enableRecurring')?.checked;
        if (!enabled) return null;
        return {
            enabled: true,
            recurring_type: byId('recurring_type')?.value || null,
            day_of_week: byId('day_of_week')?.value || null,
            day_of_month: byId('day_of_month')?.value || null,
            recurring_count: byId('recurring_count')?.value ? parseInt(byId('recurring_count').value, 10) : null
        };
    }

    function resetWizard(serviceId) {
        state.currentStep = 1;
        state.isRegistered = false;
        state.customerId = null;
        state.selectedServiceIds = [];
        state.selectedDate = '';
        state.selectedTime = '';

        const step1Form = byId('step1Form');
        if (step1Form) step1Form.reset();

        ['customerName', 'customerSurname', 'customerAge', 'referralCode', 'referralCodeStep4', 'otpCode'].forEach((id) => {
            const el = byId(id);
            if (el) el.value = '';
        });
        const customerInfo = byId('customerInfo');
        if (customerInfo) customerInfo.style.display = 'none';
        const customerInfoText = byId('customerInfoText');
        if (customerInfoText) customerInfoText.textContent = '';

        const selectedServiceId = byId('selectedServiceId');
        const selectedServiceIds = byId('selectedServiceIds');
        if (selectedServiceId) selectedServiceId.value = '';
        if (selectedServiceIds) selectedServiceIds.value = '[]';
        const selectedDate = byId('selectedDate');
        const selectedTime = byId('selectedTime');
        if (selectedDate) selectedDate.value = '';
        if (selectedTime) selectedTime.value = '';

        modalEl.querySelectorAll('.service-option-card').forEach((card) => {
            card.classList.remove('border-primary');
            const checkbox = card.querySelector('.service-checkbox');
            const icon = checkbox ? checkbox.querySelector('i') : null;
            if (checkbox) {
                checkbox.style.background = 'white';
                checkbox.style.borderColor = '#e5e7eb';
            }
            if (icon) icon.style.display = 'none';
        });
        modalEl.querySelectorAll('.date-option-card').forEach((card) => card.classList.remove('border-primary'));
        modalEl.querySelectorAll('.time-slot-btn').forEach((btn) => btn.classList.remove('btn-primary'));

        updateSelectedServicesCount();
        setStep(1);
        switchStepServiceCategory('all');
        renderDates();

        if (serviceId) selectService(serviceId);
    }

    function setButtonLoading(buttonId, loadingText) {
        const btn = byId(buttonId);
        if (!btn) return () => {};
        const original = btn.innerHTML;
        btn.disabled = true;
        btn.innerHTML = loadingText;
        return function restore() {
            btn.disabled = false;
            btn.innerHTML = original;
        };
    }

    function startResendCooldown(seconds) {
        const resendBtn = byId('resendOTPBtn');
        const countdown = byId('resendCountdown');
        const countdownText = byId('countdownText');
        if (!resendBtn || !countdown || !countdownText) return;
        state.resendCooldown = seconds;
        resendBtn.disabled = true;
        countdown.style.display = 'block';
        countdownText.textContent = String(state.resendCooldown);
        if (state.resendTimer) clearInterval(state.resendTimer);
        state.resendTimer = setInterval(() => {
            state.resendCooldown -= 1;
            countdownText.textContent = String(Math.max(0, state.resendCooldown));
            if (state.resendCooldown <= 0) {
                clearInterval(state.resendTimer);
                state.resendTimer = null;
                resendBtn.disabled = false;
                countdown.style.display = 'none';
            }
        }, 1000);
    }

    async function requestOtp(phone, name) {
        const response = await fetch('{{ route("public.appointment-request.send-otp") }}', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': '{{ csrf_token() }}',
                'Accept': 'application/json'
            },
            body: JSON.stringify({
                salon_id: state.salonId,
                phone: phone,
                name: name || ''
            })
        });
        const data = await response.json().catch(() => ({}));
        if (!response.ok) {
            throw new Error(data.message || `Sunucu hatasi: ${response.status}`);
        }
        return data;
    }

    async function verifyOtp(otpCode) {
        const phone = (byId('customerPhone')?.value || '').trim();
        const response = await fetch('{{ route("public.appointment-request.verify-otp") }}', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': '{{ csrf_token() }}',
                'Accept': 'application/json'
            },
            body: JSON.stringify({
                phone: phone,
                otp_code: otpCode
            })
        });
        const data = await response.json().catch(() => ({}));
        if (!response.ok) return false;
        return !!data.success;
    }

    async function submitAppointment() {
        const phone = (byId('customerPhone')?.value || '').trim();
        if (!phone) return notify('Telefon numarasi zorunlu.', 'warning');
        if (!state.selectedServiceIds.length) return notify('En az bir hizmet secmelisiniz.', 'warning');
        if (!state.selectedDate) return notify('Lutfen tarih seciniz.', 'warning');
        if (!state.selectedTime) return notify('Lutfen saat seciniz.', 'warning');

        let name = (byId('customerName')?.value || '').trim();
        let surname = (byId('customerSurname')?.value || '').trim();
        let age = (byId('customerAge')?.value || '').trim();

        if (!state.isRegistered && (!name || !surname || !age)) {
            return notify('Ad, soyad ve yas alanlari zorunludur.', 'warning');
        }

        const referralCode = (byId('referralCodeStep4')?.value || byId('referralCode')?.value || '').trim() || null;
        const payload = {
            salon_id: state.salonId,
            service_id: state.selectedServiceIds[0] || null,
            service_ids: state.selectedServiceIds,
            name: name || null,
            surname: surname || null,
            age: age ? parseInt(age, 10) : null,
            phone: phone,
            customer_id: state.customerId,
            referral_code: referralCode,
            date: state.selectedDate,
            time: state.selectedTime,
            recurring: getRecurringData()
        };

        const restore = setButtonLoading('submitBtn', '<i class="bi bi-hourglass-split me-1"></i>Gonderiliyor...');
        try {
            const response = await fetch('{{ route("public.appointment-request.store") }}', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': '{{ csrf_token() }}',
                    'Accept': 'application/json'
                },
                body: JSON.stringify(payload)
            });
            const data = await response.json().catch(() => ({}));
            if (!response.ok || !data.success) {
                throw new Error(data.message || 'Randevu olusturulamadi.');
            }
            const modalInstance = window.bootstrap?.Modal?.getOrCreateInstance(modalEl);
            if (modalInstance) modalInstance.hide();
            notify('Randevu talebiniz basariyla alindi.', 'success');
        } catch (error) {
            notify(error.message || 'Randevu talebi gonderilemedi.', 'error');
        } finally {
            restore();
        }
    }

    async function checkPhone() {
        const phone = (byId('customerPhone')?.value || '').trim();
        if (!phone) return notify('Lutfen telefon numarasi giriniz.', 'warning');
        const restore = setButtonLoading('verifyPhoneBtn', '<i class="bi bi-hourglass-split me-1"></i>Kontrol Ediliyor...');
        try {
            const data = await requestOtp(phone, byId('customerName')?.value || '');
            if (!data.success) throw new Error(data.message || 'Telefon kontrolu basarisiz.');

            const customerInfo = byId('customerInfo');
            const customerInfoText = byId('customerInfoText');
            if (customerInfo) customerInfo.style.display = 'block';

            state.isRegistered = !!data.is_registered;
            state.customerId = data.customer_id || null;
            if (state.isRegistered) {
                if (byId('customerName')) byId('customerName').value = data.customer_name || '';
                if (byId('customerSurname')) byId('customerSurname').value = data.customer_surname || '';
                if (byId('customerAge')) byId('customerAge').value = data.customer_age || '';
                if (customerInfoText) customerInfoText.textContent = 'Kayıtlı müşteri bulundu. OTP doğrulamasına geçiliyor.';
                setStep(3);
            } else {
                if (customerInfoText) customerInfoText.textContent = 'Yeni müşteri. Lütfen bilgilerinizi tamamlayın.';
                setStep(2);
            }
            startResendCooldown(60);
        } catch (error) {
            notify(error.message || 'Telefon kontrolu sirasinda hata olustu.', 'error');
        } finally {
            restore();
        }
    }

    async function resendOtp() {
        if (state.resendCooldown > 0) return;
        const phone = (byId('customerPhone')?.value || '').trim();
        if (!phone) return notify('Telefon numarasi eksik.', 'warning');
        const restore = setButtonLoading('resendOTPBtn', '<i class="bi bi-hourglass-split me-1"></i>Gonderiliyor...');
        try {
            const data = await requestOtp(phone, byId('customerName')?.value || '');
            if (!data.success) throw new Error(data.message || 'OTP gonderilemedi.');
            notify('Doğrulama kodu tekrar gönderildi.', 'success');
            startResendCooldown(60);
        } catch (error) {
            notify(error.message || 'OTP tekrar gonderilemedi.', 'error');
        } finally {
            restore();
        }
    }

    async function nextStep() {
        if (state.currentStep === 2) {
            const name = (byId('customerName')?.value || '').trim();
            const surname = (byId('customerSurname')?.value || '').trim();
            const age = (byId('customerAge')?.value || '').trim();
            if (!name || !surname || !age) return notify('Lutfen ad, soyad ve yas bilgilerinizi doldurun.', 'warning');
            setStep(3);
            return;
        }
        if (state.currentStep === 3) {
            const otpCode = (byId('otpCode')?.value || '').trim();
            if (otpCode.length !== 6) return notify('Lutfen 6 haneli OTP kodu giriniz.', 'warning');
            const verified = await verifyOtp(otpCode);
            if (!verified) return notify('OTP kodu dogrulanamadi.', 'error');
            const referralContainer = byId('referralCodeStep4Container');
            if (referralContainer) referralContainer.style.display = 'block';
            setStep(4);
            return;
        }
        if (state.currentStep === 4) {
            if (!state.selectedServiceIds.length) return notify('Lutfen en az bir hizmet seciniz.', 'warning');
            setStep(5);
            return;
        }
        if (state.currentStep === 5) {
            if (!state.selectedDate) return notify('Lutfen bir tarih seciniz.', 'warning');
            setStep(6);
            return;
        }
    }

    function prevStep() {
        if (state.currentStep <= 1) return;
        if (state.currentStep === 3 && state.isRegistered) {
            setStep(1);
            return;
        }
        setStep(state.currentStep - 1);
    }

    modalEl.addEventListener('click', function (event) {
        const categoryBtn = event.target.closest('.step-category-tab-btn[data-step-category]');
        if (categoryBtn && modalEl.contains(categoryBtn)) {
            event.preventDefault();
            event.stopPropagation();
            switchStepServiceCategory(categoryBtn.getAttribute('data-step-category'));
            return;
        }

        const serviceCard = event.target.closest('.service-option-card[data-service-id]');
        if (serviceCard && modalEl.contains(serviceCard)) {
            event.preventDefault();
            event.stopPropagation();
            selectService(serviceCard.getAttribute('data-service-id'));
            return;
        }

        const dateCard = event.target.closest('.date-option-card[data-date]');
        if (dateCard && modalEl.contains(dateCard)) {
            event.preventDefault();
            event.stopPropagation();
            selectDate(dateCard.getAttribute('data-date'));
            return;
        }

        const timeBtn = event.target.closest('.time-slot-btn[data-time]');
        if (timeBtn && modalEl.contains(timeBtn) && !timeBtn.disabled) {
            event.preventDefault();
            event.stopPropagation();
            selectTime(timeBtn.getAttribute('data-time'));
        }
    }, true);

    const phoneInput = byId('customerPhone');
    if (phoneInput) {
        phoneInput.addEventListener('keydown', function (event) {
            if (event.key === 'Enter') {
                event.preventDefault();
                checkPhone();
            }
        });
    }

    const verifyBtn = byId('verifyPhoneBtn');
    const resendBtn = byId('resendOTPBtn');
    const nextBtn = byId('nextStepBtn');
    const prevBtn = byId('prevStepBtn');
    const submitBtn = byId('submitBtn');
    if (verifyBtn) verifyBtn.onclick = (event) => { event.preventDefault(); event.stopPropagation(); checkPhone(); };
    if (resendBtn) resendBtn.onclick = (event) => { event.preventDefault(); event.stopPropagation(); resendOtp(); };
    if (nextBtn) nextBtn.onclick = (event) => { event.preventDefault(); event.stopPropagation(); nextStep(); };
    if (prevBtn) prevBtn.onclick = (event) => { event.preventDefault(); event.stopPropagation(); prevStep(); };
    if (submitBtn) submitBtn.onclick = (event) => { event.preventDefault(); event.stopPropagation(); submitAppointment(); };

    window.openAppointmentModal = function (serviceId) {
        resetWizard(serviceId || null);
        const instance = window.bootstrap?.Modal?.getOrCreateInstance(modalEl);
        if (instance) instance.show();
    };
    window.checkPhone = checkPhone;
    window.resendOTP = resendOtp;
    window.nextStep = nextStep;
    window.prevStep = prevStep;
    window.selectService = selectService;
    window.selectDate = selectDate;
    window.selectTime = selectTime;
    window.switchStepServiceCategory = switchStepServiceCategory;
    window.__appointmentWizardV2Api = {
        checkPhone,
        resendOtp,
        nextStep,
        prevStep,
        submitAppointment
    };

    // Force single-source click handling for critical buttons.
    document.addEventListener('click', function (event) {
        const verifyBtn = event.target.closest('#verifyPhoneBtn');
        if (verifyBtn && modalEl.contains(verifyBtn)) {
            event.preventDefault();
            event.stopPropagation();
            if (typeof event.stopImmediatePropagation === 'function') event.stopImmediatePropagation();
            checkPhone();
            return;
        }
        const resendBtn = event.target.closest('#resendOTPBtn');
        if (resendBtn && modalEl.contains(resendBtn)) {
            event.preventDefault();
            event.stopPropagation();
            if (typeof event.stopImmediatePropagation === 'function') event.stopImmediatePropagation();
            resendOtp();
            return;
        }
        const nextBtn = event.target.closest('#nextStepBtn');
        if (nextBtn && modalEl.contains(nextBtn)) {
            event.preventDefault();
            event.stopPropagation();
            if (typeof event.stopImmediatePropagation === 'function') event.stopImmediatePropagation();
            nextStep();
            return;
        }
        const prevBtn = event.target.closest('#prevStepBtn');
        if (prevBtn && modalEl.contains(prevBtn)) {
            event.preventDefault();
            event.stopPropagation();
            if (typeof event.stopImmediatePropagation === 'function') event.stopImmediatePropagation();
            prevStep();
            return;
        }
        const submitBtn = event.target.closest('#submitBtn');
        if (submitBtn && modalEl.contains(submitBtn)) {
            event.preventDefault();
            event.stopPropagation();
            if (typeof event.stopImmediatePropagation === 'function') event.stopImmediatePropagation();
            submitAppointment();
        }
    }, true);

    resetWizard(null);
})();
