<?php

namespace Tests\Feature;

use App\Models\SmsLog;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Queue;
use Tests\TestCase;

class SmsApiTest extends TestCase
{
    use RefreshDatabase;

    protected function setUp(): void
    {
        parent::setUp();
        Queue::fake();
    }

    /** @test */
    public function it_can_send_sms_via_api()
    {
        Http::fake([
            'api.interaktifsms.com/*' => Http::response([
                'success' => true,
                'message_id' => 'api-test-123',
            ], 200),
        ]);

        $response = $this->postJson('/api/sms/send', [
            'to' => '05551234567',
            'message' => 'Test SMS message',
            'provider' => 'interaktif_sms',
        ]);

        $response->assertStatus(200);
        $response->assertJson([
            'success' => true,
            'data' => [
                'provider' => 'interaktif_sms',
            ],
        ]);

        Queue::assertPushed(\App\Jobs\SendSmsJob::class);
    }

    /** @test */
    public function it_validates_phone_number_format()
    {
        $response = $this->postJson('/api/sms/send', [
            'to' => 'invalid-phone',
            'message' => 'Test message',
        ]);

        $response->assertStatus(422);
        $response->assertJsonValidationErrors(['to']);
    }

    /** @test */
    public function it_can_request_otp_via_api()
    {
        Http::fake([
            'api.interaktifsms.com/*' => Http::response([
                'success' => true,
                'message_id' => 'otp-test-123',
            ], 200),
        ]);

        $response = $this->postJson('/api/sms/otp/request', [
            'phone' => '05551234567',
            'purpose' => 'login',
        ]);

        $response->assertStatus(200);
        $response->assertJson(['success' => true]);
    }

    /** @test */
    public function it_can_verify_otp_via_api()
    {
        $otpService = app(\App\Services\Sms\OtpService::class);
        $code = $otpService->generate();
        $otpService->store('+905551234567', $code, 'login');

        $response = $this->postJson('/api/sms/otp/verify', [
            'phone' => '05551234567',
            'code' => $code,
            'purpose' => 'login',
        ]);

        $response->assertStatus(200);
        $response->assertJson(['success' => true]);
    }

    /** @test */
    public function it_rejects_invalid_otp_via_api()
    {
        $response = $this->postJson('/api/sms/otp/verify', [
            'phone' => '05551234567',
            'code' => 999999,
            'purpose' => 'login',
        ]);

        $response->assertStatus(400);
        $response->assertJson(['success' => false, 'error' => 'invalid_otp']);
    }

    /** @test */
    public function it_can_fetch_sms_logs()
    {
        SmsLog::factory()->count(5)->create([
            'provider' => 'interaktif_sms',
            'status' => 'sent',
        ]);

        $response = $this->getJson('/api/sms/logs?provider=interaktif_sms&per_page=10');

        $response->assertStatus(200);
        $response->assertJson(['success' => true]);
        $response->assertJsonStructure([
            'data' => [
                'data' => [
                    '*' => ['id', 'provider', 'to', 'message', 'status'],
                ],
            ],
        ]);
    }
}

