<?php

namespace Tests\Feature;

use App\Services\Sms\OtpService;
use App\Services\SmsService;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\RateLimiter;
use Tests\TestCase;

class SmsOtpFlowTest extends TestCase
{
    use RefreshDatabase;

    protected function setUp(): void
    {
        parent::setUp();
        Cache::flush();
        RateLimiter::clear('otp:+905551234567:minute');
        RateLimiter::clear('otp:+905551234567:10minutes');
    }

    /** @test */
    public function it_can_generate_and_store_otp()
    {
        $otpService = app(OtpService::class);
        $code = $otpService->generate();

        $this->assertIsInt($code);
        $this->assertEquals(6, strlen((string) $code));

        $otpService->store('+905551234567', $code);
        $this->assertTrue($otpService->exists('+905551234567'));
    }

    /** @test */
    public function it_can_verify_otp_code()
    {
        Http::fake([
            'api.interaktifsms.com/*' => Http::response([
                'success' => true,
                'message_id' => 'test-123',
            ], 200),
        ]);

        $smsService = app(SmsService::class);
        $otpService = app(OtpService::class);

        $code = $otpService->generate();
        $otpService->store('+905551234567', $code, 'login');

        $this->assertTrue($smsService->verifyOtp('+905551234567', $code, 'login'));
        $this->assertFalse($smsService->verifyOtp('+905551234567', $code, 'login')); // Should be deleted
    }

    /** @test */
    public function it_rejects_invalid_otp_code()
    {
        $otpService = app(OtpService::class);
        $code = $otpService->generate();
        $otpService->store('+905551234567', $code);

        $this->assertFalse($otpService->verify('+905551234567', 999999));
    }

    /** @test */
    public function it_enforces_rate_limiting()
    {
        config(['sms.rate_limit.enabled' => true]);
        config(['sms.rate_limit.max_attempts_per_minute' => 2]);

        $otpService = app(OtpService::class);

        // First request should pass
        $this->assertTrue($otpService->checkRateLimit('+905551234567'));
        $otpService->hitRateLimit('+905551234567');

        // Second request should pass
        $this->assertTrue($otpService->checkRateLimit('+905551234567'));
        $otpService->hitRateLimit('+905551234567');

        // Third request should fail
        $this->assertFalse($otpService->checkRateLimit('+905551234567'));
    }

    /** @test */
    public function it_can_send_otp_via_sms()
    {
        Http::fake([
            'api.interaktifsms.com/*' => Http::response([
                'success' => true,
                'message_id' => 'test-123',
            ], 200),
        ]);

        $smsService = app(SmsService::class);
        $result = $smsService->sendOtp('+905551234567', 'login', 'interaktif_sms');

        $this->assertTrue($result['success']);
        $this->assertArrayHasKey('code', $result);
    }

    /** @test */
    public function it_respects_rate_limit_when_sending_otp()
    {
        config(['sms.rate_limit.enabled' => true]);
        config(['sms.rate_limit.max_attempts_per_minute' => 1]);

        Http::fake([
            'api.interaktifsms.com/*' => Http::response([
                'success' => true,
                'message_id' => 'test-123',
            ], 200),
        ]);

        $smsService = app(SmsService::class);

        // First request should succeed
        $result1 = $smsService->sendOtp('+905551234567', 'login');
        $this->assertTrue($result1['success']);

        // Second request should fail due to rate limit
        $result2 = $smsService->sendOtp('+905551234567', 'login');
        $this->assertFalse($result2['success']);
        $this->assertEquals('rate_limit_exceeded', $result2['error']);
    }
}

