<?php

namespace Tests\Feature;

use App\Models\SmsLog;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\Config;
use Tests\TestCase;

class SmsWebhookTest extends TestCase
{
    use RefreshDatabase;

    /** @test */
    public function it_rejects_webhook_without_valid_token()
    {
        Config::set('sms.webhook.enabled', true);
        Config::set('sms.webhook.secret', 'test-secret-123');

        $response = $this->postJson('/api/sms/webhook/interaktif_sms', [
            'message_id' => 'test-123',
            'status' => 'delivered',
        ]);

        $response->assertStatus(401);
    }

    /** @test */
    public function it_accepts_webhook_with_valid_token()
    {
        Config::set('sms.webhook.enabled', true);
        Config::set('sms.webhook.secret', 'test-secret-123');
        Config::set('sms.webhook.header_token', 'X-SMS-Webhook-Token');

        // Create a log entry first
        $log = SmsLog::create([
            'provider' => 'interaktif_sms',
            'to' => '+905551234567',
            'message' => 'Test message',
            'status' => 'sent',
            'provider_message_id' => 'test-123',
        ]);

        $response = $this->postJson('/api/sms/webhook/interaktif_sms', [
            'message_id' => 'test-123',
            'status' => 'delivered',
        ], [
            'X-SMS-Webhook-Token' => 'test-secret-123',
        ]);

        $response->assertStatus(200);
        $response->assertJson(['success' => true]);

        // Verify log was updated
        $log->refresh();
        $this->assertEquals('delivered', $log->status);
        $this->assertNotNull($log->delivered_at);
    }

    /** @test */
    public function it_handles_webhook_with_different_payload_formats()
    {
        Config::set('sms.webhook.enabled', false); // Disable auth for this test

        $log = SmsLog::create([
            'provider' => 'vatansms',
            'to' => '+905551234567',
            'message' => 'Test message',
            'status' => 'sent',
            'provider_message_id' => 'vatansms-123',
        ]);

        // Test with 'id' field instead of 'message_id'
        $response = $this->postJson('/api/sms/webhook/vatansms', [
            'id' => 'vatansms-123',
            'durum' => 'teslim_edildi',
        ]);

        $response->assertStatus(200);

        $log->refresh();
        $this->assertEquals('delivered', $log->status);
    }

    /** @test */
    public function it_handles_webhook_for_nonexistent_message()
    {
        Config::set('sms.webhook.enabled', false);

        $response = $this->postJson('/api/sms/webhook/interaktif_sms', [
            'message_id' => 'nonexistent-123',
            'status' => 'delivered',
        ]);

        $response->assertStatus(400);
        $response->assertJson(['success' => false, 'error' => 'message_id_not_found']);
    }
}

