<?php

namespace Tests\Unit;

use App\DTO\SmsSendResult;
use App\Services\Sms\OtpService;
use App\Services\Sms\SmsManager;
use App\Services\SmsService;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Queue;
use Tests\TestCase;

class SmsServiceTest extends TestCase
{
    use RefreshDatabase;

    protected function setUp(): void
    {
        parent::setUp();
        Queue::fake();
    }

    /** @test */
    public function it_can_send_sms_synchronously()
    {
        config(['sms.queue.enabled' => false]);

        Http::fake([
            'api.interaktifsms.com/*' => Http::response([
                'success' => true,
                'message_id' => 'test-123',
            ], 200),
        ]);

        $smsService = app(SmsService::class);
        $result = $smsService->send('+905551234567', 'Test message', 'interaktif_sms');

        $this->assertTrue($result->success);
        $this->assertEquals('test-123', $result->messageId);
    }

    /** @test */
    public function it_can_send_sms_via_queue()
    {
        config(['sms.queue.enabled' => true]);

        $smsService = app(SmsService::class);
        $result = $smsService->send('+905551234567', 'Test message');

        Queue::assertPushed(\App\Jobs\SendSmsJob::class);
        $this->assertTrue($result->success);
        $this->assertEquals('queued', $result->messageId);
    }

    /** @test */
    public function it_normalizes_turkish_phone_numbers()
    {
        Http::fake([
            'api.interaktifsms.com/*' => Http::response([
                'success' => true,
                'message_id' => 'test-123',
            ], 200),
        ]);

        $smsService = app(SmsService::class);

        // Test various formats
        $formats = [
            '05551234567' => '+905551234567',
            '5551234567' => '+905551234567',
            '905551234567' => '+905551234567',
            '+905551234567' => '+905551234567',
        ];

        config(['sms.queue.enabled' => false]);

        foreach ($formats as $input => $expected) {
            Http::fake([
                'api.interaktifsms.com/*' => Http::response([
                    'success' => true,
                    'message_id' => 'test-123',
                ], 200),
            ]);

            $result = $smsService->send($input, 'Test', 'interaktif_sms');
            $this->assertTrue($result->success);
        }
    }

    /** @test */
    public function it_handles_sms_send_failure()
    {
        config(['sms.queue.enabled' => false]);

        Http::fake([
            'api.interaktifsms.com/*' => Http::response([
                'success' => false,
                'error' => 'INSUFFICIENT_BALANCE',
                'message' => 'Insufficient balance',
            ], 400),
        ]);

        $smsService = app(SmsService::class);
        $result = $smsService->send('+905551234567', 'Test message', 'interaktif_sms');

        $this->assertFalse($result->success);
        $this->assertNotNull($result->errorCode);
        $this->assertNotNull($result->errorMessage);
    }
}

